#!/usr/bin/perl -w
#
#    DateFormat.t
#
#    Copyright (c) 2000 Brian Stell
#
#    This package is free software and is provided ``as is'' without
#    express or implied warranty. It may be used, redistributed and/or
#    modified under the terms of the Perl Artistic License
#    (see http://www.perl.com/perl/misc/Artistic.html)
#

use strict;
use Test;
BEGIN { plan tests => 10 }
use ExtUtils::testlib;

use utf8;
use ICU qw(U_ZERO_ERROR);
use ICU::DateFormat qw(:EStyle :EField);
use ICU::Locale;

#
# Make sure we always use the same fallback locale
#
my $default_locale = new ICU::Locale('en_US');
ICU::Locale->setDefault($default_locale);

#
# Make sure we always use the same timezone
#
my $tz_pst = ICU::TimeZone->createTimeZone('PST');
ICU::TimeZone->setDefault($tz_pst);


ok(check_enums(), 1);
ok(check_createInstance(), 1);
ok(check_createDateInstance(), 1);
ok(check_createDateTimeInstance(), 1);
ok(check_createTimeInstance(), 1);
ok(check_getAvailableLocales(), 1);
ok(check_parse(), 1);
ok(check_toPattern(), 1);
ok(check_setTimeZone(), 1);
ok(check_setDebugLevel(), 1);

########################################################################
#
# The subroutines and tests
#
# (in alphabetic order)
#
########################################################################

sub check_createDateInstance {
    my $time = (60*60*24*365) + (60*60*24*60);
    $time *= 1000; # convert to milliseconds

    my $result = 0;

    # check params
    my $df = ICU::DateFormat->createDateInstance();
    $df = ICU::DateFormat->createDateInstance(kShort);
    my $locale = new ICU::Locale('en_US');
    $df = ICU::DateFormat->createDateInstance(kShort, $locale);

    # check formatting
    if  (  (check_createDateInstance1('en_US', kFull, $time, 'Monday, March 1, 1971'))
        && (check_createDateInstance1('fr_FR', kFull, $time, 'lundi 1 mars 1971'))
        && (check_createDateInstance1('de_DE', kFull, $time, 'Montag, 1. März 1971'))
        && (check_createDateInstance1('nl_NL', kFull, $time, 'maandag 1 maart 1971'))
        ) {
        $result = 1;
    }
    return($result);
}

sub check_createDateInstance1 {
    my ($locale_str, $style, $time, $expected_result) = @_;
    my $result = 0;

    my $locale = new ICU::Locale($locale_str);
    ICU::Locale->setDefault($locale);
    my $df = ICU::DateFormat->createDateInstance($style);
    if (($df->format($time) eq $expected_result) &&
        ($df->format($time, 0) eq $expected_result)) {
        $result = 1;
    }
    else {
        print "format($locale_str) = (".$df->format($time, 0).") "
                                           . "expect ($expected_result)\n";
    }
    ICU::Locale->setDefault($default_locale);
    return($result);
}

sub check_createDateTimeInstance {
    my $time = (60*60*24*365) + (60*60*24*60);
    $time *= 1000; # convert to milliseconds

    my $result = 0;

    # check params
    my $df = ICU::DateFormat->createDateTimeInstance();
    $df = ICU::DateFormat->createDateTimeInstance(kShort);
    my $locale = new ICU::Locale('en_US');
    $df = ICU::DateFormat->createDateTimeInstance(kShort, $locale);

    if  ((check_createDateTimeInstance1
               ('en_US', kFull, $time, 'Monday, March 1, 1971 4:00:00 PM'))
        && (check_createDateTimeInstance1
               ('fr_FR', kFull, $time, 'lundi 1 mars 1971 16:00:00'))
        && (check_createDateTimeInstance1
               ('de_DE', kFull, $time, 'Montag, 1. März 1971 16:00:00'))
        && (check_createDateTimeInstance1
               ('nl_NL', kFull, $time, 'maandag 1 maart 1971 16:00:00'))
        ) {
        $result = 1;
    }
    return($result);
}

sub check_createDateTimeInstance1 {
    my ($locale_str, $style, $time, $expected_result) = @_;
    my $result = 0;

    my $locale = new ICU::Locale($locale_str);
    ICU::Locale->setDefault($locale);
    my $df = ICU::DateFormat->createDateTimeInstance($style);
    if (($df->format($time) eq $expected_result)  &&
        ($df->format($time, 0) eq $expected_result)) {
        $result = 1;
    }
    else {
        print "format($locale_str) = (".$df->format($time, 0).") "
                                             . "expect ($expected_result)\n";
    }
    ICU::Locale->setDefault($default_locale);
    return($result);
}

sub check_createInstance {
    my $time = (60*60*24*365) + (60*60*24*60);
    $time *= 1000; # convert to milliseconds

    my $result = 0;
    if  (  (check_createInstance1('en_US', $time, '3/1/71 4:00 PM'))
        && (check_createInstance1('fr_FR', $time, '01/03/71 16:00'))
        && (check_createInstance1('de_DE', $time, '01.03.71 16:00'))
        && (check_createInstance1('nl_NL', $time, '1-3-71 16:00'))
        ) {
        $result = 1;
    }
    return($result);
}

sub check_createInstance1 {
    my ($locale_str, $time, $expected_result) = @_;
    my $result = 0;

    my $locale = new ICU::Locale($locale_str);
    ICU::Locale->setDefault($locale);
    my $df = ICU::DateFormat->createInstance();
    if (($df->format($time) eq $expected_result)  &&
        ($df->format($time, 0) eq $expected_result)) {
        $result = 1;
    }
    else {
        print "format($locale_str) = (".$df->format($time, 0).") "
                                      . "expect ($expected_result)\n";
    }
    ICU::Locale->setDefault($default_locale);
    return($result);
}

sub check_createTimeInstance {
    my $time = (60*60*24*365) + (60*60*24*60);
    $time *= 1000; # convert to milliseconds

    my $result = 0;

    # check params
    my $df = ICU::DateFormat->createTimeInstance();
    $df = ICU::DateFormat->createTimeInstance(kShort);
    my $locale = new ICU::Locale('en_US');
    $df = ICU::DateFormat->createTimeInstance(kShort, $locale);

    if  (  (check_createTimeInstance1('en_US', kFull, $time, '4:00:00 PM PST'))
        && (check_createTimeInstance1('fr_FR', kFull, $time, '16 h 00 GMT-08:00'))
        && (check_createTimeInstance1('de_DE', kFull, $time, '16:00 Uhr GMT-08:00'))
        && (check_createTimeInstance1('nl_NL', kFull, $time, '16:00:00 uur GMT-08:00'))
        ) {
        $result = 1;
    }
    return($result);
}

sub check_createTimeInstance1 {
    my ($locale_str, $style, $time, $expected_result) = @_;
    my $result = 0;

    my $locale = new ICU::Locale($locale_str);
    ICU::Locale->setDefault($locale);
    my $df = ICU::DateFormat->createTimeInstance($style);
    if (($df->format($time) eq $expected_result) &&
        ($df->format($time, 0) eq $expected_result)) {
        $result = 1;
    }
    else {
        print "format($locale_str) = (".$df->format($time, 0).") "
                                      . "expect ($expected_result)\n";
    }
    ICU::Locale->setDefault($default_locale);
    return($result);
}

sub check_enums {
    if (   (kNone       == -1)
        && (kFull       == 0)
        && (kLong       == 1)
        && (kMedium     == 2)
        && (kDefault    == 2)
        && (kShort      == 3)
        && (kDateOffset == 4)
        && (kDateTime   == 8)
        && (kEraField              == 0)
        && (kYearField             == 1)
        && (kMonthField            == 2)
        && (kDateField             == 3)
        && (kHourOfDay1Field       == 4)
        && (kHourOfDay0Field       == 5)
        && (kMinuteField           == 6)
        && (kSecondField           == 7)
        && (kMillisecondField      == 8)
        && (kDayOfWeekField        == 9)
        && (kDayOfYearField        == 10)
        && (kDayOfWeekInMonthField == 11)
        && (kWeekOfYearField       == 12)
        && (kWeekOfMonthField      == 13)
        && (kAmPmField             == 14)
        && (kHour1Field            == 15)
        && (kHour0Field            == 16)
        && (kTimezoneField         == 17)
        && (kYearWOYField          == 18)
        && (kDOWLocalField         == 19)
        )
    {
        return(1);
    }
    else {
        return(0);
    }
}

sub check_getAvailableLocales {
    my $result = 1;
    my @locales = ICU::DateFormat->getAvailableLocales();
    my $locale_length = scalar(@locales);
    if ($locale_length < 1) {
        printf "unable to get locales\n";
        $result = 0;
    }

    #
    # check that we got something
    #
    my $locale;
    my $found_en_US = 0;
    foreach $locale (@locales) {
        my $locale_name = $locale->getName();
        if ($locale_name eq 'en_US') {
            $found_en_US = 1;
            last;
        }
    }
    if (!$found_en_US) {
        $result = 0;
    }
    return($result);
}

sub check_parse {
    my $result = 1;

    my $time = (60*60*24*365) + (60*60*24*60);
    $time *= 1000; # convert to milliseconds

    if  ((!check_parse1('en_US', kFull, 'Monday, March 1, 1971 4:00:00 PM', $time))
        && (!check_parse1('fr_FR', kFull, 'lundi 1 mars 1971 16:00:00', $time))
        && (!check_parse1('de_DE', kFull, 'Montag, 1. März 1971 16:00:00', $time))
        && (!check_parse1('nl_NL', kFull, 'maandag 1 maart 1971 16:00:00', $time))
        ) {
        $result = 0;
    }

    return($result);
}

sub check_parse1 {
    my ($locale_str, $style, $datetime_str, $expected_time) = @_;
    my $result = 1;

    my $locale = new ICU::Locale($locale_str);
    ICU::Locale->setDefault($locale);
    my $df = ICU::DateFormat->createDateTimeInstance($style);
    if ($df->parse($datetime_str) != $expected_time) {
        $result = 0;
        print "$locale_str parse($datetime_str) = (".$df->parse($datetime_str).") "
                                      . "expect ($expected_time)\n";
    }
    # check that we handle status
    my $status = U_ZERO_ERROR;
    my $parsed_time = $df->parse($datetime_str, \$status);
    if ($status != U_ZERO_ERROR) {
        $result = 0;
        print "$locale_str parse($datetime_str) status = $status\n";
    }
    if ($parsed_time != $expected_time) {
        $result = 0;
        print "$locale_str parse($datetime_str) = (".$df->parse($datetime_str).") "
                                      . "expect ($expected_time)\n";
    }
    $status = int(1) ;
    my $parse_output = $df->parse($datetime_str, \$status);
    if (defined($parse_output)) {
        print "$locale_str parse with a non-zero status did not return undef\n";
        $result = 0;
    }

    ICU::Locale->setDefault($default_locale);
    return($result);
}

sub check_setDebugLevel {
    my $result = 1;
    my $old_debug_level1 = ICU::DateFormat::setDebugLevel();
    my $old_debug_level2 = ICU::DateFormat::setDebugLevel();
    if ($old_debug_level1 != $old_debug_level2) {
        $result = 0;
    }
    my $val1 = ICU::DateFormat::setDebugLevel(3);
    if ($val1 != $old_debug_level2) {
        $result = 0;
    }
    my $val2 = ICU::DateFormat::setDebugLevel();
    if ($val2 != 3) {
        $result = 0;
    }
    ICU::DateFormat::setDebugLevel($old_debug_level2);

    return($result);
}

sub check_setTimeZone {
    my $result = 1;

    my $time = (60*60*24*365) + (60*60*24*60);
    $time *= 1000; # convert to milliseconds

    my $df = ICU::DateFormat->createInstance();
    my $locale = new ICU::Locale('en_US');
    ICU::Locale->setDefault($locale);

    # check PST
    my $tz_pst = ICU::TimeZone->createTimeZone('PST');
    $df->setTimeZone($tz_pst);
    if ($df->format($time) ne '3/1/71 4:00 PM') {
        $result = 0;
    }

    # check EST
    my $tz_est = ICU::TimeZone->createTimeZone('EST');
    $df->setTimeZone($tz_est);
    if ($df->format($time) ne '3/1/71 7:00 PM') {
        $result = 0;
    }

    return($result);
}

sub check_toPattern {
    my $result = 0;

    my $time = (60*60*24*365) + (60*60*24*60);
    $time *= 1000; # convert to milliseconds

    if  (  (check_toPattern1('en_US', kFull, 'EEEE, MMMM d, yyyy h:mm:ss a'))
        && (check_toPattern1('fr_FR', kFull, 'EEEE d MMMM yyyy HH:mm:ss'))
        && (check_toPattern1('de_DE', kFull, 'EEEE, d. MMMM yyyy HH:mm:ss'))
        && (check_toPattern1('nl_NL', kFull, 'EEEE d MMMM yyyy H:mm:ss'))
        ) {
        $result = 1;
    }

    return($result);
}

sub check_toPattern1 {
    my ($locale_str, $style, $expected_result) = @_;

    my $result = 0;
    my $locale = new ICU::Locale($locale_str);
    ICU::Locale->setDefault($locale);
    my $df = ICU::DateFormat->createDateTimeInstance($style);
    if ($df->toPattern() eq $expected_result) {
        $result = 1;
    }
    else {
        print "$locale_str toPattern = (".$df->toPattern().") "
                                      . "expect ($expected_result)\n";
    }
    ICU::Locale->setDefault($default_locale);
    return($result);
}

exit(0);

