##----------------------------------------------------------------------------
## Changes file management - ~/lib/Changes/Release.pm
## Version v0.1.0
## Copyright(c) 2022 DEGUEST Pte. Ltd.
## Author: Jacques Deguest <jack@deguest.jp>
## Created 2022/11/23
## Modified 2022/11/23
## All rights reserved
## 
## 
## This program is free software; you can redistribute  it  and/or  modify  it
## under the same terms as Perl itself.
##----------------------------------------------------------------------------
package Changes::Release;
BEGIN
{
    use strict;
    use warnings;
    use warnings::register;
    use parent qw( Module::Generic );
    use vars qw( $VERSION $VERSION_CLASS $DEFAULT_DATETIME_FORMAT );
    use DateTime;
    use Nice::Try;
    use Want;
    our $VERSION_CLASS = 'Changes::Version';
    our $DEFAULT_DATETIME_FORMAT = '%FT%T%z';
    our $VERSION = 'v0.1.0';
};

use strict;
use warnings;

sub init
{
    my $self = shift( @_ );
    $self->{changes}        = [];
    $self->{container}      = undef;
    $self->{datetime}       = undef;
    $self->{datetime_formatter} = undef;
    $self->{elements}   = [];
    # DateTime format
    $self->{format}         = undef;
    $self->{line}           = undef;
    $self->{nl}             = "\n";
    $self->{note}           = undef;
    $self->{raw}            = undef;
    $self->{spacer}         = undef;
    $self->{time_zone}      = undef;
    $self->{version}        = '';
    $self->{_init_strict_use_sub} = 1;
    $self->SUPER::init( @_ ) || return( $self->pass_error );
    $self->{_reset} = 1;
    return( $self );
}

sub as_string
{
    my $self = shift( @_ );
    $self->message( 5, "Is reset set ? ", ( exists( $self->{_reset} ) ? 'yes' : 'no' ), " and what is cache value '", ( $self->{_cache_value} // '' ), "' and raw cache '", ( $self->{raw} // '' ), "'" );
    if( !exists( $self->{_reset} ) || 
        !defined( $self->{_reset} ) ||
        !CORE::length( $self->{_reset} ) )
    {
        my $cache;
        if( exists( $self->{_cache_value} ) &&
            defined( $self->{_cache_value} ) &&
            length( $self->{_cache_value} ) )
        {
            $cache = $self->{_cache_value};
        }
        elsif( defined( $self->{raw} ) && length( "$self->{raw}" ) )
        {
            $cache = $self->{raw};
        }
        
        my $lines = $self->new_array( $cache->scalar );
        $self->elements->foreach(sub
        {
            $self->message( 4, "Calling as_string on $_" );
            # XXX
            $_->debug( $self->debug );
            my $this = $_->as_string;
            if( defined( $this ) )
            {
                $self->message( 4, "Adding string '$this' to new lines" );
                $lines->push( $this->scalar );
            }
        });
        # my $str = $lines->join( "\n" );
        my $str = $lines->join( '' );
        return( $str );
    }
    my $v = $self->version;
    return( $self->error( "No version set yet. Set a version before calling as_string()" ) ) if( !defined( $v ) || !length( "$v" ) );
    my $dt = $self->datetime;
    my $code = $self->datetime_formatter;
    if( defined( $code ) && ref( $code ) eq 'CODE' )
    {
        try
        {
            $dt = $code->( defined( $dt ) ? $dt : () );
        }
        catch( $e )
        {
            warn( "Warning only: error with datetime formatter calback: $e\n" ) if( $self->_warnings_is_enabled( 'Changes' ) );
        }
    }
    if( !defined( $dt ) || !length( "$dt" ) )
    {
        $dt = DateTime->now;
    }
    
    my $fmt_pattern = $self->format;
    my $tz = $self->time_zone;
    if( ( !defined( $fmt_pattern ) || !length( "$fmt_pattern" ) ) &&
        !$dt->formatter &&
        defined( $DEFAULT_DATETIME_FORMAT ) &&
        length( "$DEFAULT_DATETIME_FORMAT" ) )
    {
        $fmt_pattern = $DEFAULT_DATETIME_FORMAT;
    }
    if( defined( $fmt_pattern ) && 
        length( "$fmt_pattern" ) )
    {
        try
        {
            require DateTime::Format::Strptime;
            my $dt_fmt = DateTime::Format::Strptime->new(
                pattern => $fmt_pattern,
                locale => 'en_GB',
            );
            $dt->set_formatter( $dt_fmt );
        }
        catch( $e )
        {
            return( $self->error( "Error trying to set formatter for format '${fmt_pattern}': $e" ) );
        }
    }
    my $nl = $self->nl;
    my $lines = $self->new_array;
    my $rel_str = $self->new_scalar( $v . ( $self->spacer // ' ' ) . "$dt" . ( $self->note->length ? ( ' ' . $self->note->scalar ) : '' ) . ( $nl // '' ) );
    $self->message( 4, "Adding release string '$rel_str' to new lines." );
    $lines->push( $rel_str->scalar );
    $self->elements->foreach(sub
    {
        $self->message( 4, "Calling as_string on $_" );
        # XXX
        $_->debug( $self->debug );
        my $this = $_->as_string;
        if( defined( $this ) )
        {
            $self->message( 4, "Adding string '$this' (", overload::StrVal( $this ), ") to new lines" );
            $lines->push( $this->scalar );
        }
    });
    # my $str = $lines->join( "$nl" );
    my $str = $lines->join( '' );
    $self->{_cache_value} = $str;
    CORE::delete( $self->{_reset} );
    return( $str );
}

sub changes
{
    my $self = shift( @_ );
    # my $a = $self->elements->grep(sub{ $self->_is_a( $_ => 'Changes::Change' ) });
    # We account for both Changes::Change objects registered directly under this release object, and
    # and Changes::Change objects registered under any Changes::Group objects
    my $a = $self->new_array;
    $self->elements->foreach(sub
    {
        if( $self->_is_a( $_ => 'Changes::Change' ) )
        {
            $a->push( $_ );
        }
        elsif( $self->_is_a( $_ => 'Changes::Group' ) )
        {
            my $changes = $_->elements->grep(sub{ $self->_is_a( $_ => 'Changes::Change' ) });
            $a->push( $changes->list ) if( defined( $changes ) );
        }
    });
    return( $a );
}

sub container { return( shift->_set_get_object_without_init( 'container', 'Changes', @_ ) ); }

sub datetime { return( shift->reset(@_)->_set_get_datetime( 'datetime', @_ ) ); }

sub datetime_formatter { return( shift->reset(@_)->_set_get_code( { field => 'datetime_formatter', undef_ok => 1 }, @_ ) ); }

sub elements { return( shift->_set_get_array_as_object( 'elements', @_ ) ); }

sub format { return( shift->reset(@_)->_set_get_scalar_as_object( 'format', @_ ) ); }

sub freeze
{
    my $self = shift( @_ );
    $self->message( 5, "Removing the reset marker -> '", ( $self->{_reset} // '' ), "'" );
    CORE::delete( @$self{qw( _reset )} );
    $self->elements->foreach(sub
    {
        if( $self->_can( $_ => 'freeze' ) )
        {
            $_->freeze;
        }
    });
    return( $self );
}

sub groups
{
    my $self = shift( @_ );
    my $a = $self->elements->grep(sub{ $self->_is_a( $_ => 'Changes::Group' ) });
    return( $a );
}

sub line { return( shift->reset(@_)->_set_get_number( 'line', @_ ) ); }

sub nl { return( shift->reset(@_)->_set_get_scalar_as_object( 'nl', @_ ) ); }

sub note { return( shift->reset(@_)->_set_get_scalar_as_object( 'note', @_ ) ); }

sub raw { return( shift->_set_get_scalar_as_object( 'raw', @_ ) ); }

sub reset
{
    my $self = shift( @_ );
    if( (
            !exists( $self->{_reset} ) ||
            !defined( $self->{_reset} ) ||
            !CORE::length( $self->{_reset} ) 
        ) && scalar( @_ ) )
    {
        $self->message( 4, "Reset called from -> ", sub{ $self->_get_stack_trace } );
        $self->{_reset} = scalar( @_ );
        # Cascade down the need for reset
        $self->changes->foreach(sub
        {
            if( $self->_can( $_ => 'reset' ) )
            {
                $_->reset(1);
            }
        });
    }
    return( $self );
}

sub spacer { return( shift->reset(@_)->_set_get_scalar_as_object( 'spacer', @_ ) ); }

sub time_zone
{
    my $self = shift( @_ );
    if( @_ )
    {
        my $v = shift( @_ );
        if( $self->_is_a( $v => 'DateTime::TimeZone' ) )
        {
            $self->{time_zone} = $v;
        }
        else
        {
            try
            {
                require DateTime::TimeZone;
                my $tz = DateTime::TimeZone->new( name => "$v" );
                $self->{time_zone} = $tz;
            }
            catch( $e )
            {
                return( $self->error( "Error setting time zone for '$v': $e" ) );
            }
        }
        $self->reset(1);
    }
    if( !defined( $self->{time_zone} ) )
    {
        if( Want::want( 'OBJECT' ) )
        {
            require Module::Generic::Null;
            rreturn( Module::Generic::Null->new( wants => 'OBJECT' ) );
        }
        else
        {
            return;
        }
    }
    else
    {
        return( $self->{time_zone} );
    }
}

sub version { return( shift->reset(@_)->_set_get_version( { field => 'version', class => $VERSION_CLASS }, @_ ) ); }

1;
# NOTE: POD
__END__

=encoding utf-8

=head1 NAME

Changes::Release - Release object class

=head1 SYNOPSIS

    use Changes::Release;
    my $rel = Changes::Release->new(
        # A Changes object
        container => $changes_object,
        datetime => '2022-11-17T08:12:42+0900',
        datetime_formatter => sub
        {
            my $dt = shift( @_ ) || DateTime->now;
            require DateTime::Format::Strptime;
            my $fmt = DateTime::Format::Strptime->new(
                pattern => '%FT%T%z',
                locale => 'en_GB',
            );
            $dt->set_formatter( $fmt );
            $dt->set_time_zone( 'Asia/Tokyo' );
            return( $dt );
        },
        format => '%FT%T%z',
        line => 12,
        note => 'Initial release',
        spacer => "\t",
        time_zone => 'Asia/Tokyo',
        version => 'v0.1.0',
    ) || die( Changes::Release->error, "\n" );
    say $rel->as_string;

=head1 VERSION

    v0.1.0

=head1 DESCRIPTION

This class implements a C<Changes> file release line. Such information usually comprise of a C<version> number, a C<release datetime> and an optional note

Each release section can contain L<group|Changes::Group> and L<changes|Changes::Change> that are all stored and accessible in L</changes>

If an error occurred, it returns an L<error|Module::Generic/error>

The result of this method is cached so that the second time it is called, the cache is used unless there has been any change.

=head1 METHODS

=head2 changes

Read only. This returns an L<array object|Module::Generic::Array> containing all the L<change objects|Changes::Change> within this release object.

=head2 container

Sets or gets the L<container object|Changes> for this release object. A container is the object representing the C<Changes> file: a L<Changes> object.

Note that if you instantiate a release object directly, this value will obviously be C<undef>. This value is set by L<Changes> upon parsing the C<Changes> file.

=head2 datetime

Sets or gets the release datetime information. This uses L<Module::Generic/_parse_datetime> to parse the string, so please check that documentation for supported format.

However, most format are supported including ISO8601 format and L<W3CDTF format|http://www.w3.org/TR/NOTE-datetime> (e.g. C<2022-07-17T12:10:03+09:00>)

You can alternatively directly set a L<DateTime> object.

It returns a L<DateTime> object whose L<date formatter|DateTime::Format::Strptime> object is set to the same format as provided. This ensures that any stringification of the L<DateTime> object reverts back to the string as found in the C<Changes> file or as provided by the user.

=head2 datetime_formatter

Sets or gets a code reference callback to be used when formatting the release datetime. This allows you to use alternative formatter and greater control over the formatting of the release datetime.

It must return a L<DateTime> object. Any other value will be discarded and it will fallback on setting up a L<DateTime> with current date and time using UTC as time zone and C<$DEFAULT_DATETIME_FORMAT> as default datetime format.

The code executed may die if needed and any exception will be caught and a warning will be issued if L<warnings> are enabled for L<Changes>.

=head2 elements

Sets or gets an L<array object|Module::Generic::Array> of all the elements within this release object. Those elements can be L<Changes::Group>, L<Changes::Change> and C<Changes::NewLine> objects.

=head2 format

Sets or gets a L<DateTime> format to be used with L<DateTime::Format::Strptime>. See L<DateTime::Format::Strptime/"STRPTIME PATTERN TOKENS"> for details on possible patterns.

You can also specify an alternative formatter with L</datetime_formatter>

It returns a L<scalar object|Module::Generic::Scalar>

=head2 groups

Read only. This returns an L<array object|Module::Generic::Array> containing all the L<group objects|Changes::Group> within this release object.

=head2 line

Sets or gets an integer representing the line number where this release line was found in the original C<Changes> file. If this object was instantiated separately, then obviously this value will be C<undef>

=head2 nl

Sets or gets the new line character, which defaults to C<\n>

It returns a L<number object|Module::Generic::Number>

=head2 note

Sets or gets an optional note that is set after the release datetime.

It returns a L<scalar object|Module::Generic::Scalar>

=head2 raw

Sets or gets the raw line as found in the C<Changes> file for this release. If nothing is change, and a raw version exists, then it is returned instead of computing the formatting of the line.

It returns a L<scalar object|Module::Generic::Scalar>

=head2 spacer

Sets or gets the space that can be found between the version information and the datetime. Normally this would be just one space, but since it can be other space, this is used to capture it and ensure the result is identical to what was parsed.

It returns a L<scalar object|Module::Generic::Scalar>

=head2 time_zone

Sets or gets a time zone to use for the release date. A valid time zone can either be an olson time zone string such as C<Asia/Tokyo>, or an L<DateTime::TimeZone> object.

It returns a L<DateTime::TimeZone> object upon success, or an L<error|Module::Generic/error> if an error occurred.

=head2 version

Sets or gets the version information for this release. This returns a L<version> object. If you prefer to use a different class, such as L<Perl::Version>, then you can set the global variable C<$VERSION_CLASS> accordingly.

It returns a L<version object|version>, or an object of whatever class you have set with C<$VERSION_CLASS>

=head2 changes

Sets or gets the L<array object|Module::Generic::Array> containing all the object representing the changes for that release. Those changes can be L<Changes::Group>, L<Changes::Change> or C<Changes::Line>

=head1 AUTHOR

Jacques Deguest E<lt>F<jack@deguest.jp>E<gt>

=head1 SEE ALSO

L<Changes>, L<Changes::Group>, L<Changes::Change>

=head1 COPYRIGHT & LICENSE

Copyright(c) 2022 DEGUEST Pte. Ltd.

All rights reserved

This program is free software; you can redistribute it and/or modify it under the same terms as Perl itself.

=cut
