package Lingua::DE::ASCII;

use 5.006;
use strict;
use warnings;

require Exporter;

our @ISA = qw(Exporter);

our @EXPORT = qw(to_ascii to_latin1);
our %EXPORT_TAGS = ( 'all' => [ @EXPORT ]);
our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );
our $VERSION = '0.01';

my %ascii = (qw(
         A
		 A
		 A
		 A
		 Ae
		 A
		 Ae
		 C
		 E
		 E
		 E
		 E
		 I
		 I
		 I
		 I
		 D
		 N
		 O
		 O
		 O
		 O
		 Oe
		 x
		 Oe
		 U
		 U
		 U
		 Ue
		 Y
		 Th
		 ss 	
		 a
		 a
		 a
		 a
		 ae
		 a
		 ae
		 c
		 e
		 e
		 e
		 e
		 i
		 i
		 i
		 i
		 p
		 n
		 o
		 o
		 o
		 o
		 oe
		 o
		 oe 
		 u
		 u
		 u
		 ue
		 y
		 th
		 y
		 +-
		 ^2
		 ^3
		 ue
		 P
		 .
		 ^1),
	     ("" => "'",
	      "" => ",",
          "" => "(R)",
          "" => "(C)")
    );

# remove all unknown chars
$ascii{$_} = '' foreach (grep {!defined($ascii{$_})} map {chr} (128..255));

my $non_ascii_char = join("", map {chr} (128..255));

sub to_ascii {
    my $text = shift or return;
    $text =~ s/([$non_ascii_char])/$ascii{$1}/eg;
    return $text;
}

my %mutation = qw(ae 
		  Ae 
		  oe 
		  Oe 
		  ue 
		  Ue );

my $vocal = qr/[aeiouAEIOU]/;
my $consonant = qr/[bcdfghjklmnpqrstvwxzBCDFGHJKLMNPQRSTVWXZ]/;
my $letter = qr/[abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ]/;

my $prefix = qr/(?:[Aa](?:[nb]|u[fs])|
                   [Bb]e(?>reit|i|vor|)|
                   [Dd](?:a(?>fr|neben|rum|)|
                       rin|
                       urch|
                       rei
                    )|
                   [Ee]in|
                   [Ee]nt|
                   [Ee]r|
                   [Ff]e(?:hl|st)|
                   [Ff]rei|
                   (?:[Gg](?:erade|
                             leich|
                             ro|
                             ross)
                   )|
                   [Ll]os|
                   [Gg]e|
                   [Gg]ut|
                   [Hh](?:alb|eraus|erum|inunter)|
                   [Kk]rank|
                   [Mm]ehr|
                   [Mm]it|
                   [Nn]ach|
                   [Nn]eun|
                   (?:[Ss](?:chn|till|tramm))|
                   [Tt]ot|
                   [Uu]m|
                   [Vv][eo]r|
                   [Vv]ier(?:tel)?|
                   [Ww]eg|
                   [Zz]u(?:rck|sammen)?|
                   [Zz]wei|
                   []ber
                )
               /x;

sub to_latin1 {
    local $_ = shift or return;

	if (/[Aa]e/) {
	    s/ (?<! [Gg]al)               # Galaempfnge
    	   (?<! [Jj]en)               # Jenaer Glas  
           ([aA] e)
     	/ $mutation{$1}/egx;
	}

	if (/[Oo]e/) {
	    # oe => 
    	s/(?<! [bB]enz )             # Benzoesure 
	      (?<! [Bb]ru tt)            # Bruttoertrge
	      (?<! [Nn]e tt)             # Nettoertrge
	      (?<! [^e]ot)               # Fotoelektrizitt != Stereotne
	      (?<! iez)                  # Piezoelektronik
		  (?<! [Tt]herm)                 # Thermoelektrizitt
	      ( [oO] e )
	      (?! u)
    	 /$mutation{$1}/egx;
	 }
    
	if (/[Uu]e/) {
	    # ue => , but take care for 'eue','ue', 'aue', 'que'
    	s/(?:(?<![aeAEqQ]) | 
        	 (?<=nde) | 
	         (?<=ga)  |                 # Jogabung
    	     (?<=era) |                 # kameraberwachte
	    	 (?<=ve)  |                 # Reservebung
   	 	     (?<=(?<![tT])[rR]e) |	  	# Ressieren, but not treuem
	         (?<=$vocal ne)|             # Routineberprfung 
             (?<=[Vv]orne)              # vorneber
    	   )
           (?<![Ss]tat)              # Statue
           ( [uU] e )
	       (?! i)                    # Zueilende
         /$mutation{$1}/egx;
        {no warnings;
         s/((?:${prefix}|en)s)?(tn(de?|\b))(?!chen|lein|lich)
          /$1 ? "$1$2" : "tuen$3"/xgeo;# Grotuende, but abstnde, Stndchen
        }
        #s/(?<=nt)(?=s?t)/ue/g;  # schntuest
        #s/(?<=sst)(?=s?t\b)/ue/gx;       # grotuest, grotut
        s/($prefix t)(?=s?t\b|risch)/$1 ? "$1ue" : "$1"/gxe;  # zurcktuest, grotuerisch 
        s/grnz/gruenz/g;
        s/(?<!en)(s?)(?!\w)/ue$1/g;   # Im deutschen enden keine Worte auf , bis auf Ausnahmen
        s/z(?!rich)([rs][befhiosz])/zue$1/g; # Zuerzhlende != zricherisch
    
        s/([uU] e) (?=bt)/$mutation{$1}/egx;  # bte
        s/(?<=[Dd])(?=ll)/ue/g;              # Duell
    }
	
	if (/ss/) {
   	     # russ => ru
    	 s/(?<=(?<![dD])[rRfF][u]) 
	       ss 
    	   (?! el) (?! le)                    # Brssel, Brssler
      	   (?! isch)                          # Russisch
           (?! land)                          # Ruland
          //gx;
    
         # ss =>  with many excptions
         s/(?<= $letter{2})
           (?<! $consonant $consonant)
           (?<! (?<! [bBfFmMsSeE] ) [u] )  # ben, Fu, ..., but Fluss
           (?<! [Mm] u)   # musst, musste, ...
           (?<! su)
           (?<! [bBdDfFhHkKlLmMnNrRtTwWzZ] i )   # 'wissen', -nisse,
           (?<! [dgsklnt] )
           (?<! [bBfFgGhHkKnNtTwWlLpPiI] a )     # is a short vocal
           (?<! [bBfFgGhHlLnNpPsSwW] )          # (short vocal) Ablsse, 
           (?<! [dDfFlmMnNrsStTzZ] e )           # is very short vocal
           (?<! ion )                            # Direktionssekretrin
           (?<! en )                             # dingenssachen 
           (?<! [fFhHoO] l o)
           (?<! (?<![gG]) [rR] o)                # Ross-Schlchter, but Baumgroe          
           (?<! [gGnNpP] [o])
           (?<! [sS]chl )
           (?<! [bBkKuU]e)                       # Kessel
	       (?<! rr $vocal)

           ss

           (?! ch )
           (?! isch )                        # genssisch
           (?! t[o])                   
           (?! tra)   # Davisstrae, but Schweitreibende
           (?! tur)   # Eissturm, but Schweituch
	       (?! t(?:ck|[hr]))  # Beweisstck,  Bischofssthle, Kursstrze, but Schweitcher
	       (?! tab)   # Preisstabilitt
           (?! ist)   # Ditassistentin
           (?! iv)    # Massiv	
           (?! lich)  # grsslich  
	       (?! ge)   # Kreissge
	       (?! [tu])    # Siegessule, Tagesstze
           (?! ier)   # Krassier
           (?! age)   # Massage
           (?! ard)   # Bussard
           (?! p []) # Ks-sptzle
           (?! pr)                   # lossprche
	      //gxo;
          
          s/(?<= [AaEe]u)                        # drauen
	        ss 
            (?! []) 
            (?! ee)                             # Chaussee
            (?=\b|e|l)
		  //gxo;                    # scheulich 

         s/((?<=[fs][]) |
            (?<=[Ss]p[a])    
		   )                      # ends on long vocal plus ss, like
           ss                                  # Gef != Schluss
          (?! [])
          (?! er)                           # Gefe != Fsser
          (?=\b|e|$consonant)                 # end of word or plural or new composite (Gefverschluss)
         //gxo;
         
         s/(?<=erg[a])ss(?=e|\b)//g;  # verge

        s/(?<!chlo)                                # Schloss
          (?<! (?<![gG]) [rR] o)
          (?<! go )  # goss
          ((?<=o) |(?<=ie))          # Flo, gro, Griebrei, Nu, but no Ross-Schlchter 
          ss
          (?! ch)
          (?! t? [])
          (?! pr)                   # lossprche
          (?=\b|es|$consonant)
        //gxo;
        s/(u|(?<!chl))sschen/$1chen/go;
        s/chlosst/chlot/go;  # geschlot
        
        s/(?<=[bBeEnN][Ss]a)ss(?=\b|en)//g; # absa, beisammensaen

        s/(?:(?<=[mM][ai])|(?<=[Ss])|(?<=[Ss]t)|(?<=[Ww]ei))ss(?=ge|lich)//go;
        
        s/(?<=[Gg]ro) ss (?=t)//gx;   # grotte
        s/(?<=[Ss]pa) ss (?!ion)//gx;         # spaig, but not Matthuspassion

        
        if (//) {
            s/(?<=[mM][u])(?=te|en|er)/ss/go;
            s/($prefix|en)?([Ss]a)([ea])/$1 ? "$1$2$3" : "$2ss$3"/goe;  
                     # Gefngnisinsasse, Sassafra != aufsaen, beisammensaen

            s/(?<=[rR] [a]) (?<![Gg]r)  (?=l |e [rl](?!$vocal) | chen)/ss/gxo;      # Rsser, Rssel
    
	        s/(?<=(?<![GgPp])
	              (?<![Bb]e)
		          (?<![Ee]nt)
    		      (?<![Vv]er)
	    	      [Rr]u
	          )
    	      
	          (?=[ei](?![sg])(|n|nnen)\b)
	        /ss/gxo;  # Russe, Russin, != Prue, != Gru, != Beruen, != Entruen, != Rues, != Ruige
            
        }
        
        s/($prefix)?scho(ss|)/$1 ? "$1schoss" : "scho"/ge;
	}
    
    # symbols
    s/\(R\)//g;
    s/\(C\)//g;

    # foreign words
    s/cademie/cadmie/g;
    s/rancais/ranais/g;
    s/leen/len/g;
    s/grement/grment/g;
    s/lencon/lenon/g;
    s/Ancien Regime/Ancien Rgime/g;
    s/Andre/Andr/g;
    s/Apercu/Aperu/g;
    s/([aA])pres/$1prs/g;
    s/Apero/Apro/g;
    s/Aragon/Aragn/g;
    s/deco/dco/g;
    s/socie/soci/g;
    s/([aA])suncion/$1suncin/g;
    s/([aA])ttache/$1ttach/g;
    s/Balpare/Balpar/g;
    s/Bartok/Bartk/g;
    s/Baumegrad/Baumgrad/g;
    s/Beaute/Beaut/g;
    s/Epoque/poque/g;
    s/Bjrnson/Bjrnson/g;
    s/Bogota/Bogot/g;
    s/Bokmal/Bokml/g;
    s/Boucle/Boucl/g;
    s/rree/rre/g;
    s/Bruyere/Bruyre/g;
    s/Bebe/Bb/g;
    s/echamel/chamel/g;
    s/Beret/Bret/g;
    s/([cC])afe/$1af/g;
    s/([cC])reme/$1rme/g;
    s/alderon/aldern/g;
    s/Cams/Cames/g;
    s/anape/anap/g;
    s/Canoa/Canossa/g;
    s/celebre/clbre/g;
    s/tesimo/tsimo/g;
    s/eparee/pare/g;
    s/Elysee/lyse/g;
    s/onniere/onnire/g;
    s/Charite/Charit/g;
    s/inee/ine/g;
    s/hicoree/hicore/g;
    s/Chateau/Chteau/g;
    s/Cigany/Cigny/g;
    s/Cinecitta/Cinecitt/g;
    s/Cliche/Clich/g;
    s/Cloisonne/Cloisonn/g;
    s/Cloque/Cloqu/g;
    s/dell\'Arte/dellArte/g;
    s/Communique/Communiqu/g;
    s/Consomme/Consomm/g;
    s/d\'Ampezzo/dAmpezzo/g;
    s/d\'Etat/dEtat/g;
    s/Coupe/Coup/g;
    s/Cox\'Z/Cox/g;
    s/Craquele/Craquel/g;
    s/roise/rois/g;
    s/(?<! l)
      (?<! pap)
      iere\b
     /ire/g;

    s/([cC])reme/$1rme/g;
    s/fraiche/frache/g;
    s/Crepe/Crpe/g;
    s/Csikos/Csiks/g;
    s/Csardas/Csrds/g;
    s/Cure/Cur/g;
    s/Cadiz/Cdiz/g;
    s/Centimo/Cntimo/g;
    s/Cezanne/Czanne/g;
    s/Cordoba/Crdoba/g;

    s/Dauphine/Dauphin/g;
    s/Dekollete/Dekollet/g;
    s/ieces/ices/g;
    s/trochu/trochuss/g;
    s/Drape/Drap/g;
    s/m(?=[et])/mss/g;
    s/Dvorak/Dvork/g;
    s/([dD])eja/$1j/g;
    s/habille/habill/g;
    s/Detente/Dtente/g;

    s/Ekarte/Ekart/g;
    s/El Nino/El Nio/g;
    s/Epingle/Epingl/g;
    s/Expose/Expos/g;
    s/Faure/Faur/g;
    s/Filler/Fillr/g;
    s/Siecle/Sicle/g;
    s/lel/lssel/g;
    s/Bergere/Bergre/g;
    s/Fouche/Fouch/g;
    s/Fouque/Fouqu/g;
    s/elementaire/lmentaire/g;
    s/ternite(s?)\b/ternit$1/g;
    s/risee/rise/g;
    s/roi(|ss)e/roiss/g;
    s/\bFrotte(?=\b|s\b)/Frott/g;
    s/Fume/Fum/g;
    s/([Gg])arcon/$1aron/g;
    s/([Gg])efss/$1ef/g;
    s/Gemechte/Gemchte/g;
    s/Geneve/Genve/g;
    s/Glace/Glac/g;
    s/Godemiche/Godemich/g;
    s/Godthab/Godthb/g;
    s/Gthe/Goethe/g;
    s/lame(?=\b|s)/lam/g;
    s/uyere/uyre/g;
    s/Grege/Grge/g;
    s/Gulyas/Gulys/g;
    s/abitue/abitu/g;
    s/Haler/Halr/g;
    s/ornuss/ornu/g;
    s/Horvath/Horvth/g;
    s/Hottehue/Hotteh/g;
    s/Hacek/Hcek/g;
    s/matozn/matozoen/g;
    s/chlosse(?![rsn])/chloe/g;
    s/doree/dore/g;
    s/Jerome/Jrme/g;
    s/Kodaly/Kodly/g;
    s/rzitiv/oerzitiv/g;
    s/nique/niqu/g;
    s/Kalman/Klmn/g;
    s/iberte/ibert/g;
    s/Egalite/galit/g;
    s/Linne/Linn/g;
    s/([fF])asss/$1as/g;
    s/Lome/Lom/g;
    s/Makore/Makor/g;
    s/Mallarme/Mallarm/g;
    s/aree/are/g;
    s/Maitre/Matre/g;
    s/([Mm]$vocal)liere/$1lire/g;
    s/Mouline/Moulin/g;
    s/Mousterien/Moustrien/g;
    s/Malaga/Mlaga/g;
    s/Meche/Mche/g;
    s/erimee/rime/g;
    s/eglige/eglig/g;
    s/eaute/eaut/g;
    s/egritude/gritude/g;
    s/anache/anach/g;
    s/Pappmache/Pappmach/g;
    s/Parana/Paran/g;
    s/Pathetique/Pathtique/g;
    s/Merite/Mrite/g;
    s/([Pp])reuss/$1reu/g;
    s/otege/oteg/g;
    s/recis/rcis/g;
    s/Prilitt/Puerilitt/g;
    s/Ratine/Ratin/g;
    s/Raye/Ray/g;
    s/Renforce/Renforc/g;
    s/Rene/Ren/g;
    s/Rev/Revue/g;
    s/Riksmal/Riksml/g;
    s/xupery/xupry/g;
    s/S(?:|ae)ns/Sans/g;
    s/Jose(?=s?\b)/Jos/g;
    s/bernaise/brnaise/g;
    s/Sassnitz/Sanitz/g;
	s/Saone/Sane/g;
	s/Schntr/Schntuer/g;   # more probable
	s/chling/chssling/g;
	s/Senor/Seor/g;
	s/Skues/Sks/g;
	s/Souffle(?=s|\b)/Souffl/g;
	s/Spass/Spa/g;
	s/(?<=[Cc])oupe/oup/g;
	s/Stl\b/Stal/g;
	s/Suarez/Surez/g;
	s/Sao\b/So/g;
	s/Tome(?=s|\b)/Tom/g;
	s/Seance/Sance/g;
	s/Serac/Srac/g;
	s/Sevres/Svres/g;
	s/Stassfurt/Stafurt/g;
	s/Troms/Troms/g;
	s/Trouvere/Trouvre/g;
	s/Tnder/Tnder/g;
	s/ariete/ariet/g;
	s/Welline/Wellin/g;
	s/Yucatan/Yucatn/g;
	s/\b($prefix g?)ass/$1a/gx;
    s/\b($prefix)sse/$1e/g;
    s/(\A|\W)sse/$1e/g;
	s/($prefix) (?<![Ee]in)    # != einflen
                (?<![Ee]inzu)  #    einzuflen
       fl(e(n?|s?t))\b
      /$1flss$2/gx;   # exception of rule
    s/(${prefix}|\b)sche/$1schsse/go; # also an exception
    {no warnings; s/($prefix)?spre/$1sprsse/go;}
    s/($prefix)dre/$1drsse/g;
	s/\b([Aa])ss(?=\b|en\b)/$1/go;  # a
    s/\^2//go;
    s/\^3//go;
    s/gemecht/gemcht/go;
    s/Musse/Mue/go;
    s/(?<=[Hh])ue\b//g;
    s/aelbe/asselbe/g;
    s/linnesch/linnsch/g;
    s/(?<=\b[Mm]u)ss(?=t?\b)//g;
    s/mech(?=e|s?t)/mch/g;
    s/metallise/mtallis/g;
    s/la(\W+)la/l$1l/g;
    s/(?<=\b[Oo]l)e\b//g;
    s/peu(\W+)a(\W+)peu/peu$1$2peu/g;
    s/reussisch/reuisch/g;
    s/sans gene\b/sans gne/g;
    s/(?<=\b[Ss]a)ss(?=(en|es?t)\b)//g;
    s/\bskal\b/skl/g;
    s/(?<=\bst)ue(?=nde)//g;
    s/(?<=[Tt]sch)ue(?=s)//g;
    s/([Tt])ete-a-([Tt])ete/$1te--$2te/g;
    s/voila/voil/g;
    s/Alandinseln/landinseln/g;
    s/Angstrm/ngstrm/g;
    s/Egalite/galit/g;
    s/(?<=[Ll]and)bue/busse/g;
    s/a(?=\W+(?:condition|deux mains|fonds perdu|gogo|jour|la))//g;
    s/a discretion/ discrtion/g;
    s/(?<=[Bb]ai)(?=e)/ss/g;
    s/(?<=[Hh]au)(?=e)/ss/g;
    s/\bue\././g;
    s/berflo/berfloss/g;
    return $_;
}

1;
__END__

=head1 NAME

Lingua::DE::ASCII - Perl extension to convert german umlauts to and from ascii

=head1 SYNOPSIS

  use Lingua::DE::ASCII;
  print to_ascii("Umlaute wie ,,, oder auch  usw. " .
                 "sind nicht im ASCII Format " .
                 "und werden deshalb umgeschrieben);
  print to_latin1("Dies muesste auch rueckwaerts funktionieren ma cherie");
                 

=head1 DESCRIPTION

This module enables conversion from and to the ASCII format of german texts.

It has two methods: C<to_ascii> and C<to_latin1> which one do exactly what they 
say.

=head2 EXPORT

to_ascii($string)
to_latin1($string)

=head1 BUGS

That's only a stupid computer program, faced with a very hard ai problem.
So there will be some words that will be always hard to retranslate from ascii 
to latin1 encoding. A known example is the difference between "Ma(einheit)" and
"Masseentropie" or similar. Another examples are "flsse" and "Fle"
or "(Der Schornstein) rue" and "Russe". 
Also, it's  hard to find the right spelling for the prefixes "miss-" or "mi-".
In doubt I tried to use to more common word.
I tried it with a huge list of german words, but please tell me if you find a 
serious back-translation bug.

This module is intended for ANSI code that is e.g. different from windows coding.

Misspelled words will create a lot of extra mistakes by the program.
In doubt it's better to write with new Rechtschreibung.

The C<to_latin1> method is not very quick,
it's programmed to handle as many exceptions as possible.

=head1 AUTHOR

Janek Schleicher, <bigj@kamelfreund.de>

=head1 SEE ALSO

Lingua::DE::Sentence   (another cool module)

=cut
