# Copyright (C) 2008-2011 Dejan Muhamedagic <dmuhamedagic@suse.de>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This software is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#

import sys
import os
import getopt
import atexit
import random

import config
import options
import vars
from msg import ErrorBuffer, syntax_err, common_err
from clidisplay import CliDisplay
from term import TerminalController
import utils
import userdir

import ui_root
import ui_context


random.seed()


def load_rc(context, rcfile):
    # only load the RC file if there is no new-style user config
    if config.has_user_config():
        return

    try:
        f = open(rcfile)
    except:
        return
    save_stdin = sys.stdin
    sys.stdin = f
    while True:
        inp = utils.multi_input()
        if inp is None:
            break
        try:
            if not context.run(inp):
                raise ValueError("Error in RC file: " + rcfile)
        except ValueError, msg:
            common_err(msg)
    f.close()
    sys.stdin = save_stdin


def check_args(args, argsdim):
    if not argsdim:
        return True
    if len(argsdim) == 1:
        minargs = argsdim[0]
        return len(args) >= minargs
    else:
        minargs, maxargs = argsdim
        return len(args) >= minargs and len(args) <= maxargs

#
# Note on parsing
#
# Parsing tables are python dictionaries.
#
# Keywords are used as keys and the corresponding values are
# lists (actually tuples, since they should be read-only) or
# classes. In the former case, the keyword is a terminal and
# in the latter, a new object for the class is created. The class
# must have the cmd_table variable.
#
# The list has the following content:
#
# function: a function to handle this command
# numargs_list: number of minimum/maximum arguments; for example,
#   (0, 1) means one optional argument, (1, 1) one required; if the
#   list is empty then the function will parse arguments itself
# required minimum skill level: operator, administrator, expert
#   (encoded as a small integer from 0 to 2)
# can the command cause transition to start (0 or 1)
#   used to check whether to wait4dc to end the transition
#


def show_usage(cmd):
    p = None
    try:
        p = cmd.__doc__
    except:
        pass
    if p:
        print >> sys.stderr, p
    else:
        syntax_err(cmd.__name__)


def exit_handler():
    '''
    Write the history file. Remove tmp files.
    '''
    if options.interactive and not options.batch:
        try:
            from readline import write_history_file
            write_history_file(userdir.HISTORY_FILE)
        except:
            pass


# prefer the user set PATH
def envsetup():
    mybinpath = os.path.dirname(sys.argv[0])
    for p in mybinpath, config.path.crm_daemon_dir:
        if p not in os.environ["PATH"].split(':'):
            os.environ['PATH'] = "%s:%s" % (os.environ['PATH'], p)


# three modes: interactive (no args supplied), batch (input from
# a file), half-interactive (args supplied, but not batch)
def cib_prompt():
    shadow = utils.get_cib_in_use()
    if not shadow:
        return vars.live_cib_prompt
    if vars.tmp_cib:
        return vars.tmp_cib_prompt
    return shadow


def usage(rc):
    f = sys.stderr
    if rc == 0:
        f = sys.stdout
    print >> f, """
usage:
    crm [OPTIONS] [ARGS...]

    -f, --file='FILE'::
        Load commands from the given file. If the file is - then
        use terminal stdin.

    -c, --cib='CIB'::
        Start the session with the given shadow CIB file.
        Equivalent to cib use.

    -D, --display='OUTPUT_TYPE'::
        Choose one of the output options: plain, color, or
        uppercase. The default is color if the terminal emulation
        supports colors. Otherwise, plain is used.

    -F, --force::
        Make crm proceed with doing changes even though it would
        normally ask user to confirm some of them. Mostly useful in
        scripts.

    -w, --wait::
        Make crm wait for the cluster transition to finish (for the
        changes to take effect) after each processed line.

    -H, --history='DIR|FILE'::
        The history commands can examine either live cluster
        (default) or a report generated by hb_report. Use this
        option to specify a directory or file containing the report.

    -h, --help::
        Print help page.

    --version::
        Print Pacemaker version and build information (Mercurial Hg
        changeset hash).

    -R, --regression-tests::
        Run in the regression test mode. Used mainly by the
        regression testing suite.

    -d, --debug::
        Print some debug information. Used by developers. [Not yet
        refined enough to print useful information for other users.]

    --scriptdir='DIR'::
        Extra directory where crm looks for cluster scripts. Can be
        a semicolon-separated list of directories.

    Use crm without arguments for an interactive session.
    Supply one or more arguments for a "single-shot" use.
    Supply level name to start working at that level.
    Specify with -f a file which contains a script. Use '-' for
    standard input or use pipe/redirection.

    Examples:

        # crm -f stopapp2.txt
        # crm -w resource stop global_www
        # echo stop global_www | crm resource
        # crm configure property no-quorum-policy=ignore
        # crm ra info pengine
        # crm status

    See the crm(8) man page or the crm help system for more details.
    """
    sys.exit(rc)

err_buf = ErrorBuffer.getInstance()
#levels = Levels.getInstance()


def set_interactive():
    '''Set the interactive option only if we're on a tty.'''
    if sys.stdin.isatty():
        options.interactive = True


def compatibility_setup():
    if utils.is_pcmk_118():
        vars.node_type_opt = True
        vars.attr_defaults["node"] = {"type": "normal"}
        vars.cib_no_section_rc = 6


def add_quotes(args):
    '''
    Add quotes if there's whitespace in one of the
    arguments; so that the user doesn't need to protect the
    quotes.

    If there are two kinds of quotes which actually _survive_
    the getopt, then we're _probably_ screwed.

    At any rate, stuff like ... '..."..."'
    as well as '...\'...\''  do work.
    '''
    l = []
    for s in args:
        if config.core.add_quotes and ' ' in s:
            q = '"' in s and "'" or '"'
            if not q in s:
                s = "%s%s%s" % (q, s, q)
        l.append(s)
    return l


def do_work(context, user_args):
    compatibility_setup()

    if options.shadow:
        if not context.run("cib use " + options.shadow):
            return 1

    # this special case is silly, but we have to keep it to
    # preserve the backward compatibility
    if len(user_args) == 1 and user_args[0].startswith("conf"):
        if not context.run("configure"):
            return 1
    elif len(user_args) > 0:
        # we're not sure yet whether it's an interactive session or not
        # (single-shot commands aren't)
        err_buf.reset_lineno()
        options.interactive = False

        l = add_quotes(user_args)
        if context.run(' '.join(l)):
            # if the user entered a level, then just continue
            if not context.previous_level():
                return 0
            set_interactive()
            if options.interactive:
                err_buf.reset_lineno(-1)
        else:
            return 1

    if options.input_file and options.input_file != "-":
        try:
            sys.stdin = open(options.input_file)
        except IOError, msg:
            common_err(msg)
            usage(2)

    if options.interactive and not options.batch:
        context.setup_readline()

    rc = 0
    while True:
        try:
            if options.interactive and not options.batch:
                # TODO: fix how color interacts with readline,
                # seems the color prompt messes it up
                termctrl = TerminalController.getInstance()
                cli_display = CliDisplay.getInstance()
                promptstr = "crm(%s)%s# " % (cib_prompt(), context.prompt())
                if cli_display.colors_enabled():
                    vars.prompt = termctrl.render(cli_display.prompt(promptstr))
                else:
                    vars.prompt = promptstr
            inp = utils.multi_input(vars.prompt)
            if inp is None:
                if options.interactive:
                    rc = 0
                context.quit(rc)
            try:
                if not context.run(inp):
                    rc = 1
            except ValueError, msg:
                rc = 1
                common_err(msg)
        except KeyboardInterrupt:
            if options.interactive and not options.batch:
                print "Ctrl-C, leaving"
            context.quit(1)
    return rc


def compgen():
    args = sys.argv[2:]
    if len(args) < 2:
        return

    options.shell_completion = True

    #point = int(args[0])
    line = args[1]

    # remove crm from commandline
    line_split = line.split(' ', 1)
    if len(line_split) == 1:
        return
    line = line_split[1].lstrip()

    options.interactive = False
    ui = ui_root.Root()
    context = ui_context.Context(ui)
    last_word = line.rsplit(' ', 1)
    if len(last_word) > 1 and ':' in last_word[1]:
        idx = last_word[1].rfind(':')
        for w in context.complete(line):
            print w[idx+1:]
    else:
        for w in context.complete(line):
            print w


def parse_options():
    try:
        opts, user_args = getopt.getopt(
            sys.argv[1:],
            'whdc:f:FX:RD:H:',
            ("wait", "version", "help", "debug",
             "cib=", "file=", "force", "profile=",
             "regression-tests", "display=", "history=",
             "scriptdir="))
        for o, p in opts:
            if o in ("-h", "--help"):
                usage(0)
            elif o == "--version":
                print >> sys.stdout, ("%s" % config.CRM_VERSION)
                sys.exit(0)
            elif o == "-d":
                config.core.debug = "yes"
            elif o == "-X":
                options.profile = p
            elif o == "-R":
                options.regression_tests = True
            elif o in ("-D", "--display"):
                config.color.style = p
            elif o in ("-F", "--force"):
                config.core.force = "yes"
            elif o in ("-f", "--file"):
                options.batch = True
                options.interactive = False
                err_buf.reset_lineno()
                options.input_file = p
            elif o in ("-H", "--history"):
                options.history = p
            elif o in ("-w", "--wait"):
                config.core.wait = "yes"
            elif o in ("-c", "--cib"):
                options.shadow = p
            elif o == "--scriptdir":
                options.scriptdir = p
        return user_args
    except getopt.GetoptError, msg:
        print msg
        usage(1)


def profile_run(context, user_args):
    import cProfile
    cProfile.runctx('do_work(context, user_args)',
                    globals(),
                    {'context': context, 'user_args': user_args},
                    filename=options.profile)
    # print how to use the profile file, but don't disturb
    # the regression tests
    if not options.regression_tests:
        stats_cmd = "; ".join(['import pstats',
                               's = pstats.Stats("%s")' % options.profile,
                               's.sort_stats("cumulative").print_stats()'])
        print "python -c '%s' | less" % (stats_cmd)
    return 0


def run():
    try:
        if len(sys.argv) >= 2 and sys.argv[1] == '--compgen':
            compgen()
            return 0
        envsetup()
        userdir.mv_user_files()

        ui = ui_root.Root()
        context = ui_context.Context(ui)

        load_rc(context, userdir.RC_FILE)
        atexit.register(exit_handler)
        options.interactive = sys.stdin.isatty()
        if not options.interactive:
            err_buf.reset_lineno()
            options.batch = True
        user_args = parse_options()
        if options.profile:
            return profile_run(context, user_args)
        else:
            return do_work(context, user_args)
    except KeyboardInterrupt:
        print "Ctrl-C, leaving"
        sys.exit(1)

# vim:ts=4:sw=4:et:
