/*
 * Syncdaemon API
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2010-2012 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 *
 */

#include "syncdaemon-status-info.h"

G_DEFINE_TYPE(SyncdaemonStatusInfo, syncdaemon_status_info, G_TYPE_OBJECT)

struct _SyncdaemonStatusInfoPrivate {
	gchar *connection;
	gchar *description;
	gboolean is_connected;
	gboolean is_online;
	gchar *name;
	gchar *queues;
};

static void
syncdaemon_status_info_finalize (GObject *object)
{
	SyncdaemonStatusInfo *sinfo = SYNCDAEMON_STATUS_INFO (object);

	if (sinfo->priv != NULL) {
		if (sinfo->priv->connection != NULL)
			g_free (sinfo->priv->connection);
		if (sinfo->priv->description != NULL)
			g_free (sinfo->priv->description);
		if (sinfo->priv->name != NULL)
			g_free (sinfo->priv->name);
		if (sinfo->priv->queues != NULL)
			g_free (sinfo->priv->queues);

		g_free (sinfo->priv);
	}

	G_OBJECT_CLASS (syncdaemon_status_info_parent_class)->finalize (object);
}

static void
syncdaemon_status_info_class_init (SyncdaemonStatusInfoClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = syncdaemon_status_info_finalize;
}

static void
syncdaemon_status_info_init (SyncdaemonStatusInfo *sinfo)
{
	sinfo->priv = g_new0 (SyncdaemonStatusInfoPrivate, 1);
}

/**
 * syncdaemon_status_info_new:
 */
SyncdaemonStatusInfo *
syncdaemon_status_info_new (void)
{
	return g_object_new (SYNCDAEMON_TYPE_STATUS_INFO, NULL);
}

/**
 * syncdaemon_status_info_new_from_hash_table:
 */
SyncdaemonStatusInfo *
syncdaemon_status_info_new_from_hash_table (GHashTable *hash)
{
	SyncdaemonStatusInfo *sinfo;

	sinfo = g_object_new (SYNCDAEMON_TYPE_STATUS_INFO, NULL);
	if (hash != NULL) {
		syncdaemon_status_info_set_connection (sinfo, g_hash_table_lookup (hash, "connection"));
		syncdaemon_status_info_set_description (sinfo, g_hash_table_lookup (hash, "description"));
		syncdaemon_status_info_set_connected (
			sinfo,
			g_strcmp0 (g_hash_table_lookup (hash, "is_connected"), "True") == 0);
		syncdaemon_status_info_set_online (
			sinfo,
			g_strcmp0 (g_hash_table_lookup (hash, "is_online"), "True") == 0);
		syncdaemon_status_info_set_name (sinfo, g_hash_table_lookup (hash, "name"));
		syncdaemon_status_info_set_queues (sinfo, g_hash_table_lookup (hash, "queues"));
	}

	return sinfo;
}

/**
 * syncdaemon_status_info_get_connection:
 */
const gchar *
syncdaemon_status_info_get_connection (SyncdaemonStatusInfo *sinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_STATUS_INFO (sinfo), NULL);

	return (const gchar *) sinfo->priv->connection;
}

/**
 * syncdaemon_status_info_set_connection:
 */
void
syncdaemon_status_info_set_connection (SyncdaemonStatusInfo *sinfo, const gchar *connection)
{
	g_return_if_fail (SYNCDAEMON_IS_STATUS_INFO (sinfo));

	if (sinfo->priv->connection != NULL)
		g_free (sinfo->priv->connection);

	sinfo->priv->connection = g_strdup (connection);
}

/**
 * syncdaemon_status_info_get_connected:
 */
gboolean
syncdaemon_status_info_get_connected (SyncdaemonStatusInfo *sinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_STATUS_INFO (sinfo), FALSE);

	return sinfo->priv->is_connected;
}

/**
 * syncdaemon_status_info_set_connected:
 */
void
syncdaemon_status_info_set_connected (SyncdaemonStatusInfo *sinfo, gboolean connected)
{
	g_return_if_fail (SYNCDAEMON_IS_STATUS_INFO (sinfo));

	sinfo->priv->is_connected = connected;
}

/**
 * syncdaemon_status_info_get_description:
 */
const gchar *
syncdaemon_status_info_get_description (SyncdaemonStatusInfo *sinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_STATUS_INFO (sinfo), NULL);

	return (const gchar *) sinfo->priv->description;
}

/**
 * syncdaemon_status_info_set_description:
 */
void
syncdaemon_status_info_set_description (SyncdaemonStatusInfo *sinfo, const gchar *description)
{
	g_return_if_fail (SYNCDAEMON_IS_STATUS_INFO (sinfo));

	if (sinfo->priv->description != NULL)
		g_free (sinfo->priv->description);

	sinfo->priv->description = g_strdup (description);
}

/**
 * syncdaemon_status_info_get_name:
 */
const gchar *
syncdaemon_status_info_get_name (SyncdaemonStatusInfo *sinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_STATUS_INFO (sinfo), NULL);

	return (const gchar *) sinfo->priv->name;
}

/**
 * syncdaemon_status_info_set_name:
 */
void
syncdaemon_status_info_set_name (SyncdaemonStatusInfo *sinfo, const gchar *name)
{
	g_return_if_fail (SYNCDAEMON_IS_STATUS_INFO (sinfo));

	if (sinfo->priv->name != NULL)
		g_free (sinfo->priv->name);

	sinfo->priv->name = g_strdup (name);
}

/**
 * syncdaemon_status_info_get_online:
 */
gboolean
syncdaemon_status_info_get_online (SyncdaemonStatusInfo *sinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_STATUS_INFO (sinfo), FALSE);

	return sinfo->priv->is_online;
}

/**
 * syncdaemon_status_info_set_online:
 */
void
syncdaemon_status_info_set_online (SyncdaemonStatusInfo *sinfo, gboolean online)
{
	g_return_if_fail (SYNCDAEMON_IS_STATUS_INFO (sinfo));

	sinfo->priv->is_online = online;
}

/**
 * syncdaemon_status_info_get_queues:
 */
const gchar *
syncdaemon_status_info_get_queues (SyncdaemonStatusInfo *sinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_STATUS_INFO (sinfo), NULL);

	return (const gchar *) sinfo->priv->queues;
}

/**
 * syncdaemon_status_info_set_queues:
 */
void
syncdaemon_status_info_set_queues (SyncdaemonStatusInfo *sinfo, const gchar *queues)
{
	g_return_if_fail (SYNCDAEMON_IS_STATUS_INFO (sinfo));

	if (sinfo->priv->queues != NULL)
		g_free (sinfo->priv->queues);

	sinfo->priv->queues = g_strdup (queues);
}
