# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The Qt graphical interface for the control panel for Ubuntu One."""

import collections
import logging

from functools import wraps

from PyQt4 import QtGui, QtCore
from twisted.internet import defer

from ubuntuone.controlpanel.gui import (
    FILE_URI_PREFIX,
    GENERAL_ERROR_TITLE,
    GENERAL_ERROR_MSG,
)


def uri_hook(uri):
    """Open an URI using the default browser/file manager."""
    if uri.startswith(FILE_URI_PREFIX):
        url = QtCore.QUrl(uri)
    else:
        url = QtCore.QUrl()
        url.setEncodedUrl(uri)
    QtGui.QDesktopServices.openUrl(url)


def pixmap_from_name(icon_name):
    """Build a QtGui.QPixmap from an icon name.

    If the theme does not have the icon, build one using the resources images.

    """
    pixmap_name = ":/%s.png" % icon_name
    return QtGui.QPixmap(pixmap_name)


def icon_from_name(icon_name):
    """Build a QtGui.QIcon from an icon name.

    If the theme does not have the icon, build one using the resources images.

    """
    if QtGui.QIcon.hasThemeIcon(icon_name):
        icon = QtGui.QIcon.fromTheme(icon_name)
    else:
        pixmap = pixmap_from_name(icon_name)
        icon = QtGui.QIcon(pixmap)

    icon.icon_name = icon_name

    return icon


# Invalid name "box"
# pylint: disable=C0103
box = None
# pylint: enable=C0103


def handle_errors(error_handler=None, logger=None):
    """Decorator to handle errors when calling a function.

    if 'error_handler' is not None, it will be yielded on if any error happens.

    """
    if logger is None:
        logger = logging.getLogger()

    def middle(f):
        """Decorator to handle errors when calling 'f'."""

        @defer.inlineCallbacks
        @wraps(f)
        def inner(*args, **kwargs):
            """Call 'f' passing 'args' and 'kwargs'.

            Catch any error and show a error message.

            """
            try:
                res = yield f(*args, **kwargs)
            except Exception, e:  # pylint: disable=W0703
                msg = 'Error while invoking %r with args %r and kwargs %r:'
                logger.exception(msg, f, args, kwargs)
            else:
                defer.returnValue(res)

            # this code will only be executed if there was an exception
            if error_handler is not None:
                yield error_handler()

            if len(e.args) > 0 and isinstance(e.args[0], collections.Mapping):
                msgs = e.args[0].itervalues()
            else:
                msgs = [e.__class__.__name__] + map(repr, e.args)
            msg = '\n'.join(msgs)

            # hack to avoid the QMessageBox being gargabe-collected too soon
            # Using the global statement, pylint: disable=W0603
            global box
            # pylint: enable=W0603
            box = QtGui.QMessageBox(QtGui.QMessageBox.Critical,
                    GENERAL_ERROR_TITLE, GENERAL_ERROR_MSG,
                    QtGui.QMessageBox.Close, parent=None)
            box.setDefaultButton(QtGui.QMessageBox.Close)
            box.setDetailedText(msg)

            # can not exec_() on the QMessageBox since that will block the Qt
            # mainloop

            d = defer.Deferred()

            def box_finished(result):
                """The QMessageBox finished."""
                logger.info('The warning dialog was shown and also closed '
                            '(message was %r).', msg)
                box.finished.disconnect(box_finished)
                box.hide()
                box.deleteLater()
                d.callback(result)

            box.finished.connect(box_finished)
            box.show()

            yield d

        return inner

    return middle
