# -*- coding: utf-8 -*-
#
# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the control panel's Qt main."""

import sys

from PyQt4 import QtCore
from twisted.internet.defer import inlineCallbacks

from ubuntuone.controlpanel.gui.qt import main
from ubuntuone.controlpanel.tests import TestCase


class FakeTranslator(object):

    """A fake QTranslator."""

    args = None

    def load(self, *args, **kwargs):
        """fake load."""
        self.args = (args, kwargs)


class FakeApplication(object):

    """A fake application."""

    def __init__(self):
        self.args = None
        self.style = None
        self.translator = None

    def __call__(self, argv, *args, **kwargs):
        """Fake arg filtering function."""
        if '-title' in argv:
            argv.remove('-title')
        self.args = (argv, args, kwargs)
        return self

    # pylint: disable=C0103
    def setStyleSheet(self, *args, **kwargs):
        """Fake setStyleSheet."""
        self.style = (args, kwargs)

    def installTranslator(self, translator):
        """Fake installTranslator."""
        self.translator = translator
    # pylint: enable=C0103

    def arguments(self):
        """Fake arguments function."""
        return self.args[0]

    def exec_(self):
        """Fake event loop."""
        pass


class FakeMainWindow(object):

    """A fake MainWindow."""

    tabname = None
    urgent = None

    def switch_to(self, tabname):
        """Fake switch_to."""
        self.tabname = tabname

    def set_urgent(self, value):
        """Fake set_urgent."""
        self.urgent = value


class FakeStart(object):

    """Fake start function."""

    def __init__(self):
        self.args = None
        self.window = None

    def __call__(self, *args, **kwargs):
        if kwargs.get('minimized', False):
            self.window = None
        else:
            self.window = FakeMainWindow()
        self.args = (args, kwargs)
        return None, self.window


class MainTestCase(TestCase):

    """Test the argument parsing and passing in main."""

    @inlineCallbacks
    def setUp(self):
        yield super(MainTestCase, self).setUp()
        self.app = FakeApplication()
        self.translator = FakeTranslator()
        self.start = FakeStart()
        self.patch(main, "UniqueApplication", self.app)
        self.patch(main, "start", self.start)
        self.patch(main.source, "main_start", lambda app: None)
        self.patch(QtCore, "QTranslator", lambda: self.translator)

    def test_wm_class(self):
        """Test that we set the 1st argument, used for WM_CLASS, correctly."""
        main.main([sys.argv[0]])
        self.assertEqual(self.app.args,
            (['ubuntuone-installer', sys.argv[0]],
            ('ubuntuone-control-panel',), {}))

    def test_title_not_fail(self):
        """Ensure -title is removed before it gets to argparse."""
        main.main([sys.argv[0], "-title"])
        # Did not crash!

    def test_truncate_argv(self):
        """Ensure the binary name is not given to argparse."""
        main.main(["foo", "bar", sys.argv[0], "--minimized"])
        self.assertEqual(self.start.args[1],
            {'minimized': True, 'with_icon': False, 'installer': False})

    def test_minimized_option(self):
        """Ensure the --minimized option is parsed and passed correctly."""
        main.main([sys.argv[0], "--minimized"])
        self.assertEqual(self.start.args[1],
            {'minimized': True, 'with_icon': False, 'installer': False})

    def test_with_icon_option(self):
        """Ensure the --minimized option is parsed and passed correctly."""
        main.main([sys.argv[0], "--with-icon"])
        self.assertEqual(self.start.args[1],
            {'minimized': False, 'with_icon': True, 'installer': False})

    def test_all_styles_load(self):
        """Ensure the platform style is loaded."""
        main.main([sys.argv[0]])
        data = []
        for qss_name in (":/ubuntuone.qss", main.source.PLATFORM_QSS):
            qss = QtCore.QResource(qss_name)
            data.append(unicode(qss.data()))
        self.assertEqual((('\n'.join(data),), {}), self.app.style)

    def test_switch_to_option(self):
        """Ensure the --switch-to option is parsed and passed correctly."""
        main.main([sys.argv[0], "--switch-to", "folders"])
        self.assertEqual(self.start.window.tabname, "folders")

    def test_installer_option(self):
        """Ensure the --installer option is parsed and passed correctly."""
        main.main([sys.argv[0], "--installer"])
        self.assertEqual(self.start.args[1],
            {'minimized': False, 'with_icon': False, 'installer': True})

    def test_minimized_with_icon_options(self):
        """Ensure you can be minimized and with icon at the same time."""
        main.main([sys.argv[0], "--minimized", "--with-icon"])
        self.assertEqual(self.start.args[1],
            {'minimized': True, 'with_icon': True, 'installer': False})

    def test_translator(self):
        """Ensure the Qt translator is loaded."""
        main.main([sys.argv[0]])
        locale = unicode(QtCore.QLocale.system().name())
        self.assertEqual(self.app.translator, self.translator)
        self.assertEqual(self.translator.args, (("qt_" + locale,
            QtCore.QLibraryInfo.location(
                QtCore.QLibraryInfo.TranslationsPath)), {}))
