# -*- coding: utf-8 -*-

# Author: Roberto Alsina <roberto.alsina@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the notification area icon."""

from PyQt4 import QtGui
from twisted.internet.defer import inlineCallbacks

from ubuntuone.controlpanel.gui.qt import systray
from ubuntuone.controlpanel.tests import TestCase
import ubuntuone.controlpanel.gui.qt.gui


class FakeSDTool(object):

    """Fake SyncDaemonTool."""

    called = False

    def quit(self):
        """Fake quit."""
        self.called = True


class FakeMainWindow(QtGui.QWidget):

    """Fake Main Window."""

    def __init__(self, *args, **kwargs):
        self.args = (args, kwargs)
        super(FakeMainWindow, self).__init__()


class SystrayTestCase(TestCase):

    """Test the notification area icon."""

    def test_quit(self):
        """Test the quit option in the menu."""
        # Not done on setup, because if I patch stop
        # after instantiation, it doesn't get called.
        self.patch(systray.TrayIcon, "stop", self._set_called)
        tray = systray.TrayIcon()
        tray.quit.trigger()
        self.assertEqual(self._called, ((False,), {}))

    @inlineCallbacks
    def test_stop_sd(self):
        """Quit should call SyncDaemonTool.quit()."""
        st = FakeSDTool()
        self.patch(systray, "SyncDaemonTool", lambda: st)
        tray = systray.TrayIcon()
        yield tray.stop()
        self.assertTrue(st.called)

    def test_restore_no_window(self):
        """Test the restore window option in the menu, with no window."""
        self.patch(ubuntuone.controlpanel.gui.qt.gui,
            "MainWindow", FakeMainWindow)
        tray = systray.TrayIcon()
        self.assertEqual(tray.window, None)
        tray.restore.trigger()
        self.assertIsInstance(tray.window, FakeMainWindow)
        self.assertTrue(tray.window.isVisible())
        self.assertEqual(tray.window.args, ((),
            {'close_callback': tray.delete_window}))

    def test_activate(self):
        """Test the icon activation."""
        tray = systray.TrayIcon()
        window = FakeMainWindow()
        tray.window = window
        self.assertFalse(tray.window.isVisible())
        tray.activated.emit(tray.Trigger)
        self.assertEqual(tray.window, window)
        self.assertTrue(tray.window.isVisible())

    def test_restore_window(self):
        """Test the restore window option in the menu, with a window."""
        tray = systray.TrayIcon()
        window = FakeMainWindow()
        tray.window = window
        self.assertFalse(tray.window.isVisible())
        tray.restore.trigger()
        self.assertEqual(tray.window, window)
        self.assertTrue(tray.window.isVisible())

    def test_delete_window(self):
        """Test deleting an existing window."""
        tray = systray.TrayIcon()
        window = FakeMainWindow()
        tray.window = window
        tray.delete_window()
        self.assertEqual(tray.window, None)
        self.assertFalse(window.isVisible())

    def test_delete_no_window(self):
        """Test deleting without an existing window."""
        tray = systray.TrayIcon()
        tray.delete_window()
        self.assertEqual(tray.window, None)

    def test_initialization(self):
        """Test that everything initializes correctly."""
        tray = systray.TrayIcon()
        self.assertTrue(tray.isVisible())
        self.assertEqual(tray.window, None)
        self.assertIsInstance(tray.context_menu, QtGui.QMenu)
        self.assertFalse(tray.icon() == None)
