/*
 * Copyright (C) 2014-2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include <click.h>

#include "interface.h"

#include <QDebug>
#include <QDir>
#include <QProcess>
#include <QStandardPaths>
#include <QString>
#include <QTimer>

#include <cstdio>
#include <list>
#include <sys/stat.h>
#include <map>
#include <sstream>

#include <boost/locale/collator.hpp>
#include <boost/locale/generator.hpp>

#include <boost/property_tree/ptree.hpp>
#include <boost/property_tree/json_parser.hpp>
#include <boost/property_tree/exceptions.hpp>
#include <boost/foreach.hpp>

#include <unity/UnityExceptions.h>
#include <unity/util/IniParser.h>

#include <click/departments-db.h>

#include <ubuntu-app-launch/registry.h>

#include <click/click-i18n.h>

namespace ual = ubuntu::app_launch;

namespace {

/* Thanks to
 *   - http://stackoverflow.com/a/14031349
 *   - http://stackoverflow.com/questions/12278448/removing-accents-from-a-qstring
 */
QString unaccent(const QString &str)
{
    QString tmp = str.normalized(QString::NormalizationForm_KD,
                                 QChar::currentUnicodeVersion());
    QString ret;
    for (int i = 0, j = tmp.length();
         i < j;
         i++) {

        // strip diacritic marks
        if (tmp.at(i).category() != QChar::Mark_NonSpacing       &&
            tmp.at(i).category() != QChar::Mark_SpacingCombining &&
            tmp.at(i).category() != QChar::Mark_Enclosing) {
            ret.append(tmp.at(i));
        }
    }

    return ret;
}

}

namespace click {

const std::unordered_set<std::string>& nonClickDesktopFiles()
{
    static std::unordered_set<std::string> set =
    {
        "address-book-app.desktop",
        "camera-app.desktop",
        "click-update-manager.desktop",
        "com.ubuntu.terminal.desktop",
        "dialer-app.desktop",
        "friends-app.desktop",
        "gallery-app.desktop",
        "mediaplayer-app.desktop",
        "messaging-app.desktop",
        "music-app.desktop",
        "ubuntu-filemanager-app.desktop",
        "ubuntu-system-settings.desktop",
        "webbrowser-app.desktop",
    };

    return set;
}

static const std::string DESKTOP_FILE_GROUP("Desktop Entry");
static const std::string DESKTOP_FILE_KEY_NAME("Name");
static const std::string DESKTOP_FILE_KEY_ICON("Icon");
static const std::string DESKTOP_FILE_KEY_KEYWORDS("Keywords");
static const std::string DESKTOP_FILE_KEY_APP_ID("X-Ubuntu-Application-ID");
static const std::string DESKTOP_FILE_KEY_DOMAIN("X-Ubuntu-Gettext-Domain");
static const std::string DESKTOP_FILE_UBUNTU_TOUCH("X-Ubuntu-Touch");
static const std::string DESKTOP_FILE_UBUNTU_DEFAULT_DEPARTMENT("X-Ubuntu-Default-Department-ID");
static const std::string DESKTOP_FILE_COMMENT("Comment");
static const std::string DESKTOP_FILE_SCREENSHOT("X-Screenshot");
static const std::string DESKTOP_FILE_NODISPLAY("NoDisplay");
static const std::string DESKTOP_FILE_ONLYSHOWIN("OnlyShowIn");
static const std::string ONLYSHOWIN_UNITY("Unity");


std::vector<click::Application> Interface::sort_apps(const std::vector<click::Application>& apps)
{
    std::vector<click::Application> result = apps;
    boost::locale::generator gen;
    const char* lang = getenv(click::Configuration::LANGUAGE_ENVVAR);
    if (lang == NULL) {
        lang = "C.UTF-8";
    }
    std::locale loc = gen(lang);
    std::locale::global(loc);
    typedef boost::locale::collator<char> coll_type;

    // Sort applications alphabetically.
    std::sort(result.begin(), result.end(), [&loc](const Application& a,
                                                   const Application& b) {
                  bool lesser = false;
                  int order = std::use_facet<coll_type>(loc)
                      .compare(boost::locale::collator_base::quaternary,
                               a.title, b.title);
                  if (order == 0) {
                      lesser = a.name < b.name;
                  } else {
                      // Because compare returns int, not bool, we have to check
                      // that 0 is greater than the result, which tells us the
                      // first element should be sorted priori
                      lesser = order < 0;
                  }
                  return lesser;
              });

    return result;
}

std::list<std::shared_ptr<ual::Application>> Interface::installed_apps() const
{
    return ual::Registry::installedApps();
}

/* search()
 *
 * Find all of the installed apps matching @query in a timeout.
 */
std::vector<click::Application> Interface::search(const std::string& query,
        const std::vector<std::string>& ignored_apps,
        const std::string& current_department,
        const std::shared_ptr<click::DepartmentsDb>& depts_db)
{
    //
    // only apply department filtering if not in root of all departments.
    bool apply_department_filter = !current_department.empty();

    // get the set of packages that belong to current deparment;
    std::unordered_set<std::string> packages_in_department;
    if (depts_db && apply_department_filter)
    {
        try
        {
            packages_in_department = depts_db->get_packages_for_department(current_department);
        }
        catch (const std::exception& e)
        {
            qWarning() << "Failed to get packages of department" << QString::fromStdString(current_department);
            apply_department_filter = false; // disable so that we are not loosing any apps if something goes wrong
        }
    }

    std::vector<click::Application> result;

    for (const auto& ualapp: installed_apps()) {
        click::Application app;

        // Get the package name and APP_ID info.
        app.name = ualapp->appId().package.value();
        app.version = ualapp->appId().version.value();
        if (app.name.empty()) {
            app.name = std::string{ualapp->appId().appname.value()} + ".desktop";
            app.url = "application:///" + app.name;
        } else {
            app.url = "appid://" + app.name + "/" + ualapp->appId().appname.value() + "/current-user-version";
        }

        if (std::find(ignored_apps.begin(), ignored_apps.end(),
                      app.name) != ignored_apps.end()) {
            // The app is ignored. Get out of here.
            qDebug() << "App is ignored, skipping:" << QString::fromStdString(app.name);
            continue;
        }

        // Get the .desktop file info from UAL, since we're not ignoring
        try {
            auto appinfo = ualapp->info();

            // Only skip legacy apps that don't support Ubuntu Lifecycle
            if (app.version.empty() &&
                !is_non_click_app(app.name) &&
                !appinfo->supportsUbuntuLifecycle()) {
                qDebug() << "Skipping legacy app:" << QString::fromStdString(app.name);
                continue;
            }

            app.title = appinfo->name();
            app.description = appinfo->description();
            app.icon_url = appinfo->iconPath();
            app.default_department = appinfo->defaultDepartment();
            app.main_screenshot = appinfo->screenshotPath();
            app.keywords = appinfo->keywords();
        } catch (const std::exception& e) {
            qDebug() << "Unable to get info, skipping:" << QString::fromStdString(app.name);
            continue;
        }

        // app from click package has non-empty name; for non-click apps use desktop filename
        const auto& department_key = app.name;
        qDebug() << "Using department key:" << QString::fromStdString(department_key);

        // check if apps is present in current department
        if (apply_department_filter) {
            if (packages_in_department.find(department_key) ==
                packages_in_department.end()) {
                if (app.default_department.empty()) {
                    // default department not present in the keyfile, skip this app
                    continue;
                } else {
                    // default department not empty: check if this app is in a different
                    // department in the db (i.e. got moved from the default department);
                    if (depts_db->has_package(department_key)) {
                        // app is now in a different department
                        continue;
                    }

                    if (app.default_department != current_department) {
                        continue;
                    }
                    // else - this package is in current department
                }
            }
        }

        // the packages_in_department set contains packages from
        // all its subdepartments; we need to find actual department now
        // to update app.real_department.
        if (depts_db) {
            if (depts_db->has_package(department_key)) {
                try {
                    app.real_department = depts_db->get_department_for_package(department_key);
                } catch (const std::exception &e) {
                    qWarning() << "Failed to get department of package:" << QString::fromStdString(department_key);
                }
            } else {
                app.real_department = app.default_department;
                if (app.real_department.empty()) {
                    qWarning() << "No default department set in the .desktop file and no entry in the database for" << QString::fromStdString(department_key);
                }
            }
        }

        if (query.empty()) {
            result.push_back(app);
        } else {
            QString lquery = ::unaccent(QString::fromStdString(query));

            // Check keywords for the search query as well.
            for (const auto& kwd: app.keywords) {
                QString keyword = ::unaccent(QString::fromStdString(kwd));
                if (keyword.contains(lquery, Qt::CaseInsensitive)) {
                    result.push_back(app);
                }
            }

            QString search_title = ::unaccent(QString::fromStdString(app.title));
            // check the app title for the search query.
            if (search_title.contains(lquery, Qt::CaseInsensitive)) {
                result.push_back(app);
            }
        }
    }

    return sort_apps(result);
}

/* is_non_click_app()
 *
 * Tests that @filename is one of the special-cased filenames for apps
 * which are not packaged as clicks, but required on Ubuntu Touch.
 */
bool Interface::is_non_click_app(const std::string& app_id)
{
    return click::nonClickDesktopFiles().count(app_id) > 0;
}

/*
 * is_icon_identifier()
 *
 * Checks if @filename has no / in it
 */
bool Interface::is_icon_identifier(const std::string &icon_id)
{
    return icon_id.find("/") == std::string::npos;
}

/*
 * add_theme_scheme()
 *
 * Adds the theme prefix if the filename is not an icon identifier
 */
std::string Interface::add_theme_scheme(const std::string& icon_id)
{
    if (is_icon_identifier(icon_id)) {
        return "image://theme/" + icon_id;
    }
    return icon_id;
}

Manifest manifest_from_json(const std::string& json)
{
    using namespace boost::property_tree;

    std::istringstream is(json);

    ptree pt;
    read_json(is, pt);

    Manifest manifest;

    manifest.name = pt.get<std::string>("name");
    manifest.version = pt.get<std::string>("version");
    manifest.removable = pt.get<bool>("_removable");

    BOOST_FOREACH(ptree::value_type &sv, pt.get_child("hooks"))
    {
        // FIXME: "primary app or scope" for a package is not defined,
        // we just use the first one in the manifest:
        auto app_name = sv.second.get("desktop", "");
        if (manifest.first_app_name.empty() && !app_name.empty()) {
            manifest.first_app_name = sv.first;
        }
        auto scope_id = sv.second.get("scope", "");
        if (manifest.first_scope_id.empty() && !scope_id.empty()) {
            manifest.first_scope_id = manifest.name + "_" + sv.first;
        }
    }
    qDebug() << "adding manifest: " << manifest.name.c_str() << manifest.version.c_str() << manifest.first_app_name.c_str();

    return manifest;
}

void Interface::get_installed_packages(std::function<void(PackageSet, InterfaceError)> callback)
{
    PackageSet packages;
    for (const auto& app: installed_apps()) {
        Package p;
        const auto appid = app->appId();
        p.name = appid.package.value();
        p.version = appid.version.value();
        if (!p.name.empty() && !p.version.empty()) {
            packages.insert(p);
        }
    }
    callback(packages, InterfaceError::NoError);
}

std::string Interface::get_manifest_json(const std::string &package) const
{
    GError* error = nullptr;

    std::shared_ptr<ClickDB> clickdb{click_db_new(),
            [](ClickDB* db){g_clear_object(&db);}};
    click_db_read(clickdb.get(), nullptr, &error);
    if (error != nullptr) {
        qCritical() << "Error reading click DB:" << error->message;
        g_error_free(error);
        return "";
    }

    std::shared_ptr<ClickUser> clickuser{click_user_new_for_user(clickdb.get(),
                                                                 nullptr,
                                                                 &error),
            [](ClickUser* cu){g_clear_object(&cu);}};
    if (error != nullptr) {
        qCritical() << "Error setting up click user:" << error->message;
        g_error_free(error);
        return "";
    }

    auto result = click_user_get_manifest_as_string(clickuser.get(),
                                                    package.c_str(),
                                                    &error);
    if (error != nullptr) {
        qCritical() << "Error getting manifest:" << error->message;
        g_error_free(error);
        return "";
    }

    std::string retval;
    if (result != nullptr) {
        retval = result;
        g_free(result);
    }
    return retval;
}

void Interface::get_manifest_for_app(const std::string &app_id,
                                     std::function<void(Manifest, InterfaceError)> callback)
{
    Manifest manifest;
    InterfaceError error;
    try {
        manifest = manifest_from_json(get_manifest_json(app_id));
        error = InterfaceError::NoError;
    } catch (...) {
        qWarning() << "Can't parse manifest for:" << QString::fromStdString(app_id);
        error = InterfaceError::ParseError;
    }
    callback(manifest, error);
}

} // namespace click
