/*
 * Copyright (C) 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include <ubuntu-app-launch/registry.h>

namespace ual = ubuntu::app_launch;

#include <gtest/gtest.h>
#include <gmock/gmock.h>

using namespace ::testing;

namespace
{

    class MockUALRegistry : public ual::Registry {
    public:
        MockUALRegistry()
        {
        }

        MOCK_METHOD1(installedApps, std::list<std::shared_ptr<ual::Application>>(std::shared_ptr<ual::Registry>));
    };

    class MockUALApplication : public ual::Application {
    public:
        class MockInfo;
        class MockInstance;

        MockUALApplication(const ual::AppID& app_id,
                           const std::shared_ptr<MockInfo>& info = {})
            : m_app_id(app_id),
              m_info(info)
        {
        }

        ual::AppID appId()
        {
            return m_app_id;
        }

        std::shared_ptr<ual::Application::Info> info()
        {
            if (!m_info) {
                throw std::runtime_error("Invalid application.");
            }
            return m_info;
        }

        static std::shared_ptr<MockUALApplication> create(ual::AppID app_id,
                                                          std::shared_ptr<MockInfo> info = {})
        {
            std::shared_ptr<MockUALApplication> result{new MockUALApplication(app_id, info)};
            return result;
        }

        class MockInfo : public ual::Application::Info
        {
        public:
            MockInfo(const std::string& title,
                     const std::string& description,
                     const std::string& iconPath)
                : m_name(ual::Application::Info::Name::from_raw(title)),
                m_description(ual::Application::Info::Description::from_raw(description)),
                m_iconPath(ual::Application::Info::IconPath::from_raw(iconPath)),
                m_defaultDept(ual::Application::Info::DefaultDepartment::from_raw("")),
                m_ssPath(ual::Application::Info::IconPath::from_raw("")),
                m_keywords(ual::Application::Info::Keywords::from_raw(std::vector<std::string>{}))
            {
                DefaultValue<const ual::Application::Info::DefaultDepartment&>::Set(m_defaultDept);
                DefaultValue<const ual::Application::Info::IconPath&>::Set(m_ssPath);
                DefaultValue<const ual::Application::Info::Keywords&>::Set(m_keywords);
                DefaultValue<ual::Application::Info::UbuntuLifecycle>::Set(ual::Application::Info::UbuntuLifecycle::from_raw(false));
            }

            const ual::Application::Info::Name& name()
            {
                return m_name;
            }

            const ual::Application::Info::Description& description()
            {
                return m_description;
            }

            const ual::Application::Info::IconPath& iconPath()
            {
                return m_iconPath;
            }

            MOCK_METHOD0(defaultDepartment, const ual::Application::Info::DefaultDepartment&());
            MOCK_METHOD0(screenshotPath, const ual::Application::Info::IconPath&());
            MOCK_METHOD0(keywords, const ual::Application::Info::Keywords&());

            MOCK_METHOD0(splash, ual::Application::Info::Splash());
            MOCK_METHOD0(supportedOrientations, ual::Application::Info::Orientations());
            MOCK_METHOD0(rotatesWindowContents, ual::Application::Info::RotatesWindow());
            MOCK_METHOD0(supportsUbuntuLifecycle, ual::Application::Info::UbuntuLifecycle());

        private:
            ual::Application::Info::Name m_name;
            ual::Application::Info::Description m_description;
            ual::Application::Info::IconPath m_iconPath;
            ual::Application::Info::DefaultDepartment m_defaultDept;
            ual::Application::Info::IconPath m_ssPath;
            ual::Application::Info::Keywords m_keywords;
        };

        class MockInstance : public ual::Application::Instance
        {
        public:
            MOCK_METHOD0(isRunning, bool());
            MOCK_METHOD0(logPath, std::string());
            MOCK_METHOD0(primaryPid, pid_t());
            MOCK_METHOD1(hasPid, bool(pid_t));
            MOCK_METHOD0(pids, std::vector<pid_t>());
            MOCK_METHOD0(pause, void());
            MOCK_METHOD0(resume, void());
            MOCK_METHOD0(stop, void());
        };

        MOCK_METHOD0(hasInstances, bool());
        MOCK_METHOD0(instances, std::vector<std::shared_ptr<ual::Application::Instance>>());
        MOCK_METHOD1(launch, std::shared_ptr<ual::Application::Instance>(const std::vector<ual::Application::URL>&));
        MOCK_METHOD1(launchTest, std::shared_ptr<ual::Application::Instance>(const std::vector<ual::Application::URL>&));

    private:
        ual::AppID m_app_id;
        std::shared_ptr<MockInfo> m_info;
    };

} // namespace

