/* ExpressionLoader.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.expr;

import org.grinvin.invariants.InvariantManager;
import org.grinvin.invariants.UnknownInvariantException;
import org.grinvin.io.IOFormatException;

import org.jdom.DataConversionException;
import org.jdom.Element;

/**
 * Helper methods for loading expressions from disk.
 */
public final class ExpressionLoader {
    
    private ExpressionLoader() {
        // not to be instantiated
    }
    
    public static Expression loadExpression(Element element) throws IOFormatException {
        String name = element.getName();
        if ("integer".equals(name)) {
            return loadIntegerLiteral(element);
        } else if ("invariant".equals(name)) {
            return loadInvariant(element);
        } else if ("compound".equals(name)) {
            return loadCompound(element);
        } else
            throw new IOFormatException("Invalid name of expression element");
    }
    
    private static Expression loadIntegerLiteral(Element element) throws IOFormatException {
        try {
            return new IntegerLiteral(
                    element.getAttribute("value").getIntValue()
                    );
        } catch (DataConversionException ex) {
            throw new IOFormatException("No valid value for integer literal", ex);
        }
    }
    
    private static Expression loadInvariant(Element element) throws IOFormatException {
        try {
            String id = element.getAttributeValue("id");
            if (id == null)
                id = element.getAttributeValue("invariantId");
            return InvariantManager.getInstance().getInvariantExpressionNode(id);
        } catch (UnknownInvariantException ex) {
            throw new IOFormatException("Unknown invariant in expression", ex);
        }
    }
    
    private static Expression loadCompound(Element element) throws IOFormatException {
        Operator operator = StandardOperator.valueOf(StandardOperator.class, element.getAttributeValue("operator"));
        if (operator == null)
            throw new IOFormatException ("Unkown operator");
        int nr = operator.getNumberOfArguments();
        Expression[] children = new Expression[nr];
        int i = 0;
        for (Object obj : element.getChildren()) {
            Element child = (Element)obj;
            if (i >= nr)
                throw new IOFormatException ("Too many arguments in compound expression");
            children[i] =loadExpression (child);
            i++;
        }
        if (i < nr)
                throw new IOFormatException ("Not enough arguments in compound expression");
        return new Compound (operator, children);
    }
    
}
