/* AbstractInvariantValue.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.values;

import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantComputer;
import org.grinvin.invariants.InvariantManager;
import org.grinvin.invariants.InvariantType;
import org.grinvin.invariants.InvariantValue;

/**
 * Wraps the result of an invariant computation and provides a
 * means to represent this result both as a string and an XML element.
 */
public abstract class AbstractInvariantValue implements InvariantValue {

    //
    private Invariant invariant;
    
    //
    private String computerId;
    
    //
    private String computerVersion;
    
    //
    private final InvariantType type;

    /**
     * Return the invariant of which this is a value.
     */
    public Invariant getInvariant() {
        return invariant;
    }

    public String getComputerId() {
        return computerId;
    }

    public String getComputerVersion() {
        return computerVersion;
    }

    public InvariantType getType() {
        return type;
    }

    /**
     * Create a value of the given type which was computed by the given invariant computer.
     */
    protected AbstractInvariantValue(InvariantType type, InvariantComputer computer) {
        this.computerId = computer.getId();
        this.computerVersion = computer.getVersion();
        this.invariant = InvariantManager.getInstance().getInvariantForComputer(computer);
        this.type = type;
    }

    AbstractInvariantValue(InvariantType type, Invariant invariant, String computerId, String computerVersion) {
        this.type = type;
        this.invariant = invariant;
        this.computerId = computerId;
        this.computerVersion = computerVersion;
    }

    protected AbstractInvariantValue(InvariantType type) {
        this.type = type;
    }

    /**
     * Convert this value to a string. 
     */
    public abstract String toText();

    //
    @Override
    public String toString() {
        return invariant.getId() + " computed by " + computerId + " (" + computerVersion + ") : " + toText();
    }
    
}
