/* GaussElimination.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.util;

import java.math.BigInteger;
import java.util.Arrays;

/**
 *
 */
public class GaussElimination {
    
    private BigInteger[][] matrix;
    
    private int rank = -1;
    
    /**
     * Creates a new instance of GaussElimination
     */
    public GaussElimination(int[][] matrix) {
        try {
            this.matrix = new BigInteger[matrix.length][matrix[0].length];
        } catch (ArrayIndexOutOfBoundsException ex) {
            this.matrix = new BigInteger[matrix.length][0];
        }
        for(int i = 0; i < matrix.length; i++)
            for(int j = 0; j < matrix[i].length; j++)
                this.matrix[i][j] = BigInteger.valueOf(matrix[i][j]);
    }
    
    public void doElimination() {
        doElimination(0);
    }
    
    private void doElimination(int currentRow) {        
        //make sure first element is non-zero
        int columnsMoved = 0;
        while(matrix[currentRow][currentRow].equals(BigInteger.ZERO) && columnsMoved <= matrix[0].length - currentRow){
            int position = currentRow;
            while(position < matrix.length && matrix[position][currentRow].equals(BigInteger.ZERO))
                position++;
            if(position==matrix.length) { //only zeroes in this column
                moveColumnToEnd(currentRow);
                columnsMoved++;
            } else
                switchRows(currentRow, position);
        }
        
        if(matrix[currentRow][currentRow].equals(BigInteger.ZERO)){ //there are only zeroes left: elimination is done
            rank = currentRow;
            return;
        }
        
        for(int i = currentRow+1; i<matrix.length; i++){
            //find lcm
            BigInteger gcd = matrix[i][currentRow].gcd(matrix[currentRow][currentRow]);
            BigInteger factor1 = matrix[i][currentRow].divide(gcd);
            BigInteger factor2 = matrix[currentRow][currentRow].divide(gcd);
            substractRows(currentRow, currentRow, factor1, i, factor2);
            simplifyRow(i);
        }
        
        if(currentRow<matrix.length-1 && currentRow<matrix[0].length-1)
            doElimination(currentRow+1);
        else
            rank = currentRow;
    }
    
    private void switchRows (int row1, int row2){
        BigInteger[] temp = matrix[row1];
        matrix[row1] = matrix[row2];
        matrix[row2] = temp;
    }
    
    private void switchColumns (int col1, int col2){
        BigInteger temp;
        for(BigInteger[] integer : matrix){
            temp = integer[col1];
            integer[col1] = integer[col2];
            integer[col2] = temp;
        }
    }
    
    private void moveColumnToEnd (int col){
        for(int i = col; i < matrix[0].length - 1; i++)
            switchColumns(i, i+1);
    }
    
    private void multiplyRow(int row, BigInteger factor){
        for(int i = 0; i<matrix[row].length; i++)
            matrix[row][i] = matrix[row][i].multiply(factor);
    }
    
    private void simplifyRow(int row){
        //find gcd
        BigInteger gcd;
        int i = 0;
        while(i<matrix[row].length && matrix[row][i].equals(BigInteger.ZERO))
            i++;
        if(i==matrix[row].length)
            return; //row only contains zeroes
        else
            gcd = matrix[row][i];
        for(int j = i; j<matrix[row].length; j++)
            if(!matrix[row][j].equals(BigInteger.ZERO))
                gcd = gcd.gcd(matrix[row][j]);
        
        //divide all elements by gcd
        for(int j = 0; j<matrix[row].length; j++)
            matrix[row][j] = matrix[row][j].divide(gcd);
    }
    
    private void substractRows(int start, int row1, BigInteger factor1, int row2, BigInteger factor2){
        for(int i = start; i<matrix[row1].length; i++)
            matrix[row2][i] = matrix[row2][i].multiply(factor2).subtract(matrix[row1][i].multiply(factor1));
    }
    
    //todo: this may be removed after debugging???
    public void print(){
        for(BigInteger[] integer2 : matrix)
            System.out.println(Arrays.toString(integer2));
        System.out.println();
    }
    
    public int getRank(){
        if(rank==-1)
            doElimination();
        return rank;
    }
    
    public static void main(String[] a){
        int[][] m = new int[5][5];
        for(int i = 0; i<5; i++)
            for(int j = i; j<5; j++)
                m[i][j] = (i+1)*(j+1) + 1;
        GaussElimination g = new GaussElimination(m);
        g.print();
        g.switchRows(1,3);
        g.print();
        g.doElimination();
        g.print();
    }
}
