/* LongestInducedCycle.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard;

import java.util.Arrays;

import org.grinvin.graphs.GraphBundleView;
import org.grinvin.invariants.computers.AbstractInvariantComputer;
import org.grinvin.invariants.values.IntegerValue;

/**
 * computes the longest induced cycle of the graph.
 */
public class LongestInducedCycle extends AbstractInvariantComputer {
    
    public IntegerValue compute(GraphBundleView bundle) {
        int[][] adjlist = bundle.adjacencyList();
        int n = adjlist.length;
        if(n <= 2)
            //can't contain a cycle: longest induced cycle = undefined
            return IntegerValue.undefinedValue(this);
        
        /*
        for(int i = 0; i<n; i++)
            System.out.println(i + " " + Arrays.toString(adjlist[i]));
        /**/
        
        int[] capacity = new int[n];
        for(int i = 0; i<n; i++)
            capacity[i] = adjlist[i].length;
        removeBranches(capacity, adjlist);
        
        int[] currentPath = new int[n];
        Arrays.fill(currentPath, 0);
        int longestInducedCycle=0;
        for(int i = 0; i < n; i++)
            if(capacity[i]>0)
                longestInducedCycle = longestInducedCycle(adjlist, i, i, 0, currentPath, longestInducedCycle, capacity);
        if(longestInducedCycle==0)
            //acyclic graph: longest induced cycle = undefined
            return IntegerValue.undefinedValue(this);
        return new IntegerValue(longestInducedCycle, this);
    }
    
    private int longestInducedCycle(int[][] adjlist, int startVertex, int currentVertex, int depth, int[] currentPath, int maximum, int[] capacity){
        if(maximum == adjlist.length || maximum >= adjlist.length - startVertex)
            //added check to see if maximum >= adjlist.length - startVertex. Any cycle found from this point on can only contain
            //at most adjlist.length - startVertex vertices so no need to keep on looking.
            return maximum;
        currentPath[currentVertex] = depth;
        /*
        System.out.println("Start vertex: " + startVertex);
        System.out.println("Current vertex: " + currentVertex);
        System.out.println("Current path: " + Arrays.toString(currentPath));
        System.out.println("Depth: " + depth);
        System.out.println("Maximum: " + maximum);
        System.out.println();
        /**/
        for(int nextVertex : adjlist[currentVertex]){
            boolean goodNextVertex = currentPath[nextVertex]==0 && nextVertex != startVertex && capacity[nextVertex] > 0;
            int depthClosingVertex = -1;
            if(currentPath[nextVertex]!=0)
                depthClosingVertex = currentPath[nextVertex];
            for(int neighbor : adjlist[nextVertex])
                if(neighbor != currentVertex && (neighbor == startVertex || currentPath[neighbor]!=0)){
                    goodNextVertex = false;
                    if(depthClosingVertex < currentPath[neighbor])
                        depthClosingVertex = currentPath[neighbor];
                }
            if(goodNextVertex)
                maximum = longestInducedCycle(adjlist, startVertex, nextVertex, depth+1, currentPath, maximum, capacity);
            else if(depthClosingVertex >= 0 && maximum < (depth + 1) - depthClosingVertex + 1)
                //maximum = depth of nextVertex - depth of neighbor + 1
                maximum = (depth + 1) - depthClosingVertex + 1;
        }
        currentPath[currentVertex] = 0;
        return maximum;
    }
    
    //remove all branches starting from the leaves, because they cannot be part of a cycle
    private void removeBranches(int[] capacity, int[][] adjlist){
        boolean removedleaves = false;
        //System.out.println("Capacity: " + Arrays.toString(capacity));
        for(int i=0; i<capacity.length; i++)
            if(capacity[i]==1){
            for(int neighbor : adjlist[i])
                capacity[neighbor]--;
            capacity[i]=0;
            removedleaves=true;
            }
        if(removedleaves)
            removeBranches(capacity, adjlist);
    }
    
    public String getInvariantId() {
        return "org.grinvin.invariants.LongestInducedCycle";
    }

    public String getVersion() {
        return "1.0.1";
    }   
}
