// ---------------------------------------------------------------------------
// - List.hpp                                                                -
// - standard object library - doubly linked list class definition           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2011 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_LIST_HPP
#define  AFNIX_LIST_HPP

#ifndef  AFNIX_SERIAL_HPP
#include "Serial.hpp"
#endif

#ifndef  AFNIX_ITERABLE_HPP
#include "Iterable.hpp"
#endif

namespace afnix {

  /// The List class implements the behavior of a doubly linked list. Each
  /// object can be added at the beginning or at the end of the list. The
  /// list can be either accessed by iteration or by index. Needless to say
  /// that index access is very slow.
  /// @author amaury darsch

  class List : public Iterable, public Serial {
  private:
    /// the root list 
    struct s_list* p_root;
    /// the last object in the list
    struct s_list* p_last;

  public:
    /// create an empty ist
    List (void);

    /// copy constructor for this list
    /// @param that the list to copy 
    List (const List& that);

    /// destroy this list
    ~List (void);

    /// @return the class name
    String repr (void) const;

    /// @return the cons cell serial code
    t_byte serialid (void) const;

    /// serialize this cons cell to an output stream
    /// @param os the output stream to write
    void wrstream (class OutputStream& os) const;

    /// deserialize a cons cell from an input stream
    /// @param is the input steam to read in
    void rdstream (class InputStream& is);

    /// assign a list to this one
    /// @param that the list to assign
    List& operator = (const List& that);

    /// @return the number of objects in this list
    long length (void) const;

    /// @return an object by index
    Object* get (const long index) const;

    /// add an object to the end of the list
    /// @param object the object to add
    void add (Object* object);

    /// insert an object at the beginning of the list
    /// @param object the object to insert
    void insert (Object* object);

    /// @return a new iterator for this list
    Iterator* makeit (void);

  private:
    // make the list iterator a friend
    friend class Listit;

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param args  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  };

  /// The Listit class is the iterator for the list class. Such iterator
  /// is constructed with the "makeit" list method. The iterator is reset to
  /// the beginning of the list
  /// @author amaury darsch

  class Listit : public Iterator {
  private:
    /// the list to iterate
    List* p_list;
    /// the current node of iteration
    struct s_list* p_node;

  public:
    /// create a new iterator from a list
    /// @param lst the list to iterate
    Listit (List* lst);

    /// destroy this list iterator
    ~Listit (void);

    /// @return the class name
    String repr (void) const;

    /// reset the iterator to the begining
    void begin (void);

    /// reset the iterator to the end
    void end (void);

    /// move the list iterator to the next position
    void next (void);

    /// move the list iterator to the previous position
    void prev (void);

    /// @return the object at the current position
    Object* getobj (void) const;

    /// @return true if the iterator is at the end
    bool isend (void);

  private:
    // make the copy constructor private
    Listit (const Listit&);
    // make the assignment operator private
    Listit& operator = (const Listit&);
  };
}

#endif
