// ---------------------------------------------------------------------------
// - Rco.cpp                                                                 -
// - afnix:csm module - registered cloud object class implementation         -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2011 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Rco.hpp"
#include "Vector.hpp"
#include "Boolean.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a nil object

  Rco::Rco (void) {
    p_rco = nilp;
  }

  // create an anonymous object

  Rco::Rco (Object* rco) {
    Object::iref (p_rco = rco);
  }

  // create a registered object

  Rco::Rco (const String& rid, Object* rco) {
    d_rid = rid;
    Object::iref (p_rco = rco);
  }

  // copy construct this registered object

  Rco::Rco (const Rco& that) {
    that.rdlock ();
    try {
      d_rid = that.d_rid;
      Object::iref (p_rco = that.p_rco);
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }

  // destroy this registered object

  Rco::~Rco (void) {
    Object::dref (p_rco);
  }

  // return the rco class name

  String Rco::repr (void) const {
    return "Rco";
  }

  // return a clone of this object

  Object* Rco::clone (void) const {
    return new Rco (*this);
  }

  // assign a registered object to this one

  Rco& Rco::operator = (const Rco& that) {
    // check for self assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.rdlock ();
    try {
      // protect the copy
      Object::iref (that.p_rco);
      // remove old object
      Object::dref (p_rco);
      // set local data
      d_rid = that.d_rid;
      p_rco = that.p_rco;
      unlock ();
      that.unlock ();
      return *this;
    } catch (...) {
      unlock ();
      that.unlock ();
      throw;
    }
  }

  // compare an rco with another one

  bool Rco::operator == (const Rco& rco) const {
    rdlock ();
    try {
      bool result = ((d_rid == rco.d_rid) && (p_rco == rco.p_rco));
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // compare an rco with another one

  bool Rco::operator != (const Rco& rco) const {
    rdlock ();
    try {
      bool result = !((d_rid == rco.d_rid) && (p_rco == rco.p_rco));
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the registration id

  String Rco::getrid (void) const {
    rdlock ();
    try {
      String result = d_rid;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the registrered object

  Object* Rco::getobj (void) const {
    rdlock ();
    try {
      Object* result = p_rco;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 2;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_GETRID = zone.intern ("get-rid");
  static const long QUARK_GETOBJ = zone.intern ("get-object");

  // create a new object in a generic way

  Object* Rco::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // create a default registered object
    if (argc == 0) return new Rco;
    // check for 1 argument
    if (argc == 1) {
      Object* rco = argv->get (0);
      return new Rco (rco);
    }
    // check for 2 arguments
    if (argc == 2) {
      String  rid = argv->getstring (0);
      Object* rco = argv->get (1);
      return new Rco (rid, rco);
    }
    throw Exception ("argument-error",
                     "too many argument with rco constructor");
  }

  // return true if the given quark is defined

  bool Rco::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Object::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }
  
  // operate this object with another object

  Object* Rco::oper (t_oper type, Object* object) {
    Rco* rco = dynamic_cast <Rco*> (object);
    switch (type) {
    case Object::EQL:
      if (rco != nilp) return new Boolean (*this == *rco);
      break;
    case Object::NEQ:
      if (rco != nilp) return new Boolean (*this != *rco);
      break;
    default:
      throw Exception ("operator-error", "unsupported rco operator");
    }
    throw Exception ("type-error", "invalid operand with rco",
                     Object::repr (object));
  }
  
  // apply this object with a set of arguments and a quark
  
  Object* Rco::apply (Runnable* robj, Nameset* nset, const long quark,
		      Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // check for 0 arguments
    if (argc == 0) {
      if (quark == QUARK_GETRID) return new String (getrid ());
      if (quark == QUARK_GETOBJ) {
	rdlock ();
	try {
	  Object* result = getobj ();
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
    }
    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
