// ---------------------------------------------------------------------------
// - Rsi.cpp                                                                 -
// - afnix:mth module - real sparse interface implementation                 -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2011 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Rsi.hpp"
#include "Real.hpp"
#include "Vector.hpp"
#include "Integer.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"
 
namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default sparse object

  Rsi::Rsi (void) {
    d_size = 0;
  }

  // create a sparse object by size

  Rsi::Rsi (const t_long size) {
    // check the size
    if (size < 0) {
      throw Exception ("size-error", "invalid sparse object size");
    }
    d_size = size;
  }

  // get the sparse object size

  t_long Rsi::getsize (void) const {
    rdlock ();
    try {
      t_long result = d_size;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 5;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the rsi supported quarks
  static const long QUARK_SET     = zone.intern ("set");
  static const long QUARK_LENGTH  = zone.intern ("length");
  static const long QUARK_GETIDX  = zone.intern ("get-index");
  static const long QUARK_GETVAL  = zone.intern ("get-value");
  static const long QUARK_GETSIZE = zone.intern ("get-size");

  // return true if the given quark is defined

  bool Rsi::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true){
      unlock ();
      return true;
    }
    bool result = hflg ? Object::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }
  
  // apply this object with a set of arguments and a quark
  
  Object* Rsi::apply (Runnable* robj, Nameset* nset, const long quark,
                      Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_LENGTH)  return new Integer (length  ());
      if (quark == QUARK_GETSIZE) return new Integer (getsize ());
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_GETIDX) {
        t_long pos = argv->getlong (0);
        return new Integer (getidx (pos));
      }
      if (quark == QUARK_GETVAL) {
        t_long pos = argv->getlong (0);
        return new Real (getval (pos));
      }
    }
    // dispatch 2 arguments
    if (argc == 2) {
      if (quark == QUARK_SET) {
        t_long sidx = argv->getlong (0);
        t_real sval = argv->getreal (1);
        return new Integer (set (sidx, sval));
      }
    }
    // call the object
    return Object::apply (robj, nset, quark, argv);
  }
}
