// ---------------------------------------------------------------------------
// - Session.hpp                                                             -
// - afnix:nwg module - http session class definition                        -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2011 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_SESSION_HPP
#define  AFNIX_SESSION_HPP

#ifndef  AFNIX_NAMEABLE_HPP
#include "Nameable.hpp"
#endif

namespace afnix {

  /// The Session class is a simple class that defines a session to be 
  /// associated with an http transaction. The session object is designed 
  /// to be persistent so that its data information can be retreived at
  /// any time. A session object has also the particularity to have a 
  /// limited lifetime. A session object is created by name with an
  /// identifier. If a path is given, such path will be used as the session
  /// file name.
  /// @author amaury darsch

  class Session : public Nameable, public Serial {
  private:
    /// the session name
    String d_name;
    /// the session user
    String d_user;
    /// the session id
    String d_shid;
    /// the session path
    String d_path;
    /// the session creation time
    t_long d_ctim;
    /// the session modification time
    t_long d_mtim;
    /// the session expiration time
    t_long d_etim;

  public:
    /// create an empty session
    Session (void);

    /// create a session by name
    /// @param name the session name
    Session (const String& name);

    /// create a session by name and path
    /// @param name the session name
    /// @param user the session user
    Session (const String& name, const String& user);

    /// create a session by name and path
    /// @param name the session name
    /// @param user the session user
    /// @param path the session path
    Session (const String& name, const String& user, const String& path);

    /// @return the class name
    String repr (void) const;

    /// @return the session serial id
    t_byte serialid (void) const;

    /// serialize this session
    /// @param os the output stream
    void wrstream (OutputStream& os) const;

    /// deserialize this session
    /// @param is the input stream
    void rdstream (InputStream& os);

    /// @return the session name
    String getname (void) const;

    /// set the session hash id
    /// @param shid the session hash id
    void setshid (const String& shid);

    /// @return the session hash id
    String getshid (void) const;

    /// set the session user
    /// @param user the session user
    void setuser (const String& user);

    /// @return the session user
    String getuser (void) const;

    /// set the session path
    /// @param path the session path
    void setpath (const String& path);

    /// @return the session path
    String getpath (void) const;

    /// @return the session creation time
    t_long getctim (void) const;

    /// @return the session modification time
    t_long getmtim (void) const;

    /// @return true if a session has expired
    bool isetim (void) const;

    /// set the session expiration time
    /// @param time the session allowed time
    void setetim (const t_long time);

    /// @return the session expiration time
    t_long getetim (void) const;

    /// mark the session as expired
    void mrketim (void);

    /// set the session maximum age
    /// @param mage the session age
    void setmage (const t_long mage);

    /// @return the session valid time
    t_long getvldt (void) const;

  private:
    // make the copy constructor private
    Session (const Session&);
    // make the assignment operator private
    Session& operator = (const Session&);

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);
  };
}

#endif
