/*******************************************************************************
 *  PROJECT: Agave
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#include "gcs-scheme.h"
#include "gcs-color.h"

namespace gcs
{

    // a helper function for sorting color schemes in order of ascending
    // luminance value
    static bool compare_luminance(ColorPtr c1, ColorPtr c2)
    {
        return *c1 < *c2;
    }


    Scheme::Scheme(ColorPtr c, tSchemeType t)
        : m_color(c), m_schemeType(t)
    {
        init();
    }


    void Scheme::init()
    {
        // clear the list of colors first.
        clear();
        switch (m_schemeType)
        {
            // for a description of the scheme types, see
            // http://www.color-wheel-pro.com/color-schemes.html
            case SCHEME_COMPLEMENTS:    // only two colors
                {
                    ColorPtr complement;
                    complement = Color::create(*m_color);
                    complement->set_hue(m_color->get_hue() + (maxHueValue / 2));

                    push_back(m_color);
                    push_back(complement);
                }
                break;

            case SCHEME_SPLIT_COMPLEMENTS:
                // three colors
                {
                    // angle to offset from 180 degrees
                    // tweak this number to get it right
                    const int offset = maxHueValue / 15;
                    ColorPtr scomplement1, scomplement2;
                    // create copies and modify them
                    scomplement1 = Color::create(*m_color);
                    scomplement1->set_hue(m_color->get_hue() + (maxHueValue / 2) - offset);
                    scomplement2 = Color::create(*m_color);
                    scomplement2->set_hue(m_color->get_hue() + (maxHueValue / 2) + offset);

                    push_back(m_color);
                    push_back(scomplement1);
                    push_back(scomplement2);
                }
                break;

            case SCHEME_TETRADS:
                // four colors (two complementary colors)
                {
                    // tetrads are technically 4 evenly-spaced colors.
                    // previously I had them non-evenly spaced
                    const int offset = maxHueValue / 4;
                    ColorPtr tetrad1, tetrad2, tetrad3;
                    tetrad1 = Color::create(*m_color);
                    tetrad1->set_hue(m_color->get_hue() + offset);
                    tetrad2 = Color::create(*m_color);
                    tetrad2->set_hue(m_color->get_hue() + maxHueValue / 2);
                    tetrad3 = Color::create(*m_color);
                    tetrad3->set_hue(m_color->get_hue() + maxHueValue / 2 + offset);

                    push_back(m_color);
                    push_back(tetrad1);
                    push_back(tetrad2);
                    push_back(tetrad3);
                }
                break;

            case SCHEME_ANALOGOUS:
                // three colors next to eachother on a color wheel
                {
                    // tweak this number to get it right
                    const int offset = maxHueValue / 12;
                    ColorPtr analogous1, analogous2;
                    analogous1 = Color::create(*m_color);
                    analogous1->set_hue(m_color->get_hue() - offset);
                    analogous2 = Color::create(*m_color);
                    analogous2->set_hue(m_color->get_hue() + offset);

                    push_back(analogous1);
                    push_back(m_color);
                    push_back(analogous2);
                }
                break;

            case SCHEME_MONOCHROMATIC:
                // several colors of a similar hue
                {
                    ColorPtr mono1, mono2;
                    mono1 = Color::create(*m_color);
                    mono2 = Color::create(*m_color);

                    if (m_color->get_saturation() < (maxSvValue / 10))
                    {
                        mono1->set_saturation((m_color->get_saturation() + maxSvValue / 3) % maxSvValue);
                        mono2->set_saturation((m_color->get_saturation() + 2* maxSvValue / 3) % maxSvValue);
                    }
                    else
                    {
                        mono1->set_value((m_color->get_value() + maxSvValue / 3) % maxSvValue);
                        mono2->set_value((m_color->get_value() + 2 * maxSvValue / 3) % maxSvValue);
                    }

                    push_back(m_color);
                    push_back(mono1);
                    push_back(mono2);

                    // sort it so that the schemes look similar no matter how
                    // bright the selected color is.  This should sort the
                    // scheme left to right from 'dark' to 'light'
                    std::sort(begin(), end(), compare_luminance);
                }
                break;

            case SCHEME_TRIADS:
            default:
                // three colors
                {
                    const int offset = maxHueValue / 3;
                    ColorPtr triad1, triad2;
                    triad1 = Color::create(m_color->gdk());
                    triad1->set_hue(m_color->get_hue() + offset);
                    triad2 = Color::create(m_color->gdk());
                    triad2->set_hue(m_color->get_hue() - offset);

                    push_back(m_color);
                    push_back(triad1);
                    push_back(triad2);
                }
                break;
        }
    }

    void Scheme::set_color(ColorPtr c)
    {
        m_color = c;
        init();
    }


    void Scheme::set_scheme_type(tSchemeType t)
    {
        m_schemeType = t;
        init();
    }


    bool Scheme::operator==(const Scheme& other)
    {
        return (*m_color == *(other.m_color)
                && m_schemeType == other.m_schemeType);
    }

} // namespace gcs
