#! /usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Copyright 2021, Nils Hilbricht, Germany ( https://www.hilbricht.net )

This file is part of the Laborejo Software Suite ( https://www.laborejo.org )

This application is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import logging; logger = logging.getLogger(__name__); logger.info("import")

#Standard Library

#Engine
import engine.api as api


#Third Party
from PyQt5 import QtCore, QtGui, QtWidgets


class DescriptionController(object):
    """Not a subclass. Controls a TextEditWidget to hold the session-description from nsm-data.
    Can be used on multiple text widgets. One controller for one textwidget."""

    def __init__(self, mainWindow, parentGroupBox:QtWidgets.QGroupBox, plainTextWidget:QtWidgets.QPlainTextEdit):
        self.mainWindow = mainWindow
        self.description = plainTextWidget
        self.parentGroupBox = parentGroupBox

        self.placeHolderDescription = QtCore.QCoreApplication.translate("LoadedSessionDescription", "Double click to add the client nsm-data to write here.\nUse it for notes, TODO, references etc…")
        #We do NOT use the Qt placeholder text because that shows even when editing is possible. Our visual feedback is changing from placeholder to empty.
        self.description.setEnabled(False)
        self._reactCallback_dataClientDescriptionChanged(None) #set up description     
        self.parentGroupBox.mouseDoubleClickEvent = self._doubleClickToResumeOrAddNsmData        
        self.description.textChanged.connect(self.sendDescriptionToApi) #TODO: this is every keystroke, but it is impossible to solve a multi-GUI network system where a save signal comes from outside.. we need update every char. Maybe a genius in the future will solve this.        
        #self.description.focusOutEvent = self._descriptionFocusOut
        
        api.callbacks.dataClientDescriptionChanged.append(self._reactCallback_dataClientDescriptionChanged) #Session description        

    def sendDescriptionToApi(self):
        """We cannot send every keystroke over the OSC-network. Therefore we wait:
        This is called by several events that "feel" like editing is done now:
        Focus out, Ctlr+S, Alt+S."""
        api.setDescription(self.description.toPlainText()) #this is not save yet. Just forward to data client.

    #def _descriptionFocusOut(self, event):
    #    self.sendDescriptionToApi()
    #    QtWidgets.QPlainTextEdit.focusOutEvent(self.description, event)

    def _reactCallback_dataClientDescriptionChanged(self, desc:str):
        """Put the session description into our text field.
        We send each change, so we receive this signal each detail change.

        The cursor changes in between so we force the position.
        """        
        self.description.blockSignals(True)        
        if not desc is None: #may be None for closing session
            oldPos = self.description.textCursor().position()            
            self.description.setPlainText(desc) #plain textedit
            self.description.setEnabled(True)            
            c = self.description.textCursor()
            c.setPosition(oldPos)
            self.description.setTextCursor(c)
        else:
            self.description.setEnabled(False)
            self.description.setPlainText(self.placeHolderDescription)                    
        self.description.blockSignals(False)

    def _doubleClickToResumeOrAddNsmData(self, event):
        """Intended for doubleClickEvent, so we get an event.
        Do nothing when nsm-data is present. Add it when it was never there.
        Resume it if stopped in the session.

        When QPlainTextEdit is disabled it will forward doubleClick to the parent widget,
        which is this function. If enabled the groupBox description and frame will be clickable, we
        do nothing in this case.
        """
        if self.description.isEnabled():
            pass
        else:           
            d = api.executableInSession("nsm-data")
            if d:
                api.clientResume(d["clientId"])
            else:            
                api.clientAdd("nsm-data")
        QtWidgets.QGroupBox.mouseDoubleClickEvent(self.parentGroupBox, event)
