/** \file src/main.cc */
/*
 * This file is part of assoGiate,
 * an editor of the file types database for GNOME.
 *
 * Copyright (C) 2007 Kevin Daughtridge <kevin@kdau.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "private.hh"
#include "main.hh"

#include <sys/stat.h>
#include <unistd.h>
#include <clocale>
#include <iostream>
#include <locale>
#include <glib/goption.h>
#include <glib/gutils.h>
#include <glibmm/optionentry.h>
#include <gtk/gtkaboutdialog.h>
#include <gtk/gtkversion.h>
#include <gtk/gtkwindow.h>
#include <gtkmm/aboutdialog.h>
#include <gtkmm/filechooserdialog.h>
#include <gtkmm/main.h>
#include <gtkmm/menu.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/radioaction.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/stock.h>
#include <gtkmm/toolbar.h>

/******************************************************************************/
/* Globals                                                                    */
/******************************************************************************/

int
main(int argc, char** argv)
{	return assoGiate::main(argc, argv); }

/******************************************************************************/
namespace assoGiate {
/******************************************************************************/

int
main(int& argc, char**& argv)
{
	bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);

	Glib::init();
	Glib::set_prgname("assogiate");
	Glib::set_application_name(_("File Types Editor"));

	try {
		MainOptionGroup opts;
		Glib::OptionContext context;
		context.set_main_group(opts);
#if GLIB_CHECK_VERSION(2, 12, 0)
		g_option_context_set_summary(context.gobj(), _("Modify the detection "
			"and display of file types"));
#endif

		Gtk::Main kit(argc, argv, context);
		std::locale::global(std::locale(std::setlocale(LC_ALL, NULL)));
		gtk_window_set_default_icon_name("assogiate"); // unwrapped
		Gtk::AboutDialog::set_email_hook(sigc::ptr_fun(&activate_email));
		Gtk::AboutDialog::set_url_hook(sigc::ptr_fun(&activate_url));

		MimeDatabaseLoadError::signal_unhandled().connect
			(sigc::ptr_fun(&die_on_database_load_error));

		MainWindow window(opts.m_user);
		window.present();

		MimeDatabaseUpdateError::signal_unhandled().connect(sigc::bind
			(sigc::ptr_fun(&show_database_update_warning), &window));
	
		Gtk::Main::run(window);
		return EXIT_SUCCESS;
	}
	catch (Glib::OptionError &e) {
		std::cerr << e.what() << std::endl;
		std::cerr << compose::ucompose(_("Run '%1 --help' to see a full list "
			"of available command line options."), argv[0]) << std::endl;
		return EXIT_FAILURE;
	}
#ifndef ENABLE_DEBUG
	catch (...) {
		std::cerr << _("An unknown error occurred.") << std::endl;
		return EXIT_FAILURE;
	}
#endif /* !ENABLE_DEBUG */
}

/******************************************************************************/
/* class MainOptionGroup                                                      */
/******************************************************************************/

MainOptionGroup::MainOptionGroup()
:	Glib::OptionGroup(ustring(), ustring()), m_user(false)
{
	set_translation_domain(GETTEXT_PACKAGE);

	Glib::OptionEntry entry_user;
	entry_user.set_long_name("user");
	entry_user.set_description(N_("Display the user database even if run as "
		"the root user. By default, the system database is displayed for the "
		"root user."));
	add_entry(entry_user, m_user);
}

/******************************************************************************/
/* class MainWindow                                                           */
/******************************************************************************/

MainWindow::MainWindow(bool want_user_db)
:	m_user_db(false), m_system_db_action(), m_database(), m_target(NO_LOCATION),
	m_ui(Gtk::UIManager::create()), m_actions(Gtk::ActionGroup::create()),
	m_categories_sel(), m_types(), m_types_view(),
	m_types_cols(TypesColumns::get()), m_last_selected(), m_find_box(false, 6),
	m_find_entry(), m_type_dialogs()
{
	bool can_system_db = (geteuid() == 0);
	m_user_db = want_user_db || !can_system_db;

	add_accel_group(m_ui->get_accel_group());

	Gtk::VBox *vbox = new Gtk::VBox(false, 0);
	add(*Gtk::manage(vbox));
	
	m_actions->add(Gtk::Action::create("DatabaseMenu", _("_Database")));

	Gtk::RadioButtonGroup db_group;

	m_actions->add(Gtk::RadioAction::create(db_group, "DatabaseUser",
		_("_User Database")),
		sigc::mem_fun(*this, &MainWindow::on_database_toggled));

	m_system_db_action = Gtk::RadioAction::create(db_group, "DatabaseSystem",
		_("_System Database"));
	m_system_db_action->set_sensitive(can_system_db);
	m_system_db_action->set_active(!m_user_db);
	m_actions->add(m_system_db_action,
		sigc::mem_fun(*this, &MainWindow::on_database_toggled));

	m_actions->add(Gtk::Action::create("DatabaseImport", _("_Import Types...")),
		sigc::mem_fun(*this, &MainWindow::on_import_types));
	m_actions->add(Gtk::Action::create("DatabaseExport", _("_Export Types...")),
		sigc::mem_fun(*this, &MainWindow::on_export_types));
	m_actions->add(Gtk::Action::create("DatabaseClear", Gtk::Stock::CLEAR,
		_("_Clear Modified Types")),
		sigc::mem_fun(*this, &MainWindow::on_clear_types));
	m_actions->add(Gtk::Action::create("DatabaseQuit", Gtk::Stock::QUIT),
		sigc::mem_fun(*this, &MainWindow::hide));

	m_actions->add(Gtk::Action::create("TypeMenu", _("_Type")));
	RefPtr<Gtk::Action> act = Gtk::Action::create("TypeNew", Gtk::Stock::NEW);
	act->property_is_important() = true;
	m_actions->add(act, sigc::mem_fun(*this, &MainWindow::on_new_type));
	act = Gtk::Action::create("TypeEdit", Gtk::Stock::EDIT);
	act->property_is_important() = true;
	m_actions->add(act, Gtk::AccelKey(GDK_Return, Gdk::ModifierType(0)),
		sigc::mem_fun(*this, &MainWindow::on_edit_type));
	act = Gtk::Action::create("TypeRevert", Gtk::Stock::REVERT_TO_SAVED);
	m_actions->add(act, Gtk::AccelKey(GDK_r, Gdk::CONTROL_MASK),
		sigc::mem_fun(*this, &MainWindow::on_revert_type));
	act = Gtk::Action::create("TypeDelete", Gtk::Stock::DELETE);
	m_actions->add(act, Gtk::AccelKey(GDK_Delete, Gdk::ModifierType(0)),
		sigc::mem_fun(*this, &MainWindow::on_delete_type));
	act = Gtk::Action::create("TypeFind", Gtk::Stock::FIND);
	m_actions->add(act, sigc::mem_fun(*this, &MainWindow::on_find_type));

	m_actions->add(Gtk::Action::create("HelpMenu", _("_Help")));
	m_actions->add(Gtk::Action::create("HelpContents", Gtk::Stock::HELP,
		_("_Contents")), Gtk::AccelKey(GDK_F1, Gdk::ModifierType(0)),
		sigc::mem_fun(*this, &MainWindow::on_help_contents));
	m_actions->add(Gtk::Action::create("HelpAbout", Gtk::Stock::ABOUT),
		sigc::mem_fun(*this, &MainWindow::on_help_about));

	m_ui->insert_action_group(m_actions);
	m_ui->add_ui_from_string(
		"<ui>"
			"<menubar name='MenuBar'>"
				"<menu action='DatabaseMenu'>"
					"<menuitem action='DatabaseUser' />"
					"<menuitem action='DatabaseSystem' />"
					"<separator />"
					"<menuitem action='DatabaseImport' />"
					"<menuitem action='DatabaseExport' />"
					"<separator />"
					"<menuitem action='DatabaseClear' />"
					"<separator />"
					"<menuitem action='DatabaseQuit' />"
				"</menu>"
				"<menu action='TypeMenu'>"
					"<menuitem action='TypeNew' />"
					"<menuitem action='TypeEdit' />"
					"<menuitem action='TypeRevert' />"
					"<menuitem action='TypeDelete' />"
					"<separator />"
					"<menuitem action='TypeFind' />"
				"</menu>"
				"<menu action='HelpMenu'>"
					"<menuitem action='HelpContents' />"
					"<menuitem action='HelpAbout' />"
				"</menu>"
			"</menubar>"
			"<toolbar name='Toolbar'>"
				"<toolitem action='TypeNew' />"
				"<toolitem action='TypeEdit' />"
				"<toolitem action='TypeRevert' />"
				"<toolitem action='TypeDelete' />"
				"<separator />"
				"<toolitem action='TypeFind' />"
			"</toolbar>"
			"<popup name='TypePopup'>"
				"<menuitem action='TypeEdit' />"
				"<menuitem action='TypeRevert' />"
				"<menuitem action='TypeDelete' />"
			"</popup>"
		"</ui>"
	);

	Gtk::Widget *widget = m_ui->get_widget("/MenuBar");
	if (widget) vbox->pack_start(*Gtk::manage(widget), false, false);

	widget = m_ui->get_widget("/Toolbar");
	if (widget) vbox->pack_start(*Gtk::manage(widget), false, false);

	Gtk::HBox *hbox = new Gtk::HBox(false, 12);
	hbox->set_border_width(12);
	vbox->pack_start(*Gtk::manage(hbox), true, true);
	
	Gtk::ScrolledWindow *categories_sw = new Gtk::ScrolledWindow();
	categories_sw->set_policy(Gtk::POLICY_NEVER, Gtk::POLICY_NEVER);
	categories_sw->set_shadow_type(Gtk::SHADOW_IN);
	hbox->pack_start(*Gtk::manage(categories_sw), false, false);

	Gtk::TreeView *categories_tv =
		new Gtk::TreeView(CategoriesStore::get_filter());
	categories_tv->set_headers_visible(false);
	categories_tv->set_enable_search(false);
	categories_tv->set_search_column(-1);
	categories_tv->set_row_separator_func
		(sigc::ptr_fun(&CategoriesStore::row_separator_func));
	categories_sw->add(*Gtk::manage(categories_tv));

	m_categories_sel = categories_tv->get_selection();
	m_categories_sel->set_mode(Gtk::SELECTION_BROWSE);
	m_categories_sel->select(CategoriesStore::get_filter_item
		(CategoriesColumns::ALL));
	m_categories_sel->signal_changed().connect
		(sigc::mem_fun(*this, &MainWindow::on_category_changed));

	Gtk::TreeViewColumn *categories_vc = new Gtk::TreeViewColumn();
	categories_vc->pack_start(CategoriesColumns::get().icon, false);
	categories_vc->pack_start(CategoriesColumns::get().name);
	categories_tv->append_column(*Gtk::manage(categories_vc));

	vbox = new Gtk::VBox(false, 12);
	hbox->pack_start(*Gtk::manage(vbox), true, true);
	
	Gtk::ScrolledWindow *types_sw = new Gtk::ScrolledWindow();
	types_sw->set_policy(Gtk::POLICY_NEVER, Gtk::POLICY_AUTOMATIC);
	types_sw->set_shadow_type(Gtk::SHADOW_IN);
	vbox->pack_start(*Gtk::manage(types_sw), true, true);

	m_find_box.set_no_show_all(true);
	vbox->pack_start(m_find_box, false, false);
	
	Gtk::Label *label = new Gtk::Label(_("_Search for:"), true);
	label->set_mnemonic_widget(m_find_entry);
	m_find_box.pack_start(*Gtk::manage(label), false, false);
	
	m_find_entry.signal_changed().connect
		(sigc::mem_fun(*this, &MainWindow::on_find_changed));
	m_find_box.pack_start(m_find_entry, true, true);
	
	Gtk::Button *cancel = new Gtk::Button();
	cancel->set_image(*Gtk::manage(new Gtk::Image(Gtk::Stock::CLOSE,
		Gtk::ICON_SIZE_MENU)));
	cancel->signal_clicked().connect
		(sigc::mem_fun(*this, &MainWindow::on_cancel_find));
	cancel->add_accelerator("clicked", get_accel_group(), GDK_Escape,
		Gdk::ModifierType(0), Gtk::AccelFlags(0));
	m_find_box.pack_start(*Gtk::manage(cancel), false, false);

	RefPtr<Gtk::TreeSelection> sel = m_types_view.get_selection();
	m_types_view.get_selection()->signal_changed().connect(sigc::mem_fun
		(*this, &MainWindow::on_selection_changed));
	m_types_view.signal_row_activated().connect(sigc::hide(sigc::hide(
		sigc::mem_fun(*this, &MainWindow::on_edit_type))));
	m_types_view.signal_right_click().connect
		(sigc::mem_fun(*this, &MainWindow::on_right_click));
	types_sw->add(m_types_view);

	switch_database(m_user_db);

	m_find_box.show_all_children();
	show_all_children();
	m_types_view.grab_focus();
}

MainWindow::~MainWindow()
{
	FOREACH(std::list<TypeDialog*>, m_type_dialogs, i)
		delete *i;
}

void
MainWindow::on_database_toggled()
{	if (m_database) switch_database(!m_system_db_action->get_active()); }

void
MainWindow::switch_database(bool user_db)
{
	if (m_database && m_user_db == user_db) return;

	FOREACH(std::list<TypeDialog*>, m_type_dialogs, i)
		delete *i;
	m_type_dialogs.clear();

	m_user_db = user_db;
	if (!m_user_db) umask(022);

	set_title(compose::ucompose(_("File Types Editor - %1"),
		m_user_db ? _("User Database") : _("System Database")));

	try {
		m_database = MimeDatabase::load(m_user_db);
	} catch (MimeDatabaseLoadError& e) {
		die_on_database_load_error(e);
	}
	m_database->signal_changed().connect
		(sigc::mem_fun(*this, &MainWindow::on_database_changed));

	m_target = m_database->get_target();
	m_types = TypesStore::create(m_database);
	m_types_view.set_model(m_types);
	
	on_database_changed();
	on_category_changed();
}

void
MainWindow::on_database_changed()
{
	m_actions->get_action("DatabaseExport")->set_sensitive
		(m_database->get_Override_pkg().has_contents());
	m_actions->get_action("DatabaseClear")->set_sensitive
		(m_database->get_Override_pkg().has_contents());

	std::list<TypeDialog*> gone;
	FOREACH(std::list<TypeDialog*>, m_type_dialogs, i) {
		ustring full_name = (*i)->get_full_name();
		if (full_name.empty()) continue; /* AddTypeDialog */

		MimeTypeMap::iterator j = m_database->get_types().find(full_name);
		if (j != m_database->get_types().end())
			(*i)->set_type(*j->second.first);
		else
			gone.push_back(*i);
	}
	
	FOREACH(std::list<TypeDialog*>, gone, i)
		on_hide_type_dialog(*i);
}

void
MainWindow::on_category_changed()
{
	store_selection();

	switch (m_categories_sel->get_selected()->
		get_value(CategoriesColumns::get().filter)) {
	case CategoriesColumns::SINGLE:
		m_types->show_category(m_categories_sel->get_selected()->
			get_value(CategoriesColumns::get().id)); break;
	case CategoriesColumns::ALL: m_types->show_all(); break;
	case CategoriesColumns::OVERRIDE: m_types->show_override(); break;
	case CategoriesColumns::SEARCH:
		m_find_box.show();
		on_find_changed();
		m_find_entry.grab_focus();
		return;
	default: /* ??? */
		return;
	}
	
	m_find_box.hide();
	restore_selection();
}

void
MainWindow::on_selection_changed()
{
	Gtk::TreeIter iter = m_types_view.get_selection()->get_selected();
	if (iter) {
		MimeType *type = iter->get_value(m_types_cols.obj);
		bool alias = iter->get_value(m_types_cols.alias),
			fixed = ((type->get_locations() & ~m_target) != NO_LOCATION),
			target = ((type->get_locations() & m_target) != NO_LOCATION);
		m_actions->get_action("TypeEdit")->set_sensitive(true);
		m_actions->get_action("TypeRevert")->set_sensitive
			(!alias && fixed && target);
		m_actions->get_action("TypeDelete")->set_sensitive
			(!alias && target && !fixed);
	} else {
		m_actions->get_action("TypeEdit")->set_sensitive(false);
		m_actions->get_action("TypeRevert")->set_sensitive(false);
		m_actions->get_action("TypeDelete")->set_sensitive(false);
	}
}

void
MainWindow::on_right_click(guint32 time)
{
	if cast(m_ui->get_widget("/TypePopup"), Gtk::Menu, menu)
		menu->popup(3, time);
}

void
MainWindow::store_selection()
{
	Gtk::TreeIter old_iter = m_types_view.get_selection()->get_selected();
	if (old_iter)
		m_last_selected = old_iter->get_value(m_types_cols.obj)
			->get_full_name();
}

void
MainWindow::restore_selection()
{
	if (!m_last_selected.empty()) {
		FOREACH_BASE(Gtk::TreeIter, m_types->children(), i)
			if (i->get_value(m_types_cols.obj)->get_full_name() ==
				m_last_selected) {
				m_types_view.get_selection()->select(i);
				m_types_view.scroll_to_row(m_types->get_path(i), 0.5);
				return;
			}
		
		m_last_selected.clear();
	}
}

void
MainWindow::on_import_types()
{
	Gtk::FileChooserDialog dlg(*this, _("Import File Types"),
		Gtk::FILE_CHOOSER_ACTION_OPEN);
	dlg.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	dlg.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_OK);
	dlg.set_default_response(Gtk::RESPONSE_OK);

	Gtk::FileFilter xml, all;
	xml.set_name(_("All XML files"));
	xml.add_pattern("*.xml");
	dlg.add_filter(xml);
	all.set_name(_("All files"));
	all.add_pattern("*");
	dlg.add_filter(all);

	if (dlg.run() != Gtk::RESPONSE_OK) return;
	dlg.hide();

	try {
		m_database->get_Override_pkg().import_types(dlg.get_filename());
	} catch (MimeDatabaseUpdateError& e) {
		show_database_update_warning(e, this);
	} catch (MimeDatabaseLoadError& e) {
		die_on_database_load_error(e);
	} catch (MimeDatabaseWriteError& e) {
		Gtk::MessageDialog err(*this, _("Could not import file types"), false,
			Gtk::MESSAGE_ERROR, Gtk::BUTTONS_CLOSE, true);
		err.set_secondary_text(compose::ucompose(_("The file types database "
			"could not be modified. It may be helpful to retry.\n\n(%1)"),
			e.what()));
		err.run();
	} catch (xmlpp::exception& e) {
		Gtk::MessageDialog err(*this, _("Could not import file types"), false,
			Gtk::MESSAGE_ERROR, Gtk::BUTTONS_CLOSE, true);
		err.set_secondary_text(compose::ucompose(_("File types could not be "
			"imported from \"%1\".\n\n(%2)"),
			Glib::filename_display_name(dlg.get_filename()), e.what()));
		err.run();
	}
}

void
MainWindow::on_export_types()
{
	Gtk::FileChooserDialog dlg(*this, _("Export File Types"),
		Gtk::FILE_CHOOSER_ACTION_SAVE);
	dlg.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	dlg.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_OK);
	dlg.set_default_response(Gtk::RESPONSE_OK);

	Gtk::FileFilter xml, all;
	xml.set_name(_("All XML files"));
	xml.add_pattern("*.xml");
	dlg.add_filter(xml);
	all.set_name(_("All files"));
	all.add_pattern("*");
	dlg.add_filter(all);

	if (dlg.run() != Gtk::RESPONSE_OK) return;
	dlg.hide();

	try {
		m_database->get_Override_pkg().export_types(dlg.get_filename());
	} catch (xmlpp::exception& e) {
		Gtk::MessageDialog err(*this, _("Could not export file types"),
			false, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_CLOSE, true);
		err.set_secondary_text(compose::ucompose(_("Modified types could not "
			"be exported to \"%1\".\n\n(%2)"),
			Glib::filename_display_name(dlg.get_filename()), e.what()));
		err.run();
	}
}

void
MainWindow::on_clear_types()
{
	m_categories_sel->select(CategoriesStore::get_filter_item
		(CategoriesColumns::OVERRIDE));
	
	{
		Gtk::MessageDialog ask(*this, _("Delete all modified types?"), false,
			Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_CANCEL, true);
		ask.add_button(Gtk::Stock::CLEAR, Gtk::RESPONSE_OK);
		ask.set_secondary_text(_("All file types that do not exist in the "
			"standard database, and all modifications to standard types, will "
			"be permanently lost."));
		if (ask.run() != Gtk::RESPONSE_OK) return;
	}

	try {
		m_database->get_Override_pkg().clear_types();
	} catch (MimeDatabaseUpdateError& e) {
		show_database_update_warning(e, this);
	} catch (MimeDatabaseLoadError& e) {
		die_on_database_load_error(e);
	} catch (MimeDatabaseWriteError& e) {
		Gtk::MessageDialog err(*this, _("Could not clear database"),
			false, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_CLOSE, true);
		err.set_secondary_text(compose::ucompose(_("The file types "
			"database could not be modified. It may be helpful to "
			"retry.\n\n(%1)"), e.what()));
		err.run();
	}
}

void
MainWindow::on_new_type()
{	type_dialog_common(*new NewTypeDialog(m_database)); }

void
MainWindow::on_edit_type()
{
	Gtk::TreeIter iter = m_types_view.get_selection()->get_selected();
	if (iter) {
		MimeType *type = iter->get_value(m_types_cols.obj);
		ustring full_name = type->get_full_name();
		FOREACH(std::list<TypeDialog*>, m_type_dialogs, i)
			if ((*i)->get_full_name() == full_name) {
				(*i)->present();
				return;
			}
		type_dialog_common(*new EditTypeDialog(m_database, *type, m_target));
	}
}

void
MainWindow::type_dialog_common(TypeDialog& dialog)
{
	dialog.signal_hide().connect(sigc::bind(sigc::mem_fun(*this,
		&MainWindow::on_hide_type_dialog), &dialog));
	dialog.set_transient_for(*this);
	m_type_dialogs.push_back(&dialog);
	dialog.present();
}

void
MainWindow::on_hide_type_dialog(TypeDialog* dialog)
{
	delete dialog;
	m_type_dialogs.remove(dialog);
}

void
MainWindow::on_revert_type()
{
	remove_type_common(_("Revert modifications to %1?"), _("All modifications "
		"to this file type will be permanently lost. The information in the "
		"standard database will be restored."), Gtk::Stock::REVERT_TO_SAVED);
}

void
MainWindow::on_delete_type()
{
	remove_type_common(_("Delete modified type %1?"), _("This file type does "
		"not exist in the standard database. All information about it will be "
		"permanently lost."), Gtk::Stock::DELETE);
}

void
MainWindow::remove_type_common(const ustring& primary,
	const ustring& secondary, const Gtk::StockID& stock)
{
	Gtk::TreeIter iter = m_types_view.get_selection()->get_selected();
	if (iter) {
		MimeType *type = iter->get_value(m_types_cols.obj);
		if ((type->get_locations() & m_target) != NO_LOCATION) {
			ustring full_name = type->get_full_name();
			int response = Gtk::RESPONSE_CANCEL;
			{
				Gtk::MessageDialog ask(*this,
					compose::ucompose(primary, full_name), false,
					Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_CANCEL, true);
				ask.add_button(stock, Gtk::RESPONSE_OK);
				ask.set_secondary_text(secondary);
				response = ask.run();
			}
			if (response == Gtk::RESPONSE_OK) {
				try {
					m_database->get_Override_pkg().remove_type(full_name);
				} catch (MimeDatabaseUpdateError& e) {
					show_database_update_warning(e, this);
				} catch (MimeDatabaseLoadError& e) {
					die_on_database_load_error(e);
				} catch (MimeDatabaseWriteError& e) {
					Gtk::MessageDialog err(*this, compose::ucompose(_("Could "
						"not revert or delete file type"), full_name), false,
						Gtk::MESSAGE_ERROR, Gtk::BUTTONS_CLOSE, true);
					err.set_secondary_text(compose::ucompose(_("The file types "
						"database could not be modified. It may be helpful to "
						"retry.\n\n(%1)"), e.what()));
					err.run();
				}
			}
		}
	}
}

void
MainWindow::on_find_type()
{
	m_categories_sel->select(CategoriesStore::get_filter_item
		(CategoriesColumns::SEARCH));
	m_find_entry.grab_focus();
}

void
MainWindow::on_find_changed()
{
	store_selection();
	m_types->show_results(m_find_entry.get_text());
	restore_selection();
}

void
MainWindow::on_cancel_find()
{
	m_categories_sel->select(CategoriesStore::get_filter_item
		(CategoriesColumns::ALL));
}

void
MainWindow::on_help_contents()
{	activate_help(*this); }

void
MainWindow::on_help_about()
{
	Gtk::AboutDialog about;
	about.set_transient_for(*this);

	about.set_name(_("assoGiate"));
	about.set_version(VERSION);
	about.set_comments(_("An editor of the file types database for GNOME."));
	about.set_website("http://www.kdau.com/projects/assogiate/");
	about.set_logo_icon_name("assogiate");

	about.set_copyright(
		"Copyright © 2007 Kevin Daughtridge"
	);
	about.set_license(_("This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.\n\nThis program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.\n\nYou should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA"));
#if GTK_CHECK_VERSION(2, 8, 0)
	gtk_about_dialog_set_wrap_license(about.gobj(), true);
#endif

	std::list<ustring> authors;
	authors.push_back("Kevin Daughtridge <kevin@kdau.com>");

	about.set_authors(authors);
	/* Translators: Replace this with your name(s) and address(es). */
	about.set_translator_credits(_("translator-credits"));

	about.present();
	about.run();
}

} /* namespace assoGiate */
