# bzr-dbus: dbus support for bzr/bzrlib.
# Copyright (C) 2007 Canonical Limited.
#   Author: Robert Collins.
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
# 

"""URL mapping facility."""


class URLMapper(object):
    """A class to help map private URL's to public ones.
    
    Instances have the following attributes:
    maps: A dict of source_url:[target_url] relationships. When mapping
        a url, all source_urls that are a prefix of the url being mapped
        are used. To prevent false matches, its recommended that source_url end
        in a / (unless of course you want the behaviour of replacing other
        directories with the same prefix.
    """

    def __init__(self):
        """Create a new URLMapper."""
        self.maps = {}

    def add_map(self, source_prefix, target_prefix):
        """Add a url prefix to be mapped when advertising revisions."""
        source_prefix = self.canonicalise_source_prefix(source_prefix)
        self.maps.setdefault(source_prefix, []).append(target_prefix)

    def canonicalise_source_prefix(self, source_prefix):
        """Remove the readonly+ prefix from source_prefix if it is there."""
        if source_prefix.startswith('readonly+'):
            return source_prefix[len('readonly+'):]
        return source_prefix

    def map(self, url):
        """map url to a list of translated urls.

        :param url: The url to map.
        :result: A list of mapped urls sorted by the inverse length of the
            source_url used to perform the mapping. When no mapping is found,
            the an empty list is returned.
        """
        maps = reversed(sorted(self.maps.items(), key=lambda x:len(x)))
        result = []
        for source, targets in maps:
            if url.startswith(source):
                suffix = url[len(source):]
                for target in targets:
                    result.append(target + suffix)
        return result

    def remove_map(self, source_prefix, target_prefix):
        """Remove the mapping of source_prefix to target_prefix."""
        source_prefix = self.canonicalise_source_prefix(source_prefix)
        pos = self.maps[source_prefix].index(target_prefix)
        del self.maps[source_prefix][pos]
        if not self.maps[source_prefix]:
            del self.maps[source_prefix]
