/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <string.h>

#include "cdw_task.h"
#include "gettext.h"
#include "cdw_string.h"
#include "cdw_widgets.h"
#include "cdw_debug.h"
#include "cdw_form.h"
#include "cdw_xorriso_options.h"
#include "cdw_iso9660.h"


static cdw_task_t *local_task = (cdw_task_t *) NULL;


static FIELD *page_xorriso_iso_fields[CDW_XORRISO_ISO_OPTIONS_N_FIELDS + 1]; /* +1 for last field == NULL */

static CDW_DROPDOWN *cdw_xorriso_iso_options_make_rock_ridge_dropdown(WINDOW *window, int begin_y, int begin_x, int width);

static cdw_form_dropdown_maker_t dropdown_makers[] = {
	cdw_xorriso_iso_options_make_rock_ridge_dropdown,
};




enum {
	f_xorriso_iso_joliet_cb,
	f_xorriso_iso_joliet_l,
	f_xorriso_iso_rock_ridge_l,
	f_xorriso_iso_rock_ridge_dd,
	f_xorriso_iso_follow_symlinks_cb,
	f_xorriso_iso_follow_symlinks_l,
	f_xorriso_iso_pad_cb,
	f_xorriso_iso_pad_l,
	f_xorriso_iso_other_l,
	f_xorriso_iso_other_i };



cdw_rv_t cdw_xorriso_iso_options_form(cdw_form_t *cdw_form, void *cdw_task, int first_col, int second_col, int width_wide, int width_narrow)
{
	cdw_task_t *task = (cdw_task_t *) cdw_task;
	local_task = task;
	cdw_form->fields = page_xorriso_iso_fields;

	int rr_width = 15;
	cdw_form_descr_t descr[] = {
		/*     type        begin_y   begin_x       n_cols           n_lines  field enum              data1                                       data2 */

		{ CDW_WIDGET_CHECKBOX, 1,  first_col + 1,  0,              0,  f_xorriso_iso_joliet_cb,           (void *) NULL,  task->create_image.joliet_information ? 1 : 0 },
		{ CDW_WIDGET_LABEL,    1,  first_col + 4,  width_narrow,   1,  f_xorriso_iso_joliet_l,            _("Joliet information"),                                    0 },

		{ CDW_WIDGET_LABEL,    2,  first_col,      width_narrow,   1,  f_xorriso_iso_rock_ridge_l,        _("Rock Ridge information"),                                0 },
		{ CDW_WIDGET_DROPDOWN, 2,  second_col,     rr_width,       1,  f_xorriso_iso_rock_ridge_dd,       dropdown_makers,                                            0 },

		{ CDW_WIDGET_CHECKBOX, 3,  first_col + 1,  0,              0,  f_xorriso_iso_follow_symlinks_cb,  (void *) NULL,     task->create_image.follow_symlinks ? 1 : 0 },
		{ CDW_WIDGET_LABEL,    3,  first_col + 4,  width_narrow,   1,  f_xorriso_iso_follow_symlinks_l,   _("Follow symbolic links"),                                 0 },

		{ CDW_WIDGET_CHECKBOX, 4,  first_col + 1,  0,              0,  f_xorriso_iso_pad_cb,              (void *) NULL,                 task->create_image.pad ? 1 : 0 },
		{ CDW_WIDGET_LABEL,    4,  first_col + 4,  width_narrow,   1,  f_xorriso_iso_pad_l,               _("Pad at the end"),                                        0 },

		{ CDW_WIDGET_LABEL,    6,  first_col,      width_wide,     1,  f_xorriso_iso_other_l,             _("Other xorriso options for ISO9660:"),                    0 },
		{ CDW_WIDGET_INPUT,    7,  first_col,      width_wide - 3, 1,  f_xorriso_iso_other_i,             task->create_image.xorriso.other_xorriso_iso_options,       0 },

		/* guard */
		{ -1,                  0,  0,              0,              0,  0,                                 (void *) NULL,                                              0 }};

	cdw_rv_t crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





#define N_RR_ITEMS_MAX 2 /* useful/full */
static cdw_id_clabel_t rr_items[N_RR_ITEMS_MAX] = {
	/* 2TRANS: useful = some values of attributes of file system are set to some more useful values */
	{ CDW_ISO9660_RR_USEFUL,       gettext_noop("Useful") },
	/* 2TRANS: full = full Rock Ridge extensions */
	{ CDW_ISO9660_RR_FULL,         gettext_noop("Full")   }};


CDW_DROPDOWN *cdw_xorriso_iso_options_make_rock_ridge_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	cdw_id_t rock_ridge = local_task->create_image.rock_ridge;
	cdw_assert (rock_ridge >= CDW_ISO9660_RR_USEFUL && rock_ridge <= CDW_ISO9660_RR_FULL,
		    "ERROR: invalid value of Rock Ridge: %lld\n", rock_ridge);
	if (rock_ridge == CDW_ISO9660_RR_NONE) {
		rock_ridge = CDW_ISO9660_RR_USEFUL;
	}
	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, N_RR_ITEMS_MAX, rr_items);
	cdw_dropdown_set_current_item_by_id(dd, rock_ridge);

	return dd;
}





cdw_rv_t cdw_xorriso_iso_options_validate(cdw_form_t *cdw_form, int *fi)
{
	char *s = cdw_ncurses_get_field_string(*(cdw_form->fields + f_xorriso_iso_other_i));
	cdw_rv_t crv = cdw_string_security_parser(s, (char *) NULL);
	if (crv != CDW_OK) {
		*fi = f_xorriso_iso_other_i;
		return CDW_NO;
	}

	return CDW_OK;
}





cdw_rv_t cdw_xorriso_iso_options_save(cdw_form_t *cdw_form, cdw_task_t *task)
{
	task->create_image.joliet_information = cdw_form_get_checkbox_state(cdw_form, f_xorriso_iso_joliet_cb);


	CDW_DROPDOWN *dd = cdw_form_get_dropdown(cdw_form, f_xorriso_iso_rock_ridge_dd);
	int rr = (int) cdw_dropdown_get_current_item_id(dd);
	cdw_assert (rr == CDW_ISO9660_RR_NONE
		    || rr == CDW_ISO9660_RR_USEFUL
		    || rr == CDW_ISO9660_RR_FULL, "ERROR: field stores invalid value of Rock Ridge: %d\n", rr);
	task->create_image.rock_ridge = rr;


	task->create_image.follow_symlinks = cdw_form_get_checkbox_state(cdw_form, f_xorriso_iso_follow_symlinks_cb);


	task->create_image.pad = cdw_form_get_checkbox_state(cdw_form, f_xorriso_iso_pad_cb);


	cdw_string_set(&(task->create_image.xorriso.other_xorriso_iso_options), cdw_ncurses_get_field_string(*(cdw_form->fields + f_xorriso_iso_other_i)));

	return CDW_OK;
}







enum {
	f_xorriso_burn_other_l,
	f_xorriso_burn_other_i };


static FIELD *page_xorriso_burn_fields[CDW_XORRISO_BURN_OPTIONS_N_FIELDS + 1]; /* +1 for last field == NULL */


cdw_rv_t cdw_xorriso_burn_options_form(cdw_form_t *cdw_form, void *cdw_task, int first_col, __attribute__((unused)) int second_col, int width_wide, __attribute__((unused)) int width_narrow)
{
	cdw_task_t *task = (cdw_task_t *) cdw_task;
	local_task = task;
	cdw_form->fields = page_xorriso_burn_fields;

	cdw_form_descr_t descr[] = {
		/*     type     begin_y  begin_x     n_cols    n_lines   field enum               data1                                       data2 */

		{ CDW_WIDGET_LABEL,  6,  first_col,  width_wide,     1,  f_xorriso_burn_other_l,  _("Other xorriso options for writing:"),        0 },
		{ CDW_WIDGET_INPUT,  7,  first_col,  width_wide - 3, 1,  f_xorriso_burn_other_i,  task->burn.xorriso.other_xorriso_burn_options,  0 },

		/* guard */
		{ -1,                0,  0,          0,              0,  0,                       (void *) NULL,                                  0 }};

	cdw_rv_t crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





cdw_rv_t cdw_xorriso_burn_options_validate(cdw_form_t *cdw_form, int *fi)
{
	char *s = cdw_ncurses_get_field_string(*(cdw_form->fields + f_xorriso_burn_other_i));
	cdw_rv_t crv = cdw_string_security_parser(s, (char *) NULL);
	if (crv != CDW_OK) {
		*fi = f_xorriso_burn_other_i;
		return CDW_NO;
	}

	return CDW_OK;
}





cdw_rv_t cdw_xorriso_burn_options_save(cdw_form_t *cdw_form, cdw_task_t *task)
{
	cdw_string_set(&(task->burn.xorriso.other_xorriso_burn_options), cdw_ncurses_get_field_string(*(cdw_form->fields + f_xorriso_burn_other_i)));

	return CDW_OK;
}
