/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
   \file cdw_iso9660_wizard.c
*/

#define _GNU_SOURCE /* asprintf() */
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>

#include "cdw_file_picker.h"
#include "cdw_config_window.h"
#include "gettext.h"
#include "cdw_string.h"
#include "cdw_widgets.h"
#include "cdw_ncurses.h"
#include "cdw_main_window.h"
#include "cdw_debug.h"
#include "cdw_ext_tools.h"
#include "cdw_form.h"
#include "cdw_task.h"
#include "cdw_mkisofs_options.h"
#include "cdw_mkisofsrc.h"
#include "cdw_xorriso_options.h"
#include "cdw_xorrisorc.h"
#include "cdw_tabs_window.h"
#include "cdw_iso9660_wizard.h"


enum field_ids {
	f_volume_id_l = 0,
	f_volume_id_i,
	f_iso_file_l,
	f_iso_file_i,
	f_iso_file_b,
	f_options_b,
	f_create_b,
	f_cancel_b,
};


#define CDW_WRITE_WIZARD_N_FIELDS 8
static FIELD *wizard_fields[CDW_WRITE_WIZARD_N_FIELDS + 1];

static cdw_rv_t cdw_iso9660_wizard_options(cdw_task_t *task);
static void     cdw_iso9660_options_window_destroy(void);
static cdw_rv_t cdw_iso9660_wizard_options_window_driver(cdw_task_t *task);

static cdw_rv_t cdw_iso9660_wizard_init(void);
static cdw_rv_t cdw_iso9660_wizard_build(cdw_task_t *task);
static void     cdw_iso9660_wizard_destroy(void);
static int      cdw_iso9660_wizard_driver(void);
static cdw_rv_t cdw_iso9660_wizard_options_validate_and_save(cdw_task_t *task, cdw_id_t *page_id, int *fi);
static cdw_rv_t cdw_iso9660_wizard_validate_and_save(cdw_task_t *task, int *fi);

static int cdw_iso9660_wizard_on_enter_on_cancel(cdw_form_t *cdw_form, void *dummy);
static int cdw_iso9660_wizard_on_enter_on_create(cdw_form_t *cdw_form, void *dummy);
static int cdw_iso9660_wizard_on_enter_on_options(cdw_form_t *cdw_form, void *dummy);
static int cdw_iso9660_wizard_on_enter_on_iso_file(cdw_form_t *cdw_form, void *dummy);

static void cdw_iso9660_wizard_debug_at_exit(cdw_task_t *task);



static struct {
	cdw_form_t *cdw_form;
} wizard;


static cdw_task_t *local_task = (cdw_task_t *) NULL;


/* *** some layout constraints *** */

#define first_col                  2  /* main message and some labels start in leftmost column */
#define volume_id_label_row        2
#define iso_file_label_row         5
#define options_button_row        10
#define buttons_row               13  /* Create and Cancel buttons are at the bottom of window */

#define window_n_cols             52
#define window_n_lines            20
#define subwindow_n_cols           (window_n_cols - 2)
#define subwindow_n_lines          (window_n_lines - 4)
#define wide_label_n_cols          (subwindow_n_cols - 2)
#define narrow_label_n_cols        (subwindow_n_cols - 15)



/**
   \brief Ncurses window showing summary of prepared write process

   A window that shows 'You are writing to X' message and mode and speed of
   writing. Mode and speed can be adjusted in the window. User can press
   'More options' button to see and change other options. Two other
   buttons available are 'Write' and 'Cancel'.

   \param task - variable describing current task

   \return CDW_OK if user selects 'Write' button
   \return CDW_CANCEL if user presses Escape key or selects 'Cancel' button
   \return CDW_GEN_ERROR on errors
*/
cdw_rv_t cdw_iso9660_wizard(cdw_task_t *task)
{
	cdw_assert (task->id == CDW_TASK_CREATE_IMAGE,
		    "ERROR: incorrect task id %lld\n", task->id);

	local_task = task;

	cdw_rv_t crv = cdw_iso9660_wizard_init();
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to init wizard\n");
		cdw_iso9660_wizard_destroy();
		return CDW_ERROR;
	}
	crv = cdw_iso9660_wizard_build(task);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to build wizard\n");
		cdw_iso9660_wizard_destroy();
		return CDW_ERROR;
	}

	wrefresh(wizard.cdw_form->subwindow);

	int rv = cdw_iso9660_wizard_driver();
	if (rv == CDW_KEY_ENTER) {
#ifndef NDEBUG
		/* use task variable, to which wizard saves result
		   at return, not values from dropdowns/checkboxes */
		cdw_iso9660_wizard_debug_at_exit(task);
#endif
	} else {
		cdw_vdm ("INFO: not attempting writing, not displaying settings\n");
	}

	cdw_iso9660_wizard_destroy();

	/* redraw parent */
	cdw_main_ui_main_window_wrefresh();

	cdw_vdm ("INFO: path at exit from wizard: %s\n", task->create_image.iso9660_file_fullpath);

	return rv == CDW_KEY_ENTER ? CDW_OK : CDW_CANCEL;
}





/**
   \brief Set initial values of some variables global in the file

   Function sets values of various fields of "wizard" variable.
*/
cdw_rv_t cdw_iso9660_wizard_init(void)
{
	wizard.cdw_form = cdw_form_new(CDW_WRITE_WIZARD_N_FIELDS);
	if (wizard.cdw_form == (cdw_form_t *) NULL) {
		cdw_vdm ("ERROR: failed to create cdw form\n");
		return CDW_ERROR;
	}

	wizard.cdw_form->fields = wizard_fields;

	return CDW_OK;
}





int cdw_iso9660_wizard_driver(void)
{
	int fi = 0;
	int key = 'a';
	while (key != CDW_KEY_ESCAPE && key != 'q' && key != 'Q') {
		key = cdw_form_driver(wizard.cdw_form, fi);

		fi = field_index(current_field(wizard.cdw_form->form));
		if (key == CDW_KEY_ENTER) {
			if (fi == f_create_b) {

				/* flush */
				form_driver(wizard.cdw_form->form, REQ_VALIDATION);

				cdw_rv_t crv = cdw_iso9660_wizard_validate_and_save(local_task, &fi);
				if (crv == CDW_NO) {
					/* one of option strings in main
					   wizard window is invalid */

					/* 2TRANS: this is title of dialog window */
					cdw_buttons_dialog(_("Error"),
							   /* 2TRANS: this is message in dialog window */
							   _("One of option fields is incorrect or contains character that is not allowed. Please fix it."),
							   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

					cdw_form_driver_go_to_field(wizard.cdw_form, fi);
					/* loop */
				} else {
					/* fields in main page of the wizard
					   are validated and saved into task
					   variable */
					return CDW_KEY_ENTER;
				}
			} else if (fi == f_cancel_b) {
				return CDW_KEY_ESCAPE;
			} else {
				;
			}
		} else if (key == CDW_KEY_ESCAPE) {
			return CDW_KEY_ESCAPE;
		} else {
			;
		}
	}

	return CDW_KEY_ESCAPE;
}






/**
   \brief Deallocate all resources allocated by write wizard
*/
void cdw_iso9660_wizard_destroy(void)
{
	if (wizard.cdw_form->subwindow != (WINDOW *) NULL) {
		delwin(wizard.cdw_form->subwindow);
		wizard.cdw_form->subwindow = (WINDOW *) NULL;
	}

	if (wizard.cdw_form->window != (WINDOW *) NULL) {
		delwin(wizard.cdw_form->window);
		wizard.cdw_form->window = (WINDOW *) NULL;
	}

	if (wizard.cdw_form != (cdw_form_t *) NULL) {
		cdw_form_delete_form_objects(wizard.cdw_form);
		cdw_form_delete(&(wizard.cdw_form));
	}

	return;
}





/**
   \brief Print to stderr values of \p task fields set in write wizard

   \param task - task variable, in which some fields were set in write wizard
*/
void cdw_iso9660_wizard_debug_at_exit(cdw_task_t *task)
{
	cdw_task_debug_print_iso9660_options(task);
	return;
}





/**
   \brief Create all UI elements in wizard window

   \param task - variable describing current task
   \param disc - variable describing disc currently in drive

   \return CDW_OK on success
   \return CDW_ERROR on failure
*/
cdw_rv_t cdw_iso9660_wizard_build(cdw_task_t *task)
{
	int begin_y = ((LINES - window_n_lines) / 2) - 2;
	int begin_x = (COLS - window_n_cols) / 2;
	wizard.cdw_form->window =
		cdw_ncurses_window_new((WINDOW *) NULL,
				       window_n_lines, window_n_cols,
				       begin_y, begin_x,
				       CDW_COLORS_DIALOG,
				       /* 2TRANS: this is title of wizard
					  window; "write" as in
					  'writing to optical disc" */
				       _("ISO9660 wizard"),
				       /* 2TRANS: this is tip at the bottom
					  of window - user can switch between
					  window elements using tab key */
				       _("Use 'Tab' key to move"));

	if (wizard.cdw_form->window == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create window\n");
		return CDW_ERROR;
	}

	wizard.cdw_form->subwindow =
		cdw_ncurses_window_new(wizard.cdw_form->window,
				       subwindow_n_lines, subwindow_n_cols,
				       3, 1,
				       CDW_COLORS_DIALOG,
				       (char *) NULL, (char *) NULL);

	if (wizard.cdw_form->subwindow == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create subwindow\n");
		return CDW_ERROR;
	}


	int iso_button_col = first_col + wide_label_n_cols - 5;

	cdw_form_descr_t descr[] = {
		/*     type             begin_y                   begin_x         n_cols                n_lines    field enum      data1                  data2 */
		{ CDW_WIDGET_LABEL,   volume_id_label_row,      first_col,      wide_label_n_cols,       1,  f_volume_id_l,  _("Volume ID:"),                                  0 },
		{ CDW_WIDGET_INPUT,   volume_id_label_row + 1,  first_col,      CDW_ISO9660_VOLI_LEN,    1,  f_volume_id_i,  task->create_image.volume_id,  CDW_ISO9660_VOLI_LEN },

		{ CDW_WIDGET_LABEL,   iso_file_label_row,       first_col,      wide_label_n_cols - 2,   1,  f_iso_file_l,   _("ISO9660 image file:"),                         0 },
		{ CDW_WIDGET_INPUT,   iso_file_label_row + 1,   first_col,      wide_label_n_cols - 8,   1,  f_iso_file_i,   task->create_image.iso9660_file_fullpath,         0 },
		{ CDW_WIDGET_BUTTON,  iso_file_label_row + 1,   iso_button_col, 1,                       1,  f_iso_file_b,   _(">"),                           CDW_COLORS_DIALOG },

		{ CDW_WIDGET_BUTTON,  options_button_row,       first_col,      1,                       1,  f_options_b,    _("More options"),                CDW_COLORS_DIALOG },

		/* 2TRANS: button label, it refers to writing to optical disc */
		{ CDW_WIDGET_BUTTON,  buttons_row,               3,             2,                       1,  f_create_b,     _("Create"),                      CDW_COLORS_DIALOG },
		/* 2TRANS: button label */
		{ CDW_WIDGET_BUTTON,  buttons_row,              15,             2,                       1,  f_cancel_b,     _("Cancel"),                      CDW_COLORS_DIALOG },

		/* guard */
		{ -1,                 0,                         0,             0,                       0,  0,              (void *) NULL,                                    0 }};


	wizard.cdw_form->n_fields = CDW_WRITE_WIZARD_N_FIELDS;

	cdw_rv_t crv = cdw_form_description_to_fields(descr, wizard.cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	}

	wizard.cdw_form->form = cdw_ncurses_new_form(wizard.cdw_form->window,
						     wizard.cdw_form->subwindow,
						     wizard.cdw_form->fields);
	if (wizard.cdw_form->form == (FORM *) NULL) {
		cdw_vdm ("ERROR: failed to create form\n");
		return CDW_ERROR;
	}

	/* current items are set in functions creating respective
	   dropdowns; now just display them */
	cdw_form_redraw_widgets(wizard.cdw_form);

	cdw_form_add_return_chars(wizard.cdw_form, CDW_KEY_ENTER, CDW_KEY_ESCAPE, 'q', 'Q', 0);

	cdw_form_set_function(wizard.cdw_form, f_create_b,   cdw_iso9660_wizard_on_enter_on_create);
	cdw_form_set_function(wizard.cdw_form, f_cancel_b,   cdw_iso9660_wizard_on_enter_on_cancel);
	cdw_form_set_function(wizard.cdw_form, f_iso_file_b, cdw_iso9660_wizard_on_enter_on_iso_file);
	cdw_form_set_function(wizard.cdw_form, f_options_b,  cdw_iso9660_wizard_on_enter_on_options);

	wrefresh(wizard.cdw_form->subwindow);
	wrefresh(wizard.cdw_form->window);

	return CDW_OK;
}





int cdw_iso9660_wizard_on_enter_on_cancel(__attribute__((unused)) cdw_form_t *cdw_form, __attribute__((unused)) void *dummy)
{
	return CDW_KEY_ESCAPE;
}





int cdw_iso9660_wizard_on_enter_on_create(__attribute__((unused)) cdw_form_t *cdw_form, __attribute__((unused)) void *dummy)
{
	return CDW_KEY_ENTER;
}





int cdw_iso9660_wizard_on_enter_on_options(cdw_form_t *cdw_form, __attribute__((unused)) void *dummy)
{
	cdw_rv_t crv = cdw_iso9660_wizard_options(local_task);

	cdw_main_ui_main_window_wrefresh();

	redrawwin(cdw_form->window);
	redrawwin(cdw_form->subwindow);
	wrefresh(cdw_form->window);
	wrefresh(cdw_form->subwindow);

	/* the function needs to return int representing a char */
	if (crv == CDW_OK) {
		return '1';
	} else {
		return '0';
	}
}




int cdw_iso9660_wizard_on_enter_on_iso_file(cdw_form_t *cdw_form, __attribute__((unused)) void *dummy)
{
	/* 2TRANS: this is title of dialog window */
	cdw_rv_t crv = cdw_fs_ui_file_picker(_("Path to iso image"),
					     /* 2TRANS: this is message in dialog window;
					     below it there is an input field where user can
					     enter path to ISO image file */
					     _("Please enter FULL path to new iso image file:"),
					     &(local_task->create_image.iso9660_file_fullpath),
					     CDW_FS_FILE, R_OK | W_OK, CDW_FS_NEW | CDW_FS_EXISTING);

	cdw_main_ui_main_window_wrefresh();

	redrawwin(wizard.cdw_form->window);
	wrefresh(wizard.cdw_form->window);
	redrawwin(wizard.cdw_form->subwindow);
	wrefresh(wizard.cdw_form->subwindow);

	char *fp = local_task->create_image.iso9660_file_fullpath;
	if (crv == CDW_OK) {
		int rv = set_field_buffer(cdw_form->fields[f_iso_file_i], 0, fp);
		if (rv != E_OK) {
			cdw_vdm ("ERROR: failed to set field buffer with string = \"%s\"\n", fp);
		} else {
			form_driver(cdw_form->form, REQ_END_LINE);
		}
		cdw_vdm ("INFO: file picker returns \"%s\"\n", fp);
		return '1';
	} else if (crv == CDW_CANCEL) {
		cdw_vdm ("INFO: pressed escape in file picker, fullpath is \"%s\"\n", fp);
		return '0';
	} else {
		cdw_vdm ("INFO: file picker returns CDW_ERROR, fullpath is \"%s\"\n", fp);
		return '0';
	}
}









/* some size and layout constants */
/* width and height of main options window can't be smaller
   than size of minimal supported terminal */

/** \brief Width of options window */
#define OPTIONS_WIDTH 80
/** \brief Height of options window */
#define OPTIONS_HEIGHT 24
/** \brief Width of right-hand area with tabs */
#define TABS_WIDTH 23
/** \brief Width of (most) labels in options window */
#define LABEL_WIDTH 30
/** \brief Width of some fields that should be as wide as possible: input
    fields storing file path, some dropdowns, "other options" fields,
    txt subwins, long labels */
#define WIDE_FIELD_WIDTH (OPTIONS_WIDTH - TABS_WIDTH - 4)
/** \brief Window column of first column of items in options page  */
#define FIRST_COL 1
/** \brief Window column of second column of items in options page  */
#define SECOND_COL (FIRST_COL + LABEL_WIDTH + 2)


/* text displayed at the bottom of wizard window */
static char *cdw_wizard_window_tip = (char *) NULL;


#define ISO9660_PAGE_ID_MKISOFS   0
#define ISO9660_PAGE_ID_XORRISO   1
#define ISO9660_PAGE_ID_MKISOFSRC 2
#define ISO9660_PAGE_ID_XORRISORC 3


/* this is first main component of options window: a big table
   aggregating data and functions that prepare the data; the data
   represents widgets displayed in pages of options window */
static cdw_config_page_t c_pages[] = {
	{ false, CDW_MKISOFS_OPTIONS_N_FIELDS,       cdw_mkisofs_options_form,       gettext_noop("ISO9660"),   (cdw_form_t *) NULL },
	{ false, CDW_XORRISO_ISO_OPTIONS_N_FIELDS,   cdw_xorriso_iso_options_form,   gettext_noop("ISO9660"),   (cdw_form_t *) NULL },
	{ false, CDW_MKISOFSRC_OPTIONS_N_FIELDS,     cdw_mkisofsrc_options_form,     gettext_noop("Meta data"), (cdw_form_t *) NULL },
	{ false, CDW_XORRISORC_N_FIELDS,             cdw_xorrisorc_form,             gettext_noop("Meta data"), (cdw_form_t *) NULL }};


/* this is a pointer to second main component of options window:
   tabbed window, which enables switching between pages */
static cdw_tabs_window_t *c_twindow = (cdw_tabs_window_t *) NULL;




cdw_rv_t cdw_iso9660_wizard_options(cdw_task_t *task)
{
	/* 2TRANS: this is message at the bottom of options window;
	   cancel means: quit without saving; '%d' is an integer used to
	   create label of function key, e.g. "F10" */
	int rv = asprintf(&cdw_wizard_window_tip, _("Press F%d key to save changes or ESC to cancel"), CDW_CONFIG_UI_SnC_KEY);
	if (cdw_wizard_window_tip == (char *) NULL || rv == -1) {
		cdw_vdm ("ERROR: failed to create window tip\n");
		return CDW_ERROR;
	}
	int n_tabs = 2; /* mkisofs|xorriso + mkisofsrc|xorrisorc */
	/* 2TRANS: this is title of cdw options main window */
	c_twindow = cdw_tabs_window_init(n_tabs, _("ISO9660 options"), cdw_wizard_window_tip);
	cdw_tabs_window_set_geometry(c_twindow, OPTIONS_HEIGHT, OPTIONS_WIDTH, 4, 4, TABS_WIDTH);


	/* prepare tabbed window */
	cdw_id_t id = 0;
	int ind = 0;
	/* tool-specific options: mkisofs or xorriso */
	if (task->create_image.tool.id == CDW_TOOL_MKISOFS) {
		id = ISO9660_PAGE_ID_MKISOFS;
		cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
		ind++;

		id = ISO9660_PAGE_ID_MKISOFSRC;
		cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
		ind++;
	} else {
		id = ISO9660_PAGE_ID_XORRISO;
		cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
		ind++;

		id = ISO9660_PAGE_ID_XORRISORC;
		cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
		ind++;
	}

	cdw_tabs_window_add_return_key(c_twindow, KEY_F(CDW_CONFIG_UI_SnC_KEY));
	cdw_tabs_window_add_return_key(c_twindow, CDW_KEY_ESCAPE);
	cdw_tabs_window_add_return_key(c_twindow, 'q');
	cdw_tabs_window_add_return_key(c_twindow, 'Q');
	c_twindow->driver_reader = cdw_config_ui_tabs_window_form_reader;
	cdw_tabs_window_finalize(c_twindow);

	/* prepare forms in the tabbed window */
	for (int i = 0; i < n_tabs; i++) {
		id = c_twindow->tabs[i].id;

		c_pages[id].cdw_form = cdw_form_new(c_pages[id].n_fields);
		c_pages[id].cdw_form->form_id = id;
		c_pages[id].cdw_form->window = c_twindow->tabs[i].window;
		c_pages[id].cdw_form->subwindow = c_twindow->tabs[i].subwindow;

		c_pages[id].visible = true;

		cdw_rv_t crv = c_pages[id].fields_builder(c_pages[id].cdw_form, task, FIRST_COL, SECOND_COL, WIDE_FIELD_WIDTH, LABEL_WIDTH);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to build fields in page #%d\n", i);
			break;
		}

		c_pages[id].cdw_form->form = cdw_ncurses_new_form(c_pages[id].cdw_form->window,
								  c_pages[id].cdw_form->subwindow,
								  c_pages[id].cdw_form->fields);

		for (int j = 0; j < n_tabs; j++) {
			cdw_form_add_return_char(c_pages[id].cdw_form, KEY_F(j + CDW_CONFIG_UI_FX_START_KEY));
		}
		cdw_form_add_return_char(c_pages[id].cdw_form, KEY_F(CDW_CONFIG_UI_SnC_KEY));
		cdw_form_add_return_char(c_pages[id].cdw_form, 'q'); /* will work only for non-text fields */
	}
	c_twindow->user_data = (void *) c_pages;

	cdw_tabs_window_draw_tabs(c_twindow);
	cdw_config_ui_tabs_window_form_draw(c_twindow, c_pages);

	bool old_follow_symlinks = task->create_image.follow_symlinks;

	cdw_rv_t crv = cdw_iso9660_wizard_options_window_driver(task);
	cdw_iso9660_options_window_destroy();
	if (crv == CDW_OK) {
		cdw_main_ui_handle_follow_symlinks_change(old_follow_symlinks, task->create_image.follow_symlinks);
		return CDW_OK;
	} else {
		return CDW_CANCEL;
	}
}





/**
   \brief Destructor function for cdw options window
*/
void cdw_iso9660_options_window_destroy(void)
{
	if (cdw_wizard_window_tip != (char *) NULL) {
		free(cdw_wizard_window_tip);
		cdw_wizard_window_tip = (char *) NULL;
	}

	for (int i = 0; i < c_twindow->n_tabs; i++) {
		cdw_id_t id = c_twindow->tabs[i].id;
		if (c_pages[id].cdw_form != (cdw_form_t *) NULL) {
			cdw_form_delete_form_objects(c_pages[id].cdw_form);
		}
		if (c_pages[id].cdw_form != (cdw_form_t *) NULL) {
			cdw_form_delete(&(c_pages[id].cdw_form));
		}
	}

	cdw_tabs_window_delete(&c_twindow);

	return;
}





/**
   \brief Function reacting to keys pressed when options window is displayed

   Function catches all keys pressed when options window is displayed,
   and reacts to them either by switching pages, or by passing the keys to
   page driver(s).

   Keys configured as hotkeys for "tabbed window" tabs are used to switch
   between pages. F(CDW_CONFIG_SnC_KEY) key is interpreted as "save and close"
   key - values of all option fields are stored in \p config and validated.
   Function then returns CDW_OK if validation is successful, or displays
   incorrect field to user.
   ESCAPE key causes the function to return with CDW_CANCEL.

   \param config - variable in which function saves values of all option fields

   \return CDW_CANCEL if user pressed ESCAPE key in options window
   \return CDW_OK if user pressed F10 and validation of \p tmp config was successful
*/
cdw_rv_t cdw_iso9660_wizard_options_window_driver(cdw_task_t *task)
{
	while (1) {
		int key = cdw_tabs_window_driver(c_twindow);
		if (key == CDW_KEY_ESCAPE || key == 'q' || key == 'Q') {
			break;
		} else if (key == KEY_F(CDW_CONFIG_UI_SnC_KEY)) { /* SnC = Save and Close */

			/* flush */
			for (int i = 0; i < c_twindow->n_tabs; i++) {
				cdw_id_t id = c_twindow->tabs[i].id;
				form_driver(c_pages[id].cdw_form->form, REQ_VALIDATION);
			}
			cdw_id_t page_id = 0;
			int fi = 0;
			cdw_rv_t valid = cdw_iso9660_wizard_options_validate_and_save(task, &page_id, &fi);
			if (valid == CDW_NO) { /* some option field is invalid */

				/* 2TRANS: this is title of dialog window */
				cdw_buttons_dialog(_("Error"),
						   /* 2TRANS: this is message in dialog window */
						   _("One of option fields is incorrect or contains character that is not allowed. Please fix it."),
						   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

				cdw_tabs_window_show_tab_by_id(c_twindow, page_id);
				cdw_form_driver_go_to_field(c_pages[page_id].cdw_form, fi);

				/* loop */
			} else {
				/* options from window with advanced
				   options are validated and saved into
				   task variable */
				return CDW_OK;
			}
		} else {
			/* loop */
		}
	} /* while (1) */

	return CDW_CANCEL;
}





cdw_rv_t cdw_iso9660_wizard_options_validate_and_save(cdw_task_t *task, cdw_id_t *page_id, int *fi)
{
	if (task->create_image.tool.id == CDW_TOOL_MKISOFS) {
		cdw_id_t id = ISO9660_PAGE_ID_MKISOFS;
		cdw_rv_t crv = cdw_mkisofs_options_validate(c_pages[id].cdw_form, fi);
		if (crv == CDW_NO) {
			*page_id = id;
			return CDW_NO;
		}
		id = ISO9660_PAGE_ID_MKISOFSRC;
		crv = cdw_mkisofsrc_options_validate(c_pages[id].cdw_form, fi);
		if (crv == CDW_NO) {
			*page_id = id;
			return CDW_NO;
		}
	} else if (task->create_image.tool.id == CDW_TOOL_XORRISO) {
		cdw_id_t id = ISO9660_PAGE_ID_XORRISO;
		cdw_rv_t crv = cdw_xorriso_iso_options_validate(c_pages[id].cdw_form, fi);
		if (crv == CDW_NO) {
			*page_id = id;
			return CDW_NO;
		}
		id = ISO9660_PAGE_ID_XORRISORC;
		crv = cdw_xorrisorc_validate(c_pages[id].cdw_form, fi);
		if (crv == CDW_NO) {
			*page_id = id;
			return CDW_NO;
		}
	} else {
		cdw_assert (0, "ERROR: invalid tool id for task \"create image\": %lld\n",
			    task->create_image.tool.id);
		return CDW_NO;
	}


	if (task->create_image.tool.id == CDW_TOOL_MKISOFS) {
		cdw_id_t id = ISO9660_PAGE_ID_MKISOFS;
		cdw_mkisofs_options_save(c_pages[id].cdw_form, task);

		id = ISO9660_PAGE_ID_MKISOFSRC;
		cdw_mkisofsrc_options_save(c_pages[id].cdw_form, task);
	} else {
		cdw_id_t id = ISO9660_PAGE_ID_XORRISO;
		cdw_xorriso_iso_options_save(c_pages[id].cdw_form, task);

		id = ISO9660_PAGE_ID_XORRISORC;
		cdw_xorrisorc_save(c_pages[id].cdw_form, task);
	}

	return CDW_OK;
}





/**
   \brief Validate and save values of options
   available in main page of the wizard

   Values of options will be saved in \p task.
   If any value is invalid, index of corresponding field
   will be saved in \fi, and function will return CDW_NO.

   \param task - variable describing current task
   \param fi - field index

   \return CDW_OK on success
   \return CDW_NO if any value in main wizard page is invalid
*/
cdw_rv_t cdw_iso9660_wizard_validate_and_save(cdw_task_t *task, int *fi)
{
	/* validate */
	const char *s = field_buffer(wizard.cdw_form->fields[f_volume_id_i], 0);
	cdw_rv_t crv = cdw_string_security_parser(s, (char *) NULL);
	if (crv == CDW_NO) {
		*fi = f_volume_id_i;
		return CDW_NO;
	}

	s = field_buffer(wizard.cdw_form->fields[f_iso_file_i], 0);
	crv = cdw_string_security_parser(s, (char *) NULL);
	if (crv == CDW_NO) {
		*fi = f_iso_file_i;
		return CDW_NO;
	}

	/* save */
	s = cdw_form_get_string(wizard.cdw_form, f_volume_id_i);
	strncpy(task->create_image.volume_id, s, CDW_ISO9660_VOLI_LEN);
	task->create_image.volume_id[CDW_ISO9660_VOLI_LEN] = '\0';

	s = cdw_form_get_string(wizard.cdw_form, f_iso_file_i);
	cdw_string_set(&(task->create_image.iso9660_file_fullpath), s);

	return CDW_OK;
}
