/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
   \file write_wizard.c
   \brief Window with basic write options, displayed when user selects writing action

   The window displays message "You are writing to X disc", allows user to
   select writing speed, and writing mode. 'More options' button is also
   available, so user can jump to separate window to set more advanced
   options without closing write wizard.

   If current task is writing ISO image to blank CD/DVD, then 'Verify' checkbox
   is also available. It is coupled with task->burn.verify (where task
   is argument passed to write_wizard()).

   Writing speed values and modes that can be selected in wizard window are
   calculated every time the window is displayed and are disc-specific.
   They are calculated by functions defined in this file using values stored
   in current_disc variable. Be sure to call run_command_cdrecord_atip(),
   run_command_cdrecord_msinfo() or run_command_dvd_rw_mediainfo() before
   calling write_wizard().

   Code in this file does not check writing preconditions - be sure to check
   burning preconditions before calling write_wizard().
*/

#define _GNU_SOURCE /* asprintf() */
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>

#include "cdw_config.h"
#include "cdw_config_window.h"
#include "gettext.h"
#include "cdw_string.h"
#include "cdw_widgets.h"
#include "cdw_ncurses.h"
#include "cdw_write_wizard.h"
#include "cdw_main_window.h"
#include "cdw_debug.h"
#include "cdw_ext_tools.h"
#include "cdw_form.h"
#include "cdw_growisofs.h"
#include "cdw_growisofs_options.h"
#include "cdw_cdrecord.h"
#include "cdw_cdrecord_options.h"
#include "cdw_mkisofs_options.h"
#include "cdw_xorriso.h"
#include "cdw_xorriso_options.h"
#include "cdw_utils.h"

extern cdw_config_t global_config; /* main cdw configuration variable */


enum field_ids {
	f_header_l = 0,
	f_volume_id_l,
	f_volume_id_i,
	f_speed_l,
	f_speed_dd,
	f_session_l,
	f_session_dd,
	f_disc_l,
	f_disc_dd,
	f_verify_l,
	f_verify_i,
	f_dummy_l,
	f_dummy_i,
	f_eject_l,
	f_eject_cb,
	f_other_options_b,
	f_write_i,
	f_cancel_i
};


static cdw_rv_t cdw_write_wizard_init(cdw_task_t *task, cdw_disc_t *disc);
static cdw_rv_t cdw_write_wizard_build(cdw_task_t *task, cdw_disc_t *disc);
static void     cdw_write_wizard_destroy(void);
static int      cdw_write_wizard_driver(void);
static cdw_rv_t cdw_write_wizard_build_fields(cdw_task_t *task);
static cdw_rv_t cdw_write_wizard_validate_and_save(cdw_task_t *task, int *fi);

static CDW_DROPDOWN *cdw_write_wizard_make_speed_dropdown(WINDOW *window, int begin_y, int begin_x, int width);
static CDW_DROPDOWN *cdw_write_wizard_make_session_mode_dropdown(WINDOW *window, int begin_y, int begin_x, int width);
static CDW_DROPDOWN *cdw_write_wizard_make_disc_mode_dropdown(WINDOW *window, int begin_y, int begin_x, int width);

static cdw_rv_t cdw_write_wizard_options(cdw_task_t *task);
static cdw_rv_t cdw_write_wizard_options_validate_and_save(cdw_task_t *task, cdw_id_t *page_id, int *fi);
static int      cdw_write_wizard_options_window_driver(cdw_task_t *task);
static void     cdw_write_wizard_options_window_destroy(void);

static int cdw_write_wizard_on_enter_on_cancel(cdw_form_t *cdw_form, void *dummy);
static int cdw_write_wizard_on_enter_on_write(cdw_form_t *cdw_form, void *dummy);
static int cdw_write_wizard_on_enter_on_config(cdw_form_t *cdw_form, void *dummy);

static void cdw_write_wizard_debug_at_exit(cdw_task_t *task);
static void cdw_write_wizard_debug_verify_invisible(cdw_task_t *task, cdw_disc_t *disc);

static bool cdw_write_wizard_show_verify_checkbox(cdw_task_t *task, cdw_disc_t *disc);
static bool cdw_write_wizard_show_dummy_checkbox(cdw_task_t *task, cdw_disc_t *disc);


#define CDW_WRITE_WIZARD_N_FIELDS 18
static FIELD *wizard_fields[CDW_WRITE_WIZARD_N_FIELDS + 1];

static struct {
	/* used in code related to 'Verify' checkbox, we can do verification
	   only for first track of data on CD*/
	bool verify_checkbox_visible;

	/* only some disc types allow "dummy" option; use
	   cdw_growisofs_allow_dummy() or cdw_cdrecord_allow_dummy()
	   to check if the checkbox should be visible */
	bool dummy_checkbox_visible;

	/* you can't pass dao/tao/sao parameter to xorriso, so
	   there is no point in showing the dropdown in wizard */
	bool disc_mode_dd_visible;

	bool volume_id_visible;

	char *header; /* main message in wizard window */

	cdw_form_t *cdw_form;
} wizard;



static cdw_form_dropdown_maker_t dropdown_makers[] = {
	cdw_write_wizard_make_speed_dropdown,
	cdw_write_wizard_make_session_mode_dropdown,
	cdw_write_wizard_make_disc_mode_dropdown
};

static cdw_task_t *local_task = (cdw_task_t *) NULL;
static cdw_disc_t *local_disc = (cdw_disc_t *) NULL;


/* *** some layout constraints *** */

#define first_col                  2  /* main message and some labels start in leftmost column */

#define top_label_row              1  /* main message is displayed on top of window */
#define volume_id_label_row        2
#define write_speed_label_row      4
#define session_mode_label_row     6
#define disc_mode_label_row        8
#define verify_checkbox_label_row 11
#define dummy_checkbox_label_row  12
#define eject_checkbox_label_row  13
#define other_options_button_row  14  /* button invoking "other options" window */
#define buttons_row               16  /* Write, Cancel and Help buttons are at the bottom of window */

#define window_n_cols             52
#define window_n_lines            20
#define subwindow_n_cols           (window_n_cols - 2)
#define subwindow_n_lines          (window_n_lines - 2)
#define wide_label_n_cols          (subwindow_n_cols - 2)
#define narrow_label_n_cols        (subwindow_n_cols - 15)
#define wide_dropdown_n_cols       (subwindow_n_cols - 6)
#define narrow_dropdown_n_cols     8



/**
   \brief Ncurses window showing summary of prepared write process

   A window that shows 'You are writing to X' message and mode and speed of
   writing. Mode and speed can be adjusted in the window. User can press
   'Configuration' button to see and change other options. Two other
   buttons available are 'Write' and 'Cancel'.

   \param task - variable describing current task
   \param disc - variable describing disc currently in drive

   \return CDW_OK if user selects 'Write' button
   \return CDW_CANCEL if user presses Escape key or selects 'Cancel' button
   \return CDW_GEN_ERROR on errors
*/
cdw_rv_t cdw_write_wizard(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->id == CDW_TASK_BURN_FROM_FILES || task->id == CDW_TASK_BURN_FROM_IMAGE,
		    "ERROR: incorrect task id %lld\n", task->id);
	cdw_assert (disc->type_writable == CDW_TRUE, "ERROR: disc type is not writable\n");

	local_task = task;
	local_disc = disc;
	wizard.header = (char *) NULL;

	if (disc->type_writable != CDW_TRUE) {
		cdw_vdm ("ERROR: disc type is not writable\n");
		return CDW_ERROR;
	}

	cdw_rv_t crv = cdw_write_wizard_init(task, disc);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to init wizard\n");
		cdw_write_wizard_destroy();
		return CDW_ERROR;
	}
	crv = cdw_write_wizard_build(task, disc);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to build wizard\n");
		cdw_write_wizard_destroy();
		return CDW_ERROR;
	}

	wrefresh(wizard.cdw_form->subwindow);

	int key = cdw_write_wizard_driver();

	cdw_write_wizard_destroy();
	/* redraw parent */
	cdw_main_ui_main_window_wrefresh();

	if (key == CDW_KEY_ENTER) {
#ifndef NDEBUG
		cdw_write_wizard_debug_at_exit(task);
#endif
		if (wizard.disc_mode_dd_visible) {
			cdw_assert (task->burn.disc_mode != CDW_DISC_MODE_INIT, "ERROR: write wizard didn't set disc writing mode\n");
		}
		cdw_assert (task->burn.session_mode != CDW_SESSION_MODE_INIT, "ERROR: write wizard didn't set session writing mode\n");
		return CDW_OK;
	} else {
		cdw_sdm ("INFO: not attempting writing, not displaying settings\n");
		return CDW_CANCEL;
	}
}





/**
   \brief Set initial values of some variables global in the file

   Function sets values of various fields of "wizard" variable.
*/
cdw_rv_t cdw_write_wizard_init(cdw_task_t *task, cdw_disc_t *disc)
{
	wizard.verify_checkbox_visible = cdw_write_wizard_show_verify_checkbox(task, disc);
	wizard.dummy_checkbox_visible = cdw_write_wizard_show_dummy_checkbox(task, disc);
	wizard.disc_mode_dd_visible = task->burn.tool.id == CDW_TOOL_XORRISO ? false : true;
	wizard.volume_id_visible = task->id == CDW_TASK_BURN_FROM_FILES ? true : false;

	wizard.cdw_form = cdw_form_new(CDW_WRITE_WIZARD_N_FIELDS);
	if (wizard.cdw_form == (cdw_form_t *) NULL) {
		cdw_vdm ("ERROR: failed to create cdw form\n");
		return CDW_ERROR;
	}

	return CDW_OK;
}





int cdw_write_wizard_driver(void)
{
	int fi = 0;
	int key = 'a';
	while (key != CDW_KEY_ESCAPE && key != 'q' && key != 'Q') {
		key = cdw_form_driver(wizard.cdw_form, fi);

		/* in this wizard driver we are interested only in
		   ENTER being pressed on either "write" or "cancel"
		   buttons */
		if (key == CDW_KEY_ENTER) {
			fi = field_index(current_field(wizard.cdw_form->form));
			if (fi == f_write_i) {

				/* flush */
				form_driver(wizard.cdw_form->form, REQ_VALIDATION);

				cdw_write_wizard_validate_and_save(local_task, &fi);
				/* not checking return value after validation
				   as none of options available directly in
				   wizard options are validated */
				return CDW_KEY_ENTER;
			} else if (fi == f_cancel_i) {
				return CDW_KEY_ESCAPE;
			} else {
				;
			}
		} else {
			;
		}
	}

	return CDW_KEY_ESCAPE;
}





/* labels for _all_ possible session modes allowed for any optical disc */
cdw_id_clabel_t all_session_modes[] = {
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_SESSION_MODE_START_MULTI,      gettext_noop("Start new, appendable disc") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_SESSION_MODE_CREATE_SINGLE,    gettext_noop("Create non-appendable disc") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
        { CDW_SESSION_MODE_CONTINUE_MULTI,   gettext_noop("Append data, don't close disc") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_SESSION_MODE_WRITE_FINAL,      gettext_noop("Append data and close disc") },
	/* 2TRANS: label visible in dropdown menu - some error occurred */
	{ CDW_SESSION_MODE_ERROR,            gettext_noop("(ERROR)") },
	{ -1,                                (char *) NULL } };


/**
   \brief Create dropdown with values showing allowed session modes

   Function returns dropdown with items presenting session modes acceptable
   by current disc.
   Location of the dropdown in wizard window is specified by variables
   global in this file.

   The function creates all necessary dropdown labels and calls
   cdw_dropdown_finalize() on the dropdown, so result of this function
   is a ready to use dropdown.

   \param task - variable describing current task

   \return NULL on errors
   \return pointer to dropdown on success
*/
CDW_DROPDOWN *cdw_write_wizard_make_session_mode_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	cdw_id_clabel_t items[CDW_SESSION_MODE_N_MAX];
	int n_items = 0;
	for (n_items = 0; n_items < CDW_SESSION_MODE_N_MAX; ) {
		/* session_modes is non-sparse up until
		   element of value CDW_SESSION_MODE_INIT */

		int allowed_mode_id = local_task->burn.session_modes[n_items];
		if (allowed_mode_id == CDW_SESSION_MODE_INIT) {
			break;
		} else {
			items[n_items].id = allowed_mode_id;
			items[n_items].label = cdw_utils_id_label_table_get_label(all_session_modes, allowed_mode_id);
			n_items++;
		}
	}

	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, n_items, items);
	/* 0 - first mode (set by "task" module) is the default one */
	cdw_dropdown_set_current_item_by_ind(dd, 0);
	return dd;
}





/* _all_ possible types of writing mode allowed for any optical disc */
cdw_id_clabel_t all_disc_modes[] = {
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_DISC_MODE_UNSPECIFIED,    gettext_noop("Unspecified") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_DISC_MODE_TAO,            gettext_noop("TAO") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_DISC_MODE_DAO,            gettext_noop("DAO") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_DISC_MODE_SAO,            gettext_noop("SAO") },
#if 0 /* unsupported disc modes */
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_DISC_MODE_RAW,            gettext_noop("RAW") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_DISC_MODE_RAW96P,         gettext_noop("RAW96R") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_DISC_MODE_RAW96R,         gettext_noop("RAW96P") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_DISC_MODE_RAW16,          gettext_noop("RAW16") },
#endif
	/* 2TRANS: label visible in dropdown menu - some error occurred */
	{ CDW_DISC_MODE_ERROR,          gettext_noop("(ERROR)") },

	{ -1,                           "ERROR: guard value in \"disc mode labels\" table" }
};





/**
   \brief Create dropdown with values showing allowed disc modes

   Function returns dropdown with items presenting allowed (and supported
   by cdw) modes of writing to current disc.
   Location of the dropdown in wizard window is specified by variables
   global in this file.

   The function creates all necessary dropdown labels and calls
   cdw_dropdown_finalize() on the dropdown, so result of this function
   is a ready to use dropdown.

   \param task - variable describing current task

   \return NULL on errors
   \return pointer to dropdown on success
*/
CDW_DROPDOWN *cdw_write_wizard_make_disc_mode_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	cdw_id_clabel_t items[CDW_DISC_MODE_N_MODES];
	int n_items = 0;
	for (n_items = 0; n_items < CDW_DISC_MODE_N_MODES; ) {
		/* disc_modes is non-sparse up until
		   element of value CDW_DISC_MODE_INIT */

		cdw_id_t allowed_mode_id = local_task->burn.disc_modes[n_items];
		if (allowed_mode_id == CDW_DISC_MODE_INIT) {
			break;
		} else {
			items[n_items].id = allowed_mode_id;
			items[n_items].label = cdw_utils_id_label_table_get_label(all_disc_modes, allowed_mode_id);
			n_items++;
		}
	}

	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, n_items, items);
	/* 0 - first mode (set by "task" module) is the default one */
	cdw_dropdown_set_current_item_by_ind(dd, 0);
	return dd;
}





/**
   \brief Create dropdown with values showing allowed write speeds

   Function returns dropdown with items presenting write speeds acceptable
   by current disc.
   Location of the dropdown in wizard window is specified by variables
   global in this file.

   The function creates all necessary dropdown labels and calls
   cdw_dropdown_finalize() on the dropdown, so result of this function
   is a ready to use dropdown.

   \param disc - variable describing current disc

   \return NULL on errors
   \return pointer to dropdown on success
*/
CDW_DROPDOWN *cdw_write_wizard_make_speed_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	CDW_DROPDOWN *dropdown = cdw_dropdown_new(window,
						  begin_y,
						  begin_x,
						  width,
						  local_disc->write_speeds.n_speeds,  /* n_items_max */
						  CDW_COLORS_DIALOG);

	if (dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create new dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}

	for (int i = 0; i < local_disc->write_speeds.n_speeds; i++) {
		char label[4 + 1];
		snprintf(label, 4 + 1, "%d", local_disc->write_speeds.speeds[i]);
		cdw_rv_t crv = cdw_dropdown_add_item(dropdown, local_disc->write_speeds.speeds[i], label);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to create dropdown label #%d of value %d\n", i, local_disc->write_speeds.speeds[i]);
			cdw_dropdown_delete(&dropdown);
			return (CDW_DROPDOWN *) NULL;
		}
	}

	cdw_rv_t crv = cdw_dropdown_finalize(dropdown);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to finalize dropdown\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	} else {
		cdw_dropdown_set_current_item_by_id(dropdown, local_task->burn.speed_id);
		return dropdown;
	}
}





/**
   \brief Deallocate all resources allocated by write wizard
*/
void cdw_write_wizard_destroy(void)
{
	if (wizard.cdw_form->subwindow != (WINDOW *) NULL) {
		delwin(wizard.cdw_form->subwindow);
		wizard.cdw_form->subwindow = (WINDOW *) NULL;
	}

	if (wizard.cdw_form->window != (WINDOW *) NULL) {
		delwin(wizard.cdw_form->window);
		wizard.cdw_form->window = (WINDOW *) NULL;
	}

	if (wizard.cdw_form != (cdw_form_t *) NULL) {
		cdw_form_delete_form_objects(wizard.cdw_form);
		cdw_form_delete(&(wizard.cdw_form));
	}

	if (wizard.header != (char *) NULL) {
		free(wizard.header);
		wizard.header = (char *) NULL;
	}

	return;
}





/**
   \brief Print to stderr values of \p task fields set in write wizard

   \param task - task variable, in which some fields were set in write wizard
*/
void cdw_write_wizard_debug_at_exit(cdw_task_t *task)
{
	/* use "task" variable to which wizard saves result at return,
	   not values from dropdowns/checkboxes */

	cdw_vdm ("INFO: disc mode dropdown is %svisible\n", wizard.disc_mode_dd_visible ? "" : "not ");
	cdw_vdm ("INFO: volume id field is %svisible\n", wizard.volume_id_visible ? "" : "not ");

	cdw_task_debug_print_burn_options(task);
	cdw_task_debug_print_iso9660_options(task);

	return;
}





bool cdw_write_wizard_show_verify_checkbox(cdw_task_t *task, cdw_disc_t *disc)
{
	bool disc_blank = disc->state_empty == CDW_TRUE;
	bool disc_overwritable = disc->type == CDW_DVD_RWP
		|| disc->type == CDW_DVD_RW_RES;
	bool tool_available = cdw_ext_tools_is_digest_tool(task->calculate_digest.tool.id);

	if ((disc_blank || disc_overwritable)
	    && task->id == CDW_TASK_BURN_FROM_IMAGE
	    && tool_available) {

		return true;
	} else {
		return false;
	}
}





/* for some combinations of burn tool and disc type
   dummy burning is just not possible */
bool cdw_write_wizard_show_dummy_checkbox(cdw_task_t *task, cdw_disc_t *disc)
{
	if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
		return cdw_cdrecord_allow_dummy(disc);
	} else if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {
		return cdw_growisofs_allow_dummy(disc);
	} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
		return cdw_xorriso_allow_dummy(disc);
	} else {
		return false;
	}
}





/**
   \brief Print to stderr information why "verify" checkbox is not visible

   If some conditions are met, write wizard may display "verify write"
   checkbox. This function should be called if any of the conditions is
   not met to investigate these conditions and print to stderr information
   which of them was not met. This is a debug function

   \param task - variable describing current task
   \param disc - variable describing current disc
*/
void cdw_write_wizard_debug_verify_invisible(cdw_task_t *task, cdw_disc_t *disc)
{
	if (disc->state_empty != CDW_TRUE) {
		cdw_vdm ("INFO: \"verify\" checkbox is not visible because disc is not blank (according to cdio)\n");
	}
	if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_DVD) {
		if (! (disc->type == CDW_DVD_RWP || disc->type == CDW_DVD_RW_RES)) {
			cdw_vdm ("INFO: \"verify\" checkbox is not visible because disc is not DVD+RW or DVD-RW RES\n");
		}
	}
	if (task->id != CDW_TASK_BURN_FROM_IMAGE) {
		cdw_vdm ("INFO: \"verify\" checkbox is not visible because task is not CDW_TASK_BURN_FROM_IMAGE\n");
	}
	if (!cdw_ext_tools_digest_tool_available()) {
		cdw_vdm ("INFO: \"verify\" checkbox is not visible because there is no digest tool available\n");
	}
	if (!cdw_ext_tools_is_digest_tool(task->calculate_digest.tool.id)) {
		cdw_vdm ("INFO: \"verify\" checkbox is not visible because digest tool is not set (is %lld)\n", task->calculate_digest.tool.id);
	}

	return;
}





/**
   \brief Create all UI elements in wizard window

   \param task - variable describing current task
   \param disc - variable describing disc currently in drive

   \return CDW_OK on success
   \return CDW_ERROR on failure
*/
cdw_rv_t cdw_write_wizard_build(cdw_task_t *task, cdw_disc_t *disc)
{
	int begin_y = ((LINES - window_n_lines) / 2);
	int begin_x = (COLS - window_n_cols) / 2;
	wizard.cdw_form->window = cdw_ncurses_window_new((WINDOW *) NULL,
							 window_n_lines, window_n_cols,
							 begin_y, begin_x,
							 CDW_COLORS_DIALOG,
							 /* 2TRANS: this is title of wizard window; 'write' as in
							    'writing to optical disc */
							 _("Write wizard"),
							 /* 2TRANS: this is tip at the bottom of window - user can
							    switch between window elements using tab key */
							 _("Use 'Tab' key to move"));
	if (wizard.cdw_form->window == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create window\n");
		return CDW_ERROR;
	}

	wizard.cdw_form->subwindow = cdw_ncurses_window_new(wizard.cdw_form->window,
							    subwindow_n_lines, subwindow_n_cols,
							    1, 1,
							    CDW_COLORS_DIALOG, (char *) NULL, (char *) NULL);

	if (wizard.cdw_form->subwindow == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create subwindow\n");
		return CDW_ERROR;
	}

	/* 2TRANS: this is message to user: current action will
	   be writing to %s, where %s is DVD or CD */
	int rv = asprintf(&(wizard.header), _("You are writing to %s disc"), disc->simple_type_label);
	if (rv == -1) {
		cdw_vdm ("ERROR: failed to create top message\n");
		return CDW_ERROR;
	}

	cdw_write_wizard_build_fields(task);

	wizard.cdw_form->form = cdw_ncurses_new_form(wizard.cdw_form->window,
						     wizard.cdw_form->subwindow,
						     wizard.cdw_form->fields);
	if (wizard.cdw_form->form == (FORM *) NULL) {
		cdw_vdm ("ERROR: failed to create form\n");
		return CDW_ERROR;
	}
	set_current_field(wizard.cdw_form->form, wizard.cdw_form->fields[f_cancel_i]);

	if (!wizard.volume_id_visible) {
		field_opts_off(wizard.cdw_form->fields[f_volume_id_l], O_VISIBLE);
		field_opts_off(wizard.cdw_form->fields[f_volume_id_i], O_VISIBLE);
	}

	CDW_CHECKBOX *cb = cdw_form_get_checkbox(wizard.cdw_form, f_verify_i);
	if (wizard.verify_checkbox_visible) { /* this last one is added just in case */
		cdw_checkbox_set_state(cb, task->burn.verify);
	} else {
		cdw_checkbox_set_visibility(cb, false);
		/* invisible AND inactive */
		field_opts_off(wizard.cdw_form->fields[f_verify_i], O_VISIBLE);
		field_opts_off(wizard.cdw_form->fields[f_verify_l], O_VISIBLE);

#ifndef NDEBUG
		cdw_write_wizard_debug_verify_invisible(task, disc);
#endif
	}

	cb = cdw_form_get_checkbox(wizard.cdw_form, f_dummy_i);
	if (wizard.dummy_checkbox_visible) {
		cdw_checkbox_set_state(cb, task->burn.dummy);
	} else {
		cdw_checkbox_set_visibility(cb, false);
		/* invisible AND inactive */
		field_opts_off(wizard.cdw_form->fields[f_dummy_i], O_VISIBLE);
		field_opts_off(wizard.cdw_form->fields[f_dummy_l], O_VISIBLE);

#ifndef NDEBUG
		// TODO: to be implemented: cdw_write_wizard_debug_dummy_invisible(task, disc);
#endif
	}
	if (!wizard.disc_mode_dd_visible) {
		CDW_DROPDOWN *dd = cdw_form_get_dropdown(wizard.cdw_form, f_disc_dd);
		cdw_dropdown_make_invisible(dd);
		field_opts_off(wizard.cdw_form->fields[f_disc_l], O_VISIBLE);
		field_opts_off(wizard.cdw_form->fields[f_disc_dd], O_VISIBLE);
	}

	/* current items are set in functions creating respective
	   dropdowns; now just display them */
	cdw_form_redraw_widgets(wizard.cdw_form);

	cdw_form_add_return_chars(wizard.cdw_form, CDW_KEY_ENTER, CDW_KEY_ESCAPE, 'q', 'Q', 0);

	cdw_form_set_function(wizard.cdw_form, f_write_i, cdw_write_wizard_on_enter_on_write);
	cdw_form_set_function(wizard.cdw_form, f_cancel_i, cdw_write_wizard_on_enter_on_cancel);
	cdw_form_set_function(wizard.cdw_form, f_other_options_b, cdw_write_wizard_on_enter_on_config);

	wrefresh(wizard.cdw_form->subwindow);
	wrefresh(wizard.cdw_form->window);

	return CDW_OK;
}





cdw_rv_t cdw_write_wizard_build_fields(cdw_task_t *task)
{
	char *other_options = _("Other options");
	int other_options_n_cols = (int) strlen(other_options) + 2;

	cdw_form_descr_t descr[] = {
		/*     type             begin_y                        begin_x         n_cols                n_lines    field enum      data1                  data2 */

		{ CDW_WIDGET_LABEL,      top_label_row,               first_col,      wide_label_n_cols,          1,    f_header_l,     wizard.header,                      0 },

		{ CDW_WIDGET_LABEL,      volume_id_label_row,         first_col,      wide_label_n_cols,          1,    f_volume_id_l,  _("Volume ID:"),                    0 },
		{ CDW_WIDGET_INPUT,      volume_id_label_row + 1,     first_col + 3,  CDW_ISO9660_VOLI_LEN,       1,    f_volume_id_i,  task->create_image.volume_id,  CDW_ISO9660_VOLI_LEN },

		/* 2TRANS: this is a label in write wizard, after which a writing speed selection list will be displayed */
		{ CDW_WIDGET_LABEL,      write_speed_label_row,       first_col,      wide_label_n_cols,          1,    f_speed_l,      _("Writing speed:"),                0 },
		{ CDW_WIDGET_DROPDOWN,   write_speed_label_row + 1,   first_col + 3,  narrow_dropdown_n_cols - 2, 1,    f_speed_dd,     dropdown_makers,                    0 },

		/* 2TRANS: this is a label in write wizard, after which a session writing mode selection list will be displayed */
		{ CDW_WIDGET_LABEL,      session_mode_label_row,      first_col,      wide_label_n_cols,          1,    f_session_l,    _("Session writing mode:"),         0 },
		{ CDW_WIDGET_DROPDOWN,   session_mode_label_row + 1,  first_col + 3,  wide_dropdown_n_cols - 2,   1,    f_session_dd,   dropdown_makers,                    1 },

		/* 2TRANS: this is a label in write wizard, after which a disc writing mode selection list will be displayed */
		{ CDW_WIDGET_LABEL,      disc_mode_label_row,         first_col,      wide_label_n_cols,          1,    f_disc_l,       _("Disc writing mode:"),            0 },
		{ CDW_WIDGET_DROPDOWN,   disc_mode_label_row + 1,     first_col + 3,  wide_dropdown_n_cols - 2,   1,    f_disc_dd,      dropdown_makers,                    2 },

		/* 2TRANS: this is label next to checkbox; marked checkbox enables verification of correctness of writing to CD; this feature is experimental */
		{ CDW_WIDGET_LABEL,      verify_checkbox_label_row,   first_col + 4,  narrow_label_n_cols,        1,    f_verify_l,     _("Verify write (experimental)"),   0 },
		{ CDW_WIDGET_CHECKBOX,   verify_checkbox_label_row,   first_col + 1,  1,                          1,    f_verify_i,     (void *) NULL,    task->burn.verify ? 1 : 0 },

		/* 2TRANS: this is checkbox label: keep no longer than original; "Dummy write" means attempting to write
		   to a disc and performing all normal operations (just for a try), but without actual burning */
		{ CDW_WIDGET_LABEL,      dummy_checkbox_label_row,    first_col + 4,  narrow_label_n_cols,        1,    f_dummy_l,    _("Dummy write"),                         0 },
		{ CDW_WIDGET_CHECKBOX,   dummy_checkbox_label_row,    first_col + 1,  1,                          1,    f_dummy_i,    (void *) NULL,     task->burn.dummy ? 1 : 0 },

		{ CDW_WIDGET_LABEL,      eject_checkbox_label_row,    first_col + 4,  narrow_label_n_cols,        1,    f_eject_l,    _("Eject tray after writing"),            0 },
		{ CDW_WIDGET_CHECKBOX,   eject_checkbox_label_row,    first_col + 1,  1,                          1,    f_eject_cb,   (void *) NULL,     task->burn.eject ? 1 : 0 },

		/* 2TRANS: button label */
		{ CDW_WIDGET_BUTTON,     other_options_button_row,    first_col,      other_options_n_cols,       1,    f_other_options_b,   other_options,  CDW_COLORS_DIALOG },

		/* 2TRANS: button label, it refers to writing to optical disc */
		{ CDW_WIDGET_BUTTON,     buttons_row,                 3,              2,                          1,    f_write_i,    _("Write"),          CDW_COLORS_DIALOG },
		/* 2TRANS: button label */
		{ CDW_WIDGET_BUTTON,     buttons_row,                 15,             2,                          1,    f_cancel_i,   _("Cancel"),         CDW_COLORS_DIALOG },

		/* guard */
		{ -1,                    0,                           0,              0,                          0,    0,            (void *) NULL,                       0 }};

	wizard.cdw_form->n_fields = CDW_WRITE_WIZARD_N_FIELDS;
	wizard.cdw_form->fields = wizard_fields;

	/* the function adds guard at the end of fields */
	cdw_rv_t crv = cdw_form_description_to_fields(descr, wizard.cdw_form);
	if (crv == CDW_OK) {
		return CDW_OK;
	} else {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	}
}





int cdw_write_wizard_on_enter_on_cancel(__attribute__((unused)) cdw_form_t *cdw_form, __attribute__((unused)) void *dummy)
{
	return CDW_KEY_ESCAPE;
}





int cdw_write_wizard_on_enter_on_write(__attribute__((unused)) cdw_form_t *cdw_form, __attribute__((unused)) void *dummy)
{
	return CDW_KEY_ENTER;
}





int cdw_write_wizard_on_enter_on_config(cdw_form_t *cdw_form, __attribute__((unused)) void *dummy)
{
	cdw_write_wizard_options(local_task);

	cdw_main_ui_main_window_wrefresh();
	redrawwin(cdw_form->window);
	redrawwin(cdw_form->subwindow);
	wrefresh(cdw_form->window);
	wrefresh(cdw_form->subwindow);

	return 'a';
}




/* some size and layout constants */
/* width and height of main options window can't be smaller
   than size of minimal supported terminal */

/** \brief Width of options window */
#define OPTIONS_WIDTH 80
/** \brief Height of options window */
#define OPTIONS_HEIGHT 24
/** \brief Width of right-hand area with tabs */
#define TABS_WIDTH 23
/** \brief Width of (most) labels in options window */
#define LABEL_WIDTH 30
/** \brief Width of some fields that should be as wide as possible: input
    fields storing file path, some dropdowns, "other options" fields,
    txt subwins, long labels */
#define WIDE_FIELD_WIDTH (OPTIONS_WIDTH - TABS_WIDTH - 4)
/** \brief Window column of first column of items in options page  */
#define FIRST_COL 1
/** \brief Window column of second column of items in options page  */
#define SECOND_COL (FIRST_COL + LABEL_WIDTH + 2)


/* text displayed at the bottom of wizard window */
static char *cdw_wizard_window_tip = (char *) NULL;


#define WRITE_CDRECORD        0
#define WRITE_GROWISOFS       1
#define WRITE_XORRISO_BURN    2
#define WRITE_MKISOFS         3
#define WRITE_XORRISO_ISO     4


/* this is first main component of options window: a big table
   aggregating data and functions that prepare the data; the data
   represents widgets displayed in pages of options window */
static cdw_config_page_t c_pages[] = {
	{ false, CDW_CDRECORD_OPTIONS_N_FIELDS,      cdw_cdrecord_options_form,        gettext_noop("Write"),   (cdw_form_t *) NULL },
	{ false, CDW_GROWISOFS_OPTIONS_N_FIELDS,     cdw_growisofs_options_form,       gettext_noop("Write"),   (cdw_form_t *) NULL },
	{ false, CDW_XORRISO_BURN_OPTIONS_N_FIELDS,  cdw_xorriso_burn_options_form,    gettext_noop("Write"),   (cdw_form_t *) NULL },
	{ false, CDW_MKISOFS_OPTIONS_N_FIELDS,       cdw_mkisofs_options_form,         gettext_noop("ISO9660"), (cdw_form_t *) NULL },
	{ false, CDW_XORRISO_ISO_OPTIONS_N_FIELDS,   cdw_xorriso_iso_options_form,     gettext_noop("ISO9660"), (cdw_form_t *) NULL }};


/* this is a pointer to second main component of options window:
   tabbed window, which enables switching between pages */
static cdw_tabs_window_t *c_twindow = (cdw_tabs_window_t *) NULL;




cdw_rv_t cdw_write_wizard_options(cdw_task_t *task)
{
	/* 2TRANS: this is message at the bottom of options window;
	   cancel means: quit without saving; '%d' is an integer used to
	   create label of function key, e.g. "F10" */
	int rv = asprintf(&cdw_wizard_window_tip, _("Press F%d key to save changes or ESC to cancel"), CDW_CONFIG_UI_SnC_KEY);
	if (cdw_wizard_window_tip == (char *) NULL || rv == -1) {
		cdw_vdm ("ERROR: failed to create window tip\n");
		return CDW_ERROR;
	}
	int n_tabs = 0;
	if (task->id == CDW_TASK_BURN_FROM_FILES) {
		n_tabs = 2; /* (cdrecord|growisofs|xorriso) + (mkisofs|xorriso) */
	} else { /* CDW_TASK_BURN_FROM_IMAGE */
		n_tabs = 1; /* (cdrecord|growisofs|xorriso) */
	}
	/* 2TRANS: this is title of cdw options main window */
	c_twindow = cdw_tabs_window_init(n_tabs, _("Write options"), cdw_wizard_window_tip);
	cdw_tabs_window_set_geometry(c_twindow, OPTIONS_HEIGHT, OPTIONS_WIDTH, 4, 4, TABS_WIDTH);


	/* prepare tabbed window */

	cdw_id_t id = 0;
	int ind = 0;

	/* tool-specific options: cdrecord or growisofs */
	if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
		id = WRITE_CDRECORD;
		cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
	} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
		id = WRITE_XORRISO_BURN;
		cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
	} else {
		id = WRITE_GROWISOFS;
		cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
	}
	/* options of tool creating ISO9660 file system;
	   the tool may be mkisofs or xorriso */
	if (task->id == CDW_TASK_BURN_FROM_FILES) {
		if (task->burn.tool.id == CDW_TOOL_CDRECORD || task->burn.tool.id == CDW_TOOL_GROWISOFS) {
			cdw_assert (task->create_image.tool.id == CDW_TOOL_MKISOFS, "ERROR: invalid ISO tool: %lld\n", task->create_image.tool.id);
			ind++;
			id = WRITE_MKISOFS;
			cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
		} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
			cdw_assert (task->create_image.tool.id == CDW_TOOL_XORRISO, "ERROR: invalid ISO tool: %lld\n", task->create_image.tool.id);
			ind++;
			id = WRITE_XORRISO_ISO;
			cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
		} else {
			cdw_assert (0, "ERROR: invalid burn tool: %lld\n", task->burn.tool.id);
		}
	}


	cdw_tabs_window_add_return_key(c_twindow, KEY_F(CDW_CONFIG_UI_SnC_KEY));
	cdw_tabs_window_add_return_key(c_twindow, CDW_KEY_ESCAPE);
	cdw_tabs_window_add_return_key(c_twindow, 'q');
	cdw_tabs_window_add_return_key(c_twindow, 'Q');
	c_twindow->driver_reader = cdw_config_ui_tabs_window_form_reader;
	cdw_tabs_window_finalize(c_twindow);

	/* prepare forms in the tabbed window */
	for (int i = 0; i < n_tabs; i++) {
		id = c_twindow->tabs[i].id;

		c_pages[id].cdw_form = cdw_form_new(c_pages[id].n_fields);
		c_pages[id].cdw_form->form_id = id;
		c_pages[id].cdw_form->window = c_twindow->tabs[i].window;
		c_pages[id].cdw_form->subwindow = c_twindow->tabs[i].subwindow;

		c_pages[id].visible = true;

		cdw_rv_t crv = c_pages[id].fields_builder(c_pages[id].cdw_form, task, FIRST_COL, SECOND_COL, WIDE_FIELD_WIDTH, LABEL_WIDTH);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to build fields in page #%d\n", i);
			return CDW_ERROR;
		}

		c_pages[id].cdw_form->form = cdw_ncurses_new_form(c_pages[id].cdw_form->window,
								  c_pages[id].cdw_form->subwindow,
								  c_pages[id].cdw_form->fields);

		for (int j = 0; j < n_tabs; j++) {
			cdw_form_add_return_char(c_pages[id].cdw_form, KEY_F(j + CDW_CONFIG_UI_FX_START_KEY));
		}
		cdw_form_add_return_char(c_pages[id].cdw_form, KEY_F(CDW_CONFIG_UI_SnC_KEY));
	}
	c_twindow->user_data = (void *) c_pages;

	cdw_tabs_window_draw_tabs(c_twindow);
	cdw_config_ui_tabs_window_form_draw(c_twindow, c_pages);

	bool follow_symlinks = task->create_image.follow_symlinks;
	cdw_write_wizard_options_window_driver(task);
	cdw_main_ui_handle_follow_symlinks_change(follow_symlinks, task->create_image.follow_symlinks);

	cdw_write_wizard_options_window_destroy();

	return CDW_OK;
}





/**
   \brief Destructor function for write options window
*/
void cdw_write_wizard_options_window_destroy(void)
{
	if (cdw_wizard_window_tip != (char *) NULL) {
		free(cdw_wizard_window_tip);
		cdw_wizard_window_tip = (char *) NULL;
	}

	for (int i = 0; i < c_twindow->n_tabs; i++) {
		cdw_id_t id = c_twindow->tabs[i].id;
		if (c_pages[id].cdw_form != (cdw_form_t *) NULL) {
			cdw_form_delete_form_objects(c_pages[id].cdw_form);
		}
		if (c_pages[id].cdw_form != (cdw_form_t *) NULL) {
			cdw_form_delete(&(c_pages[id].cdw_form));
		}
	}

	cdw_tabs_window_delete(&c_twindow);

	return;
}





/**
   \brief Function reacting to keys pressed when options window is displayed

   Function catches all keys pressed when options window is displayed,
   and reacts to them either by switching pages, or by passing the keys to
   page driver(s).

   Keys configured as hotkeys for "tabbed window" tabs are used to switch
   between pages. F(CDW_CONFIG_SnC_KEY) key is interpreted as "save and close"
   key - values of all option fields are stored in \p config and validated.
   Function then returns CDW_OK if validation is successful, or displays
   incorrect field to user.
   ESCAPE key causes the function to return with CDW_CANCEL.

   \param config - variable in which function saves values of all option fields

   \return CDW_KEY_ESCAPE if user pressed ESCAPE key in options window
   \return CDW_KEY_ENTER if user pressed F10 and validation of \p tmp config was successful
*/
int cdw_write_wizard_options_window_driver(cdw_task_t *task)
{
	while (1) {
		int key = cdw_tabs_window_driver(c_twindow);
		if (key == CDW_KEY_ESCAPE) {
			break;
		} else if (key == KEY_F(CDW_CONFIG_UI_SnC_KEY)) { /* SnC = Save and Close */

			/* flush */
			for (int i = 0; i < c_twindow->n_tabs; i++) {
				cdw_id_t id = c_twindow->tabs[i].id;
				form_driver(c_pages[id].cdw_form->form, REQ_VALIDATION);
			}
			cdw_id_t page_id = 0;
			int fi = 0;
			cdw_rv_t valid = cdw_write_wizard_options_validate_and_save(task, &page_id, &fi);
			if (valid == CDW_NO) { /* some option field is invalid */

				/* 2TRANS: this is title of dialog window */
				cdw_buttons_dialog(_("Error"),
						   /* 2TRANS: this is message in dialog window */
						   _("One of option fields is incorrect or contains character that is not allowed. Please fix it."),
						   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

				cdw_tabs_window_show_tab_by_id(c_twindow, page_id);
				cdw_form_driver_go_to_field(c_pages[page_id].cdw_form, fi);

				/* loop */
			} else {
				/* options from window with advanced
				   options are validated and saved into
				   task variable */
				return CDW_KEY_ENTER;
			}
		} else {
			/* loop */
		}
	} /* while (1) */

	return CDW_KEY_ESCAPE;
}





cdw_rv_t cdw_write_wizard_options_validate_and_save(cdw_task_t *task, cdw_id_t *page_id, int *fi)
{
	cdw_form_t *cdw_form = (cdw_form_t *) NULL;
	if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
		cdw_id_t id = WRITE_CDRECORD;
		cdw_form = c_pages[id].cdw_form;
		cdw_rv_t crv = cdw_cdrecord_options_validate(cdw_form, fi);
		if (crv == CDW_NO) {
			*page_id = id;
			return CDW_NO;
		}
	} else if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {
		cdw_id_t id = WRITE_GROWISOFS;
		cdw_form = c_pages[id].cdw_form;
		cdw_rv_t crv = cdw_growisofs_options_validate(cdw_form, fi);
		if (crv == CDW_NO) {
			*page_id = id;
			return CDW_NO;
		}
	} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
		cdw_id_t id = WRITE_XORRISO_BURN;
		cdw_form = c_pages[id].cdw_form;
		cdw_rv_t crv = cdw_xorriso_burn_options_validate(cdw_form, fi);
		if (crv == CDW_NO) {
			*page_id = id;
			return CDW_NO;
		}
	} else {
		cdw_assert (0, "ERROR: invalid tool id for task \"burn\": %lld\n",
			    task->burn.tool.id);
		return CDW_NO;
	}

	if (task->create_image.tool.id == CDW_TOOL_MKISOFS) {
		cdw_id_t id = WRITE_MKISOFS;
		cdw_form = c_pages[id].cdw_form;
		cdw_rv_t crv = cdw_mkisofs_options_validate(cdw_form, fi);
		if (crv == CDW_NO) {
			*page_id = id;
			return CDW_NO;
		}
	} else if (task->create_image.tool.id == CDW_TOOL_XORRISO) {
		cdw_id_t id = WRITE_XORRISO_ISO;
		cdw_form = c_pages[id].cdw_form;
		cdw_rv_t crv = cdw_xorriso_iso_options_validate(cdw_form, fi);
		if (crv == CDW_NO) {
			*page_id = id;
			return CDW_NO;
		}
	} else {
		;
	}

	if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
		cdw_form = c_pages[WRITE_CDRECORD].cdw_form;
		cdw_cdrecord_options_save(cdw_form, task);
	} else if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {
		cdw_form = c_pages[WRITE_GROWISOFS].cdw_form;
		cdw_growisofs_options_save(cdw_form, task);
	} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
		cdw_form = c_pages[WRITE_XORRISO_BURN].cdw_form;
		cdw_xorriso_burn_options_save(cdw_form, task);
	} else {
		;
	}

	if (task->create_image.tool.id == CDW_TOOL_MKISOFS) {
		cdw_form = c_pages[WRITE_MKISOFS].cdw_form;
		cdw_mkisofs_options_save(cdw_form, task);
	} else if (task->create_image.tool.id == CDW_TOOL_XORRISO) {
		cdw_form = c_pages[WRITE_XORRISO_ISO].cdw_form;
		cdw_xorriso_iso_options_save(cdw_form, task);
	} else {
		;
	}

	return CDW_OK;
}





/**
   \brief Validate options from main wizard window, and save them

   Function validates values of all options in wizard window (but not
   in "other options" window), and stores the values in \p task.

   Since there are no variables in the window that could be really
   validated, the function jumps right to saving values.
   Since the result of validation can't be negative, the function
   always returns CDW_OK.

   \param task - task variable to save values to
   \param fi - unused, field index

   return CDW_OK
*/
cdw_rv_t cdw_write_wizard_validate_and_save(cdw_task_t *task, __attribute__((unused)) int *fi)
{
	/* validate */
	const char *s = field_buffer(wizard.cdw_form->fields[f_volume_id_i], 0);
	cdw_rv_t crv = cdw_string_security_parser(s, (char *) NULL);
	if (crv == CDW_NO) {
		*fi = f_volume_id_i;
		return CDW_NO;
	}

	/* save */
	s = cdw_form_get_string(wizard.cdw_form, f_volume_id_i);
	strncpy(task->create_image.volume_id, s, CDW_ISO9660_VOLI_LEN);
	task->create_image.volume_id[CDW_ISO9660_VOLI_LEN] = '\0';

	if (wizard.verify_checkbox_visible) {
		task->burn.verify = cdw_form_get_checkbox_state(wizard.cdw_form, f_verify_i);
	} else {
		task->burn.verify = false;
	}
	if (wizard.dummy_checkbox_visible) {
		task->burn.dummy = cdw_form_get_checkbox_state(wizard.cdw_form, f_dummy_i);
	} else {
		task->burn.dummy = false;
	}
	task->burn.eject = cdw_form_get_checkbox_state(wizard.cdw_form, f_eject_cb);

	CDW_DROPDOWN *dd = cdw_form_get_dropdown(wizard.cdw_form, f_speed_dd);
	task->burn.speed_id = cdw_dropdown_get_current_item_id(dd);

	dd = cdw_form_get_dropdown(wizard.cdw_form, f_session_dd);
	task->burn.session_mode = cdw_dropdown_get_current_item_id(dd);

	if (wizard.disc_mode_dd_visible) {
		dd = cdw_form_get_dropdown(wizard.cdw_form, f_disc_dd);
		task->burn.disc_mode = cdw_dropdown_get_current_item_id(dd);
	}

	return CDW_OK;
}

