/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "config.h"
#include "cdw_config.h"
#include "gettext.h"
#include "cdw_logging.h"
#include "cdw_text_file_viewer.h"
#include "cdw_widgets.h"
#include "cdw_fs.h"
#include "cdw_file_picker.h"
#include "cdw_string.h"
#include "cdw_debug.h"
#include "cdw_utils.h"




/**
   \file cdw_logging.c
   \brief Implementation of logging for cdw

   File implements logging module: function that perform initialization of
   log file when application is started, functions that can be used to
   write to log file when application runs, function that closes log file
   and does necessary clean up before application exits.
*/




extern cdw_config_t global_config;

/** \brief Default name of application log file */
const char *cdw_log_file_name = "cdw.log";


/* *** definitions for logging module *** */

/** \brief Application's log file */
static FILE *log_file = (FILE *) NULL;

static bool full_path_silently_modified = false; /* was the path cleared (free()d set to NULL) */

static void cdw_logging_display_problem(int problem);

static cdw_rv_t cdw_logging_search_for_default_full_path(void);
static cdw_rv_t cdw_logging_ask_for_dir_path(int problem);

static cdw_rv_t cdw_logging_check_available_space(void);
static void cdw_logging_path_reset_message(void);

static void cdw_logging_handle_old_log_file(void);

/* constants used when dealing with problems when log file is initialized */
#define INVALID_PATH 1
#define NO_SPACE 2
#define EMPTY_PATH 3
#define GEN_ERROR 4





/**
   \brief Initialize logging module: find and open log file

   Try to open log file using file path set in configuration. If opening
   file with path set in configuration fails then try opening file with
   some default path. If all of above fails then ask user for path to
   given file.

   When correct full path to log file is known to this function, the
   function opens the file for reading and writing.

   If there is no place left on device for writing to the file, function
   closes old file and asks user for new path to log file.

   If after calling the function learns that current path is empty or invalid,
   it tries to silently set path to some default value. It does so silently,
   but in case of success user is informed about the fact of (re)setting
   the path to new value.

   If log file already existed then file is truncated to zero at opening.
   Correct full path to log file is stored in config.log_full_path

   \return CDW_OK if path is set correctly
   \return CDW_NO if user cancels entering new path or user didn't enter correct path in n tries
*/
cdw_rv_t cdw_logging_init(void)
{

	cdw_logging_handle_old_log_file();
	/*
	loop
		if (path correct) {
			open
			if (open = success)
				check space
				if (check space = success)
					return CDW_OK;
				else ask user (no space, new disc);
			else ask user (path invalid)
		else ask user (path invalid)
	end loop
	*/

	full_path_silently_modified = false;

	int i = 0;
	int n = 5;
	for (i = 0; i < n; i++) {
		int problem = 0;
		cdw_rv_t crv;
		if ((global_config.log_full_path != (char *) NULL) && strlen(global_config.log_full_path)) {
			log_file = fopen(global_config.log_full_path, "w+");
			if (log_file != (FILE *) NULL) {
				crv = cdw_logging_check_available_space();
				if (crv == CDW_OK) {
					if (full_path_silently_modified) {
						cdw_logging_path_reset_message();
						full_path_silently_modified = false;
					}
					rewind(log_file);
					return CDW_OK;
				} else {
					fclose(log_file);
					log_file = (FILE *) NULL;

					cdw_vdm ("no space available for this log file: \"%s\"\n", global_config.log_full_path);
					problem = NO_SPACE;
				}
			} else {
				cdw_vdm ("log file path is invalid: \"%s\"\n", global_config.log_full_path);
				problem = INVALID_PATH;
			}
		} else {
			cdw_vdm ("log file is null or empty: \"%s\"\n", global_config.log_full_path);
			problem = EMPTY_PATH;

		}

		if (i == n - 1) {
			cdw_logging_display_problem(problem);
			return CDW_ERROR;
		}

		if (problem == EMPTY_PATH
		    || (i == 0 && problem == INVALID_PATH)) {
			/* returns CDW_OK, CDW_NO, CDW_ERROR */
			crv = cdw_logging_search_for_default_full_path();
		} else {
			/* returns CDW_OK, CDW_CANCEL, CDW_MEM_ERROR */
			crv = cdw_logging_ask_for_dir_path(problem);
		}

		if (crv == CDW_OK) {
			full_path_silently_modified = true;
		} else if (crv == CDW_NO) {
			/* log_search_for_default_full_path() can't find
			   default path, ask user for path */
			problem = INVALID_PATH;
		} else if (crv == CDW_CANCEL) {
			/* user pressed Escape key when asked for path */
			return CDW_NO;
		} else { /* crv == CDW_ERROR */
			cdw_logging_display_problem(GEN_ERROR);
			return CDW_NO;
		}
	}

	return CDW_NO;
}





/**
 * \brief Deallocate all resources opened by logging module
 *
 * Close log file, clean up all other logging resources
 */
void cdw_logging_clean(void)
{
	if (log_file != (FILE *) NULL) {
		fflush(log_file);
		fclose(log_file);
		log_file = (FILE *) NULL;
	}

	/* note that config.log_full_path is free()d somewhere else */

	return;
}





/**
 * \brief Print message to log file
 *
 * printf()-like function that writes given format string and all
 * following arguments to log file.
 *
 * Function created to hide FILE *log_file from user and to provide
 * convenient way of writing to log.
 *
 * Function calls fflush() for file.
 *
 * \param format - format string of message that you want to print to log
 */
void cdw_logging_write(const char *format, ...)
{
	cdw_assert (log_file != (FILE *) NULL, "log file is not open when trying to write to it\n");

	va_list ap;
	va_start(ap, format);

	vfprintf(log_file, format, ap);
	fflush(log_file);

	va_end(ap);

	return;
}





/**
 * \brief Write some separator between parts of log file
 *
 * Current implementation of log file appends more and more data for
 * every task performed during one cdw session. This can make the log
 * file very long and hard to browse. Inserting some separator
 * between log file parts can make it easier to browse log file.
 */
void cdw_logging_write_separator(void)
{
	cdw_assert (log_file != (FILE *) NULL, "log file is not open when trying to write to it\n");

	fprintf(log_file, "\n\n");
	/* 2TRANS: this is header printed in log file;
	   first %s is program name, second %s is program version */
	fprintf(log_file, _("   ::: Log file for %s %s :::   "), PACKAGE, VERSION);
	fprintf(log_file, "\n\n\n");

	return;
}





/**
   \brief Search for valid full path to log file in some default locations

   Search for correct full path to log file in user's home directory, and
   if this fails, in tmp directory. The tmp directory is tmp_dir set
   in cdw_graftpoints.c. Therefore you have to call cdw_graftpoints_init()
   before initializing logging module.

   The function is non-interactive: user is not informed about new path
   to log file.

   Log file name is CDW_LOG_FILE_NAME ("cdw.log"). If function succeeds
   then full path to log file is copied to config.log_full_path. Otherwise
   config.log_full_path is set to NULL.

   The function does not try to open the file, it just tries to set
   config.log_full_path;

   \return CDW_ERROR if malloc() fails
   \return CDW_NO if function failed to set path to log file
   \return CDW_OK if path to file is set correctly.
*/
cdw_rv_t cdw_logging_search_for_default_full_path(void)
{
#if 0
	/* log file should be in user home dir by default */
	const char *dir_path = cdw_fs_get_home_dir_fullpath();
	if (dir_path == (char *) NULL) {
		dir_path = cdw_fs_get_tmp_dir_fullpath();
		/* log_init() should be called after
		   tmp dir is set, but let's check just in case */
		if (dir_path == (char *) NULL) {
			/* there was no HOME dir, and now it turned
			    out that there is no tmp dir :( */
			return CDW_NO;
		} else {
			; /* pass to concatenating full path */
		}

	} else {
		; /* pass to concatenating full path */
	}
#endif
	const char *dir = cdw_config_get_config_dir();
	char *tmp = cdw_string_concat(dir, cdw_log_file_name, (char *) NULL);
	if (tmp == (char *) NULL) {
		return CDW_ERROR;
	} else {
		if (global_config.log_full_path != (char *) NULL) {
			free(global_config.log_full_path);
			global_config.log_full_path = (char *) NULL;
		}
		global_config.log_full_path = tmp;
		return CDW_OK;
	}
}





/**
 * \brief Display information about resetting log file path
 *
 * This function displays dialog box informing about resetting full
 * path to log file in cdw configuration. It should be used when during
 * search for log file the log file path was reset from it's previous
 * to new value, or when the path was set form empty path to non-empty
 * path.
 *
 * Reason for using this function is that log file path is important
 * parameter and user must be informed about any changes.
 *
 * The function was created to avoid shifting code too much to the right side ;)
 */
void cdw_logging_path_reset_message(void)
{
	/* 2TRANS: this is title of dialog window */
	cdw_buttons_dialog(_("Message"),
			   /* 2TRANS: this is message in dialog window */
			   _("Path to CDW log file has been reset. You may want to check current path in Configuration -> Log."),
			   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);

}





/**
 * \brief Display dialog window in which user can enter path to
 * directory where log file will be put
 *
 * Display dialog box where user can enter new path to directory.
 *
 * If user entered new path, the function appends log file name to it
 * and puts result to config.log_full_path.
 * Otherwise config.log_full_path is not modified in any way.
 *
 * \param problem - identifier of problem, defined values are NO_SPACE, EMPTY_PATH, INVALID_PATH
 *
 * \return CDW_OK if user entered valid path
 * \return CDW_CANCEL if user canceled entering the path, or the path was invalid in n tries
 * \return CDW_ERROR on malloc errors
 */
cdw_rv_t cdw_logging_ask_for_dir_path(int problem)
{
	char *message = (char *) NULL;
	if (problem == NO_SPACE) {
		/* 2TRANS: this is message in dialog window */
		message = _("Enter path to directory on another device.");
	} else if (problem == EMPTY_PATH) {
		/* 2TRANS: this is message in dialog window */
		message = _("Enter non-empty path to initial (e.g. Home) directory.");
	} else { /* problem == INVALID_PATH */
		/* 2TRANS: this is message in dialog window */
		message = _("Enter valid path to directory to store cdw log file.");
	}
	char *buffer = (char *) NULL;
	/* 2TRANS: this is title of dialog window */
	cdw_rv_t crv = cdw_fs_ui_file_picker(_("Configuring cdw..."),
					     message,
					     &(buffer),
					     CDW_FS_DIR, R_OK | W_OK, CDW_FS_EXISTING);

	if (crv == CDW_OK) {
		char *tmp = cdw_string_concat(buffer, cdw_log_file_name, (char *) NULL);
		free(buffer);
		buffer = (char *) NULL;
		if (tmp == (char *) NULL) {
			return CDW_ERROR;
		} else {
			if (global_config.log_full_path != (char *) NULL) {
				free(global_config.log_full_path);
				global_config.log_full_path = (char *) NULL;
			}
			global_config.log_full_path = tmp;
			return CDW_OK;
		}
	} else {
		/* cdw_fs_ui_file_picker() does not allocate path if
		   user pressed ESCAPE, but it allocates memory for
		   empty string when user clears input field */
		if (buffer != (char *) NULL) {
			free(buffer);
			buffer = (char *) NULL;
		}
		return crv;
	}
}





/**
 * \brief Display message about current problem with initializing log path
 *
 * Display dialog box informing about specific problem with given path.
 *
 * \param problem - identifier of problem, defined values are NO_SPACE, EMPTY_PATH, INVALID_PATH
 */
void cdw_logging_display_problem(int problem)
{
	if (problem == NO_SPACE) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Too little space left on disc, can't write to log file."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
	} else if (problem == EMPTY_PATH) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Path to log file is empty, can't open file."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
	} else if (problem == INVALID_PATH) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Path to log file is invalid, can't open file"),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
	} else { /* problem == GEN_ERROR */
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("cdw can't configure its log file. Please restart the program."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
	}
}





/**
 * \brief Check if there is enough space on a device to store log file
 *
 * Try to write to log file some amount of data. If writing fails then return
 * error - caller will interpret this as not enough space on device.
 *
 * Global variable log_file must be valid (FILE *) pointer referring to
 * file open for writing.
 *
 * The function clears open file before returning.
 *
 * Current size of test data is 50k chars. When creating 2GB iso image
 * log file grows to ~7kB.
 *
 * \return CDW_ERROR on errors
 * \return CDW_OK if there is some space left on a device and using the file as log file is probably safe
 */
cdw_rv_t cdw_logging_check_available_space(void)
{
	bool failed = false;
	if (log_file == (FILE *) NULL) {
		return CDW_ERROR;
	}

	char buffer[1000];
	int i = 0;
	for (i = 0; i < 1000; i++) {
		buffer[i] = 'a';
	}
	buffer[i - 1] = '\0';

	for (i = 0; i < 50; i++) {
		int a = fprintf(log_file, "%s\n", buffer);
		int b = fflush(log_file);
		if (a < 0) {
			failed = true;
			break;
		}
		if (b == EOF) {
			failed = true;
			break;
		}
	}

	freopen(global_config.log_full_path, "w+", log_file); /* reset file size to zero */

	if (failed) {
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}






/**
 * \brief Display log file
 *
 * Wrapper using file_display() and local FILE *log_file variable
 * to simplify displaying log file: callers don't have to use
 * config.log_full_path anymore.
 *
 * \p title may be empty string or NULL.
 *
 * \param title - title of window in which log file will be displayed
 */
cdw_rv_t cdw_logging_display_log(const char *title)
{
	rewind(log_file);
	cdw_rv_t crv = cdw_text_file_viewer(log_file, title);
	fseek(log_file, 0L, SEEK_END);

	if (crv != CDW_OK) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Cannot show log file. Unknown error occurred."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Display log file if "show log after write" checkbox is checked

   Wrapper using file_display() and local FILE *log_file variable
   to simplify displaying log file: callers don't have to use
   config.log_full_path anymore. "conditional" means that the log
   is displayed only if "show log after write" checkbox in configuration
   window is checked.

   \p title may be empty string or NULL.

   \param title - title of window in which log file will be displayed
*/
cdw_rv_t cdw_logging_display_log_conditional(const char *title)
{
	if (global_config.showlog) {
		return cdw_logging_display_log(title);
	} else {
		return CDW_OK;
	}
}





/**
   \brief Remove log file from old location

   cdw is migrating from log file in old location: ~/.cdw.log,
   into new location: ~/.cdw/cdw.log. This function removes the log
   file from old location.
*/
void cdw_logging_handle_old_log_file(void)
{
	const char *dir = cdw_fs_get_home_dir_fullpath();
	if (dir == (char *) NULL) {
		return;
	}

	char *fullpath = cdw_string_concat(dir, ".cdw.log", (char *) NULL);
	if (fullpath == (char *) NULL) {
		cdw_vdm ("ERROR: failed to concatenate path\n");
		return;
	}

	if (access(fullpath, F_OK)) {
		free(fullpath);
		fullpath = (char *) NULL;
		return;
	}


	FILE *file = fopen(fullpath, "r");
	if (file == (FILE *) NULL) {
		cdw_vdm ("ERROR: failed to open file \"%s\"\n", fullpath);
		free(fullpath);
		fullpath = (char *) NULL;
		return;
	}
	bool found = false;
	for (int i = 0; i < 100; i++) {
		char *line = my_readline_10k(file);
		if (line != (char *) NULL) {
			char *ptr = strstr(line, "   ::: Log file for cdw");
			free(line);
			if (ptr != (char *) NULL) {
				found = true;
				break;
			}
		} else {
			/* EOF ? */
			break;
		}
	}

	fclose(file);
	file = (FILE *) NULL;
	if (!found) {
		; /* the file doesn't appear to be cdw log file, despite its name */
	} else {
		unlink(fullpath);
	}
	free(fullpath);
	fullpath = (char *) NULL;

	return;
}


