/* $Id: cpl_fft.c,v 1.5 2011/01/21 16:24:21 llundin Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


/*----------------------------------------------------------------------------
                                   Includes
 ----------------------------------------------------------------------------*/

#include <math.h>
#include <complex.h>

#include "cpl_tools.h"
#include "cpl_memory.h"
#include "cpl_error_impl.h"
#include "cpl_propertylist_impl.h"
#include "cpl_image_io_impl.h"
#include "cpl_mask.h"
#include "cpl_image_bpm.h"

#include "cpl_image_defs.h"
#include "cpl_imagelist_defs.h"

#include "cpl_fft.h"

#ifdef CPL_FFTWF_INSTALLED    /* If FFTW (single-precision) is installed */
#include <fftw3.h>
#endif

#ifdef CPL_FFTW_INSTALLED     /* If FFTW (normal-precision) is installed */
#include <fftw3.h>
#endif

/*---------------------------------------------------------------------------*/
/**
 * @defgroup cpl_fft FFTW wrappers
 *
 * This module provides FFTW wrappers
 *
 * @par Synopsis:
 * @code
 *   #include "cpl_fft.h"
 * @endcode
 */
/*---------------------------------------------------------------------------*/
/**@{*/


/*---------------------------------------------------------------------------*/
/**
  @brief    Perform a FFT operation on an image
  @param    out   Pre-allocated output image
  @param    in    Input image
  @param    mode  At present, either CPL_FFT_FORWARD or CPL_FFT_BACKWARD
  @return   CPL_ERROR_NONE or the corresponding #_cpl_error_code_
  
  This function performs a FFT operation on an image, using the FFTW library.
  If the CPL has not been configured providing this library, this function will
  return CPL_ERROR_UNSUPPORTED_MODE.

  When performing a forward FFT, the input image must be of a real type and the
  output image of a complex type, and viceversa when performing an inverse FFT.

  Both image types must match in precision level. Integer images are not 
  supported.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if any of the images is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the images types (real or complex) are not what
    it is expected for the type of operation (forward or inverse)
                            if image sizes do not match
  - CPL_ERROR_TYPE_MISMATCH if images are of different precision
  - CPL_ERROR_UNSUPPORTED_MODE if FFTW has not been installed

 */
/*---------------------------------------------------------------------------*/
cpl_error_code cpl_fft_image(cpl_image * out,
                 const cpl_image * in,
                 unsigned mode)
{
    cpl_type typin;
    cpl_type typout;

    int nxin, nyin, nxout, nyout;

    cpl_ensure_code(out != NULL && in != NULL, CPL_ERROR_NULL_INPUT);

    typin = cpl_image_get_type(in);
    typout = cpl_image_get_type(out);

    cpl_ensure_code((typin & CPL_TYPE_COMPLEX) ^ (typout & CPL_TYPE_COMPLEX),
            CPL_ERROR_ILLEGAL_INPUT);

    cpl_ensure_code(((typin & CPL_TYPE_FLOAT)  && (typout & CPL_TYPE_FLOAT)) ||
            ((typin & CPL_TYPE_DOUBLE) && (typout & CPL_TYPE_DOUBLE)),
            CPL_ERROR_TYPE_MISMATCH);

    cpl_ensure_code(((mode & CPL_FFT_FORWARD)  && (typout & CPL_TYPE_COMPLEX)) ||
            ((mode & CPL_FFT_BACKWARD) && (typin  & CPL_TYPE_COMPLEX)),
            CPL_ERROR_ILLEGAL_INPUT);

    nxin  = cpl_image_get_size_x(in);
    nyin  = cpl_image_get_size_y(in);
    nxout = cpl_image_get_size_x(out);
    nyout = cpl_image_get_size_y(out);

   cpl_ensure_code(nxin == nxout && nyin == nyout, CPL_ERROR_ILLEGAL_INPUT);

    if (typout & CPL_TYPE_FLOAT) {
    if (mode & CPL_FFT_FORWARD) {
#ifdef CPL_FFTWF_INSTALLED
        const float   * in_b  = cpl_image_get_data_float_const(in);
        fftwf_complex * out_b = (fftwf_complex *)cpl_image_get_data_float_complex(out);

        fftwf_plan          p = fftwf_plan_dft_r2c_2d(nxin, nyin,
                              (float *)in_b, out_b,
                              FFTW_ESTIMATE);

        fftwf_execute(p);

        fftwf_destroy_plan(p);
#else
        return cpl_error_set_(CPL_ERROR_UNSUPPORTED_MODE);
#endif
    } else {
#ifdef CPL_FFTWF_INSTALLED

        /*FIXME: FFTW destroys the input array in the inverse transform.
              The flag FFTW_PRESERVE_INPUT should solve this issue but it
              seems to produce a segfault. */ 
        cpl_image * dupl = cpl_image_duplicate(in);

        fftwf_complex * in_b  = (fftwf_complex *)cpl_image_get_data_float_complex(dupl);
        float         * out_b = cpl_image_get_data_float(out);

        fftwf_plan          p = fftwf_plan_dft_c2r_2d(nxin, nyin,
                              in_b, out_b,
                              FFTW_ESTIMATE);

        fftwf_execute(p);

        fftwf_destroy_plan(p);

        cpl_image_divide_scalar(out, nxin * nyin);
        cpl_image_delete(dupl);
#else
        return cpl_error_set_(CPL_ERROR_UNSUPPORTED_MODE);
#endif
    }
    } else {
    if (mode & CPL_FFT_FORWARD) {
#ifdef CPL_FFTW_INSTALLED
        const double * in_b  = cpl_image_get_data_double_const(in);
        fftw_complex * out_b = (fftw_complex *)cpl_image_get_data_double_complex(out);

        fftw_plan          p = fftw_plan_dft_r2c_2d(nxin, nyin,
                                    (double *)in_b, out_b,
                            FFTW_ESTIMATE);

        fftw_execute(p);

        fftw_destroy_plan(p);
#else
        return cpl_error_set_(CPL_ERROR_UNSUPPORTED_MODE);
#endif

    } else {
#ifdef CPL_FFTW_INSTALLED
        cpl_image * dupl = cpl_image_duplicate(in);
        fftw_complex * in_b  = (fftw_complex *)cpl_image_get_data_double_complex(dupl);
        double       * out_b = cpl_image_get_data_double(out);

        fftw_plan          p = fftw_plan_dft_c2r_2d(nxin, nyin,
                                    in_b, out_b,
                            FFTW_ESTIMATE);

        fftw_execute(p);

        fftw_destroy_plan(p);

        cpl_image_divide_scalar(out, nxin * nyin);
        cpl_image_delete(dupl);
#else
        return cpl_error_set_(CPL_ERROR_UNSUPPORTED_MODE);
#endif

    }

    }

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Perform a FFT operation on the images of an imagelist
  @param    out   Pre-allocated output imagelist
  @param    in    Input imagelist
  @param    mode  At present, either CPL_FFT_FORWARD or CPL_FFT_BACKWARD
  @return   CPL_ERROR_NONE or the corresponding #_cpl_error_code_
  @see cpl_fft_image()

 */
/*----------------------------------------------------------------------------*/
cpl_error_code cpl_fft_imagelist(cpl_imagelist * out, const cpl_imagelist * in,
                 unsigned mode)
{
    int i;
    int sizein, sizeout;

    cpl_ensure_code(out != NULL && in != NULL, CPL_ERROR_NULL_INPUT);
        
    sizein  = cpl_imagelist_get_size(in);
    sizeout = cpl_imagelist_get_size(out);

    cpl_ensure_code(sizein == sizeout, CPL_ERROR_ILLEGAL_INPUT);

    for (i = 0; i < sizein; i++)
        cpl_ensure_code(!cpl_fft_image(out->images[i], in->images[i], mode),
                         cpl_error_get_code());

    return cpl_error_get_code();
}

/**@}*/
