/*
  gflops: seccomp benchmark
  Copyright (C) 2005-2006  Andrea Arcangeli <andrea@cpushare.com>
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation;
  only version 2.1 of the License.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <signal.h>
#include <math.h>
#include <string.h>

#include <seccomp-loader.h>

#define barrier() __asm__ __volatile__("": : :"memory")

/* char x[400*1024] = "x"; */

char mode;
volatile int stop;

void sighandler(int signal)
{
	if (sys_write(1, "done", 4) != 4)
		sys_exit(3);
	switch (mode) {
	case 4:
	case 0:
		stop = 1;
		break;
	default:
		sys_exit(253);
	}
}

double __mflops(unsigned int * mflops_end)
{
	double data[8][3];
	double a0, a1, a2, a3, a4, a5, a6, a7;
	double b0, b1, b2, b3, b4, b5, b6, b7;
	double c0, c1, c2, c3, c4, c5, c6, c7;
	char z;
	int x;
	unsigned int mflops_stop, mflops;
	double inc = 0;

	for (x = 0; x < sizeof(data)/sizeof(double); x++) {
		data[0][x] = 1. + inc;
		inc += 0.000000001;
	}

	if (sys_read(0, &z, 1) != 1)
		sys_exit(3);

	mflops_stop = ~0U;
	if (z == 1) {
		/*
		 * CPUShare servers will send the right endianess
		 * depending on the architecture.
		 */
		if (sys_read(0, (char *) &data, sizeof(data)) != sizeof(data))
			sys_exit(3);
		if (sys_read(0, (char *) &mflops_stop,
			     sizeof(unsigned int)) != sizeof(unsigned int))
			sys_exit(3);
	}

	a0 = data[0][0];
	a1 = data[1][0];
	a2 = data[2][0];
	a3 = data[3][0];
	a4 = data[4][0];
	a5 = data[5][0];
	a6 = data[6][0];
	a7 = data[7][0];

	b0 = data[0][2];
	b1 = data[1][2];
	b2 = data[2][2];
	b3 = data[3][2];
	b4 = data[4][2];
	b5 = data[5][2];
	b6 = data[6][2];
	b7 = data[7][2];

	c0 = data[0][1];
	c1 = data[1][1];
	c2 = data[2][1];
	c3 = data[3][1];
	c4 = data[4][1];
	c5 = data[5][1];
	c6 = data[6][1];
	c7 = data[7][1];

	barrier();

	for (mflops = 0; mflops < mflops_stop && !stop; mflops++) {
		for (x = 0; x < 1000; x++) {
			a0 *= b0;
			a1 *= b1;
			a2 *= b2;
			a3 *= b3;
			a4 *= b4;
			a5 *= b5;
			a6 *= b6;
			a7 *= b7;

			a0 /= c0;
			a1 /= c1;
			a2 /= c2;
			a3 /= c3;
			a4 /= c4;
			a5 /= c5;
			a6 /= c6;
			a7 /= c7;
		}
	}

	barrier();

	data[0][0] = a0;
	data[1][0] = a1;
	data[2][0] = a2;
	data[3][0] = a3;
	data[4][0] = a4;
	data[5][0] = a5;
	data[6][0] = a6;
	data[7][0] = a7;

	*mflops_end = mflops;

	inc = 0;
	for (x = 0; x < 8; x++)
		inc += data[x][0];
	return inc;
}

void mflops(unsigned char * mem, int heap_size, int stack_size)
{
	double ret;
	char z;
	unsigned int mflops;

	memset(mem, 0, heap_size);

	if (sys_write(1, &z, 1) != 1)
		sys_exit(3);

	if (sys_read(0, &z, 1) != 1)
		sys_exit(3);

	ret = __mflops(&mflops);

	/* barrier(); */ /* not needed with strict aliasing on */

	z += 1;
	if (sys_write(1, &z, 1) != 1)
		sys_exit(4);
	if (sys_write(1, (char *) &mflops, sizeof(unsigned int)) != sizeof(unsigned int))
		sys_exit(4);
	if (sys_write(1, (char *) &ret, sizeof(double)) != sizeof(double))
		sys_exit(4);

#if 0 /* debug pipe write throttling */
	for (;;) {
		char a[1024];
		sys_write(1, a, 1024);
	}
#endif

	sys_exit(0);
	*(volatile int *)0 = ret;
}

unsigned int __mips(unsigned int * mips_end)
{
	unsigned int data[8][3];
	unsigned int a0, b0, c0;
	unsigned int a1, b1, c1;
	unsigned int a2, b2, c2;
	unsigned int a3, b3, c3;
	unsigned int a4, b4, c4;
	unsigned int a5, b5, c5;
	unsigned int a6, b6, c6;
	unsigned int a7, b7, c7;
	unsigned int inc = 5000;
	char z;
	int x;
	unsigned int mips_stop, mips;

	for (x = 0; x < sizeof(data)/sizeof(unsigned int); x++) {
		data[0][x] = inc;
		inc += 10000;
	}

	if (sys_read(0, &z, 1) != 1)
		sys_exit(3);

	mips_stop = ~0U;
	if (z == 1) {
		/*
		 * CPUShare servers will send the right endianess
		 * depending on the architecture.
		 */
		if (sys_read(0, (char *) &data, sizeof(data)) != sizeof(data))
			sys_exit(3);
		if (sys_read(0, (char *) &mips_stop,
			     sizeof(unsigned int)) != sizeof(unsigned int))
			sys_exit(3);
	}

	a0 = data[0][0];
	a1 = data[1][0];
	a2 = data[2][0];
	a3 = data[3][0];
	a4 = data[4][0];
	a5 = data[5][0];
	a6 = data[6][0];
	a7 = data[7][0];

	b0 = data[0][2];
	b1 = data[1][2];
	b2 = data[2][2];
	b3 = data[3][2];
	b4 = data[4][2];
	b5 = data[5][2];
	b6 = data[6][2];
	b7 = data[7][2];

	c0 = data[0][1];
	c1 = data[1][1];
	c2 = data[2][1];
	c3 = data[3][1];
	c4 = data[4][1];
	c5 = data[5][1];
	c6 = data[6][1];
	c7 = data[7][1];

	barrier();

	for (mips = 0; mips < mips_stop && !stop; mips++) {
		for (x = 0; x < 1000; x++) {
			a0 *= b0;
			a1 *= b1;
			a2 *= b2;
			a3 *= b3;
			a4 *= b4;
			a5 *= b5;
			a6 *= b6;
			a7 *= b7;

			a0 /= c0;
			a1 /= c1;
			a2 /= c2;
			a3 /= c3;
			a4 /= c4;
			a5 /= c5;
			a6 /= c6;
			a7 /= c7;
		}
	}

	barrier();

	data[0][0] = a0;
	data[1][0] = a1;
	data[2][0] = a2;
	data[3][0] = a3;
	data[4][0] = a4;
	data[5][0] = a5;
	data[6][0] = a6;
	data[7][0] = a7;

	*mips_end = mips;

	inc = 0;
	for (x = 0; x < 8; x++)
		inc += data[x][0];
	return inc;
}

void mips(unsigned char * mem, int heap_size, int stack_size)
{
	unsigned int ret;
	char z;
	unsigned int mips;

	memset(mem, 0, heap_size);

	if (sys_write(1, &z, 1) != 1)
		sys_exit(3);

	if (sys_read(0, &z, 1) != 1)
		sys_exit(3);

	ret = __mips(&mips);

	z += 1;
	if (sys_write(1, &z, 1) != 1)
		sys_exit(4);
	if (sys_write(1, (char *) &mips, sizeof(unsigned int)) != sizeof(unsigned int))
		sys_exit(4);
	if (sys_write(1, (char *) &ret, sizeof(unsigned int)) != sizeof(unsigned int))
		sys_exit(4);

#if 0 /* debug pipe write throttling */
	for (;;) {
		char a[1024];
		sys_write(1, a, 1024);
	}
#endif

	sys_exit(0);
	*(volatile int *)0 = ret;
}

void getpid(void)
{
	sys_getpid();
}

void sigsegv(void)
{
	*(volatile int *)0 = 0;
}

void rtt(void)
{
	char c;
	if (sys_read(0, &c, 1) != 1)
		sys_exit(100);
	if (sys_write(1, &c, 1) != 1)
		sys_exit(101);
}

void bytecode(unsigned char * mem, int heap_size, int stack_size)
{
	char c;
	for (;;) {
		if (sys_read(0, &c, 1) != 1)
			sys_exit(-5);
		if (sys_write(1, &c, 1) != 1)
			sys_exit(-6);
	}
#if 0
	/* flooding output */
	for (;;)
		sys_write(1, &mode, 1);
#endif
#if 0
	/* flooding input */
	for (;;);
#endif

	if (sys_read(0, &mode, 1) != 1)
		sys_exit(255);

	switch (mode) {
	case 0:
		mflops(mem, heap_size, stack_size);
		break;
	case 4:
		mips(mem, heap_size, stack_size);
		break;
	case 1:
		getpid();
		break;
	case 2:
		sigsegv();
		break;
	case 3:
		rtt();
		break;
	default:
		sys_exit(254);
	}
}
