# legends.rb: command-line parsing for legends
# copyright (c) 2008 by Vincent Fourmond: 
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'CTioga/log'
require 'CTioga/debug'
require 'CTioga/curve_style'
require 'CTioga/dimension'

module CTioga

  Version::register_svn_info('$Revision: 870 $', '$Date: 2009-01-20 14:28:32 +0100 (Tue, 20 Jan 2009) $')

  # A module to deal with legends...
  module Legends

    # Parse the legend positions
    def self.legend_position(spec)
      specs = {}
      begin
        margins = Utils::inset_margins(spec)
        specs = margins.to_frame("legend_%s_margin")
      rescue                  # in case of an incorrect margin
        # Then we look for a position in the spirit of:
        if spec =~ /\s*([tbc])([rlc])\s*(?::\s*([\d.eE+-]+),([\d.eE+-]+))?\s*$/
          if($3 && $4)
            x = Float($3)
            y = Float($4)
          else
            x = y = nil
          end
          if $1 == 't'
            specs['legend_top'] = y || 0.95 
          elsif $1 == 'b'
            specs['legend_bottom'] = y || 0.05 
          else
            specs['legend_vcenter'] = y || 0.5
          end
          if $2 == 'l'
            specs['legend_left'] = x || 0.05
          elsif $2 == 'r'
            specs['legend_right'] = x || 0.95
          else
            specs['legend_hcenter'] = x || 0.5
          end
        else
          raise "Incorrect legend (inside) position: '#{spec}'"
        end
      end
      return specs
    end

    # Fills up an OptionParser, _parser_ with legend-related command-line
    # options. Takes a Plotmaker _target_
    def self.fill_option_parser(parser, target)

      parser.separator "\nLegends:"
      
      parser.on("-l","--[no-]legend [LEGEND]",
                 "Sets the legend of the next curve.") do |l|
        target.override_style.legend = l
      end

      parser.on("--[no-]auto-legend", 
                 "Whether to automatically add legends",
                 "to curves") do |l|
        target.autolegends = l
      end

      parser.on('-N', "--no-legends", 
                 "Alias for --no-auto-legend") do
        target.autolegends = false
      end

      ## Legend scale
      parser.on("--legend-scale FACT",
                 "Sets the scale for the legends") do |l|
        target.add_elem_funcall(:legend_scale=, Float(l))
      end

      parser.on("--legend-line TEXT",
                 "Adds the line TEXT in the legend") do |l|
        target.current_object.add_legend_info(LegendLine.new(l))
      end


      parser.on("--legend-dy DIM",
                 "The distance between two legend lines, ",
                 "expressed in terms of text line size") do |l|
        target.current_object.plot_style.legend_style.dy = 
          TextDimension.new(l)
      end

      parser.on("--legend-pos SPEC", 
                 "Sets the legend position and its size.",
                 "SPEC looks like pos,size[,delta]") do |a|
        w,s,d = a.split /\s*,\s*/
        if d
          warn "The delta specification (#{d}) for #{a} is ignored"
        end
        target.debug "Legend position #{w.inspect} #{s.inspect}"
        target.current_object.layout_preferences.legend_position = w.to_sym
        target.current_object.layout_preferences.legend_size = Dimension.new(s)
      end

      parser.on("--legend-inside SPEC", 
                 "Makes a legend inside the plot.",
                 "See --inset for the format of SPEC") do |a|
        specs = legend_position(a)
        target.debug "Specs: #{specs.inspect}"
        target.current_object.layout_preferences.legend_position = :inside
        target.current_object.layout_preferences.legend_spec = specs
      end

      # Frames around legends:
      parser.on("--[no-]legend-frame [WHAT]",
                "Which frame should ctioga draw around legends ?") do |l|
        if l =~ /square|round/i
          l = l.downcase.to_sym
        elsif l =~ /none/i || (!l)
          l = false
        else
          warn "Incorrect legend frame: '#{l}'"
        end
        target.current_object.plot_style.legend_style.box_style = l
      end

      # TODO: make all the legend styles...
      parser.on("--[no-]legend-background [COLOR]",
                "Background color for the legend") do |c|
        c = CTioga.get_tioga_color(c) if c
        target.current_object.plot_style.legend_style.background_color = c
      end

      parser.on("--[no-]legend-transparency [VALUE]",
                "Transparency for the background") do |f|
        if !f 
          f = 0.0
        else
          f = f.to_f
        end
        target.current_object.plot_style.legend_style.
          background_transparency = f
      end

      parser.on("--[no-]legend-color [COLOR]",
                "Color for the line around the legend") do |c|
        c = CTioga.get_tioga_color(c) if c
        target.current_object.plot_style.legend_style.line_color = c
      end

      parser.on("--legend-line-width WIDTH",
                "Line width for drawing the legend") do |c|
        c = Float(c)
        target.current_object.plot_style.legend_style.line_width = c
      end

      parser.on("--legend-line-style STYLE",
                "Line style for drawing the legend") do |c|
        c = MetaBuilder::ParameterType.from_string(:line_style,c)
        target.current_object.plot_style.legend_style.line_style = c
      end
      
      parser.on("--[no-]separate-legends",
                "If set, PDF files are produced that contain",
                "what the legend pictogram would look like") do |l|
        target.separate_legends = l
        if l
          # Switches off autolegends by default
          target.autolegends = false
        end
      end

    end

  end

end
