#include <assert.h>
#include <errno.h>
#include <getopt.h>
#include <limits.h>
#include <signal.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <unistd.h>
#include <map>
#include <set>
#include <string>
#include <utility>
#include <vector>

#include "acceptor.h"
#include "accesslog.h"
#include "config.h"
#include "input.h"
#include "input_stats.h"
#include "log.h"
#include "sa_compare.h"
#include "serverpool.h"
#include "state.pb.h"
#include "stats.h"
#include "stream.h"
#include "util.h"
#include "version.h"

using namespace std;

AccessLogThread *access_log = NULL;
ServerPool *servers = NULL;
volatile bool hupped = false;
volatile bool stopped = false;

struct InputWithRefcount {
	Input *input;
	int refcount;
};

void hup(int signum)
{
	hupped = true;
	if (signum == SIGINT) {
		stopped = true;
	}
}

void do_nothing(int signum)
{
}

CubemapStateProto collect_state(const timeval &serialize_start,
                                const vector<Acceptor *> acceptors,
                                const multimap<string, InputWithRefcount> inputs,
                                ServerPool *servers)
{
	CubemapStateProto state = servers->serialize();  // Fills streams() and clients().
	state.set_serialize_start_sec(serialize_start.tv_sec);
	state.set_serialize_start_usec(serialize_start.tv_usec);
	
	for (size_t i = 0; i < acceptors.size(); ++i) {
		state.add_acceptors()->MergeFrom(acceptors[i]->serialize());
	}

	for (multimap<string, InputWithRefcount>::const_iterator input_it = inputs.begin();
	     input_it != inputs.end();
	     ++input_it) {
		state.add_inputs()->MergeFrom(input_it->second.input->serialize());
	}

	return state;
}

// Find all port statements in the configuration file, and create acceptors for htem.
vector<Acceptor *> create_acceptors(
	const Config &config,
	map<sockaddr_in6, Acceptor *, Sockaddr6Compare> *deserialized_acceptors)
{
	vector<Acceptor *> acceptors;
	for (unsigned i = 0; i < config.acceptors.size(); ++i) {
		const AcceptorConfig &acceptor_config = config.acceptors[i];
		Acceptor *acceptor = NULL;
		map<sockaddr_in6, Acceptor *, Sockaddr6Compare>::iterator deserialized_acceptor_it =
			deserialized_acceptors->find(acceptor_config.addr);
		if (deserialized_acceptor_it != deserialized_acceptors->end()) {
			acceptor = deserialized_acceptor_it->second;
			deserialized_acceptors->erase(deserialized_acceptor_it);
		} else {
			int server_sock = create_server_socket(acceptor_config.addr, TCP_SOCKET);
			acceptor = new Acceptor(server_sock, acceptor_config.addr);
		}
		acceptor->run();
		acceptors.push_back(acceptor);
	}

	// Close all acceptors that are no longer in the configuration file.
	for (map<sockaddr_in6, Acceptor *, Sockaddr6Compare>::iterator
	         acceptor_it = deserialized_acceptors->begin();
	     acceptor_it != deserialized_acceptors->end();
	     ++acceptor_it) {
		acceptor_it->second->close_socket();
		delete acceptor_it->second;
	}

	return acceptors;
}

void create_config_input(const string &src, multimap<string, InputWithRefcount> *inputs)
{
	if (src.empty()) {
		return;
	}
	if (inputs->count(src) != 0) {
		return;
	}

	InputWithRefcount iwr;
	iwr.input = create_input(src);
	if (iwr.input == NULL) {
		log(ERROR, "did not understand URL '%s', clients will not get any data.",
			src.c_str());
		return;
	}
	iwr.refcount = 0;
	inputs->insert(make_pair(src, iwr));
}

// Find all streams in the configuration file, and create inputs for them.
void create_config_inputs(const Config &config, multimap<string, InputWithRefcount> *inputs)
{
	for (unsigned i = 0; i < config.streams.size(); ++i) {
		const StreamConfig &stream_config = config.streams[i];
		if (stream_config.src != "delete") {
			create_config_input(stream_config.src, inputs);
		}
	}
	for (unsigned i = 0; i < config.udpstreams.size(); ++i) {
		const UDPStreamConfig &udpstream_config = config.udpstreams[i];
		create_config_input(udpstream_config.src, inputs);
	}
}

void create_streams(const Config &config,
                    const set<string> &deserialized_urls,
                    multimap<string, InputWithRefcount> *inputs)
{
	// HTTP streams.
	set<string> expecting_urls = deserialized_urls;
	for (unsigned i = 0; i < config.streams.size(); ++i) {
		const StreamConfig &stream_config = config.streams[i];
		int stream_index;

		expecting_urls.erase(stream_config.url);

		// Special-case deleted streams; they were never deserialized in the first place,
		// so just ignore them.
		if (stream_config.src == "delete") {
			continue;
		}

		if (deserialized_urls.count(stream_config.url) == 0) {
			stream_index = servers->add_stream(stream_config.url,
			                                   stream_config.backlog_size,
			                                   Stream::Encoding(stream_config.encoding));
		} else {
			stream_index = servers->lookup_stream_by_url(stream_config.url);
			assert(stream_index != -1);
			servers->set_backlog_size(stream_index, stream_config.backlog_size);
			servers->set_encoding(stream_index,
			                      Stream::Encoding(stream_config.encoding));
		}

		servers->set_pacing_rate(stream_index, stream_config.pacing_rate);

		string src = stream_config.src;
		if (!src.empty()) {
			multimap<string, InputWithRefcount>::iterator input_it = inputs->find(src);
			if (input_it != inputs->end()) {
				input_it->second.input->add_destination(stream_index);
				++input_it->second.refcount;
			}
		}
	}

	// Warn about any streams servers we've lost.
	for (set<string>::const_iterator stream_it = expecting_urls.begin();
	     stream_it != expecting_urls.end();
	     ++stream_it) {
		string url = *stream_it;
		log(WARNING, "stream '%s' disappeared from the configuration file. "
		             "It will not be deleted, but clients will not get any new inputs. "
		             "If you really meant to delete it, set src=delete and reload.",
		             url.c_str());
	}

	// UDP streams.
	for (unsigned i = 0; i < config.udpstreams.size(); ++i) {
		const UDPStreamConfig &udpstream_config = config.udpstreams[i];
		int stream_index = servers->add_udpstream(
			udpstream_config.dst,
			udpstream_config.pacing_rate,
			udpstream_config.ttl,
			udpstream_config.multicast_iface_index);

		string src = udpstream_config.src;
		if (!src.empty()) {
			multimap<string, InputWithRefcount>::iterator input_it = inputs->find(src);
			assert(input_it != inputs->end());
			input_it->second.input->add_destination(stream_index);
			++input_it->second.refcount;
		}
	}
}
	
void open_logs(const vector<LogConfig> &log_destinations)
{
	for (size_t i = 0; i < log_destinations.size(); ++i) {
		if (log_destinations[i].type == LogConfig::LOG_TYPE_FILE) {
			add_log_destination_file(log_destinations[i].filename);
		} else if (log_destinations[i].type == LogConfig::LOG_TYPE_CONSOLE) {
			add_log_destination_console();
		} else if (log_destinations[i].type == LogConfig::LOG_TYPE_SYSLOG) {
			add_log_destination_syslog();
		} else {
			assert(false);
		}
	}
	start_logging();
}
	
bool dry_run_config(const std::string &argv0, const std::string &config_filename)
{
	char *argv0_copy = strdup(argv0.c_str());
	char *config_filename_copy = strdup(config_filename.c_str());

	pid_t pid = fork();
	switch (pid) {
	case -1:
		log_perror("fork()");
		free(argv0_copy);
		free(config_filename_copy);
		return false;
	case 0:
		// Child.
		execlp(argv0_copy, argv0_copy, "--test-config", config_filename_copy, NULL);
		log_perror(argv0_copy);
		_exit(1);
	default:
		// Parent.
		break;
	}
		
	free(argv0_copy);
	free(config_filename_copy);

	int status;
	pid_t err;
	do {
		err = waitpid(pid, &status, 0);
	} while (err == -1 && errno == EINTR);

	if (err == -1) {
		log_perror("waitpid()");
		return false;
	}	

	return (WIFEXITED(status) && WEXITSTATUS(status) == 0);
}

void find_deleted_streams(const Config &config, set<string> *deleted_urls)
{
	for (unsigned i = 0; i < config.streams.size(); ++i) {
		const StreamConfig &stream_config = config.streams[i];
		if (stream_config.src == "delete") {
			log(INFO, "Deleting stream '%s'.", stream_config.url.c_str());
			deleted_urls->insert(stream_config.url);
		}
	}
}

int main(int argc, char **argv)
{
	signal(SIGHUP, hup);
	signal(SIGINT, hup);
	signal(SIGUSR1, do_nothing);  // Used in internal signalling.
	signal(SIGPIPE, SIG_IGN);
	
	// Parse options.
	int state_fd = -1;
	bool test_config = false;
	for ( ;; ) {
		static const option long_options[] = {
			{ "state", required_argument, 0, 's' },
			{ "test-config", no_argument, 0, 't' },
			{ 0, 0, 0, 0 }
		};
		int option_index = 0;
		int c = getopt_long(argc, argv, "s:t", long_options, &option_index);
     
		if (c == -1) {
			break;
		}
		switch (c) {
		case 's':
			state_fd = atoi(optarg);
			break;
		case 't':
			test_config = true;
			break;
		default:
			fprintf(stderr, "Unknown option '%s'\n", argv[option_index]);
			exit(1);
		}
	}

	string config_filename = "cubemap.config";
	if (optind < argc) {
		config_filename = argv[optind++];
	}

	// Canonicalize argv[0] and config_filename.
	char argv0_canon[PATH_MAX];
	char config_filename_canon[PATH_MAX];

	if (realpath("/proc/self/exe", argv0_canon) == NULL) {
		log_perror(argv[0]);
		exit(1);
	}
	if (realpath(config_filename.c_str(), config_filename_canon) == NULL) {
		log_perror(config_filename.c_str());
		exit(1);
	}

	// Now parse the configuration file.
	Config config;
	if (!parse_config(config_filename_canon, &config)) {
		exit(1);
	}
	if (test_config) {
		exit(0);
	}
	
	// Ideally we'd like to daemonize only when we've started up all threads etc.,
	// but daemon() forks, which is not good in multithreaded software, so we'll
	// have to do it here.
	if (config.daemonize) {
		if (daemon(0, 0) == -1) {
			log_perror("daemon");
			exit(1);
		}
	}

start:
	// Open logs as soon as possible.
	open_logs(config.log_destinations);

	log(INFO, "Cubemap " SERVER_VERSION " starting.");
	if (config.access_log_file.empty()) {
		// Create a dummy logger.
		access_log = new AccessLogThread();
	} else {
		access_log = new AccessLogThread(config.access_log_file);
	}
	access_log->run();

	servers = new ServerPool(config.num_servers);

	// Find all the streams that are to be deleted.
	set<string> deleted_urls;
	find_deleted_streams(config, &deleted_urls);

	CubemapStateProto loaded_state;
	struct timeval serialize_start;
	set<string> deserialized_urls;
	map<sockaddr_in6, Acceptor *, Sockaddr6Compare> deserialized_acceptors;
	multimap<string, InputWithRefcount> inputs;  // multimap due to older versions without deduplication.
	if (state_fd != -1) {
		log(INFO, "Deserializing state from previous process...");
		string serialized;
		if (!read_tempfile_and_close(state_fd, &serialized)) {
			exit(1);
		}
		if (!loaded_state.ParseFromString(serialized)) {
			log(ERROR, "Failed deserialization of state.");
			exit(1);
		}

		serialize_start.tv_sec = loaded_state.serialize_start_sec();
		serialize_start.tv_usec = loaded_state.serialize_start_usec();

		// Deserialize the streams.
		map<string, string> stream_headers_for_url;  // See below.
		for (int i = 0; i < loaded_state.streams_size(); ++i) {
			const StreamProto &stream = loaded_state.streams(i);

			if (deleted_urls.count(stream.url()) != 0) {
				// Delete the stream backlogs.
				for (int j = 0; j < stream.data_fds_size(); ++j) {
					safe_close(stream.data_fds(j));
				}
			} else {
				vector<int> data_fds;
				for (int j = 0; j < stream.data_fds_size(); ++j) {
					data_fds.push_back(stream.data_fds(j));
				}

				servers->add_stream_from_serialized(stream, data_fds);
				deserialized_urls.insert(stream.url());

				stream_headers_for_url.insert(make_pair(stream.url(), stream.stream_header()));
			}
		}

		// Deserialize the inputs. Note that we don't actually add them to any stream yet.
		for (int i = 0; i < loaded_state.inputs_size(); ++i) {
			InputProto serialized_input = loaded_state.inputs(i);

			InputWithRefcount iwr;
			iwr.input = create_input(serialized_input);
			iwr.refcount = 0;
			inputs.insert(make_pair(serialized_input.url(), iwr));
		} 

		// Deserialize the acceptors.
		for (int i = 0; i < loaded_state.acceptors_size(); ++i) {
			sockaddr_in6 sin6 = extract_address_from_acceptor_proto(loaded_state.acceptors(i));
			deserialized_acceptors.insert(make_pair(
				sin6,
				new Acceptor(loaded_state.acceptors(i))));
		}

		log(INFO, "Deserialization done.");
	}

	// Add any new inputs coming from the config.
	create_config_inputs(config, &inputs);
	
	// Find all streams in the configuration file, create them, and connect to the inputs.
	create_streams(config, deserialized_urls, &inputs);
	vector<Acceptor *> acceptors = create_acceptors(config, &deserialized_acceptors);
	
	// Put back the existing clients. It doesn't matter which server we
	// allocate them to, so just do round-robin.
	for (int i = 0; i < loaded_state.clients_size(); ++i) {
		if (deleted_urls.count(loaded_state.clients(i).url()) != 0) {
			safe_close(loaded_state.clients(i).sock());
		} else {
			servers->add_client_from_serialized(loaded_state.clients(i));
		}
	}
	
	servers->run();

	// Now delete all inputs that are longer in use, and start the others.
	for (multimap<string, InputWithRefcount>::iterator input_it = inputs.begin();
	     input_it != inputs.end(); ) {
		if (input_it->second.refcount == 0) {
			log(WARNING, "Input '%s' no longer in use, closing.",
			    input_it->first.c_str());
			input_it->second.input->close_socket();
			delete input_it->second.input;
			inputs.erase(input_it++);
		} else {
			input_it->second.input->run();
			++input_it;
		}
	}

	// Start writing statistics.
	StatsThread *stats_thread = NULL;
	if (!config.stats_file.empty()) {
		stats_thread = new StatsThread(config.stats_file, config.stats_interval);
		stats_thread->run();
	}

	InputStatsThread *input_stats_thread = NULL;
	if (!config.input_stats_file.empty()) {
		vector<Input*> inputs_no_refcount;
		for (multimap<string, InputWithRefcount>::iterator input_it = inputs.begin();
		     input_it != inputs.end(); ++input_it) {
			inputs_no_refcount.push_back(input_it->second.input);
		}

		input_stats_thread = new InputStatsThread(config.input_stats_file, config.input_stats_interval, inputs_no_refcount);
		input_stats_thread->run();
	}

	struct timeval server_start;
	gettimeofday(&server_start, NULL);
	if (state_fd != -1) {
		// Measure time from we started deserializing (below) to now, when basically everything
		// is up and running. This is, in other words, a conservative estimate of how long our
		// “glitch” period was, not counting of course reconnects if the configuration changed.
		double glitch_time = server_start.tv_sec - serialize_start.tv_sec +
			1e-6 * (server_start.tv_usec - serialize_start.tv_usec);
		log(INFO, "Re-exec happened in approx. %.0f ms.", glitch_time * 1000.0);
	}

	while (!hupped) {
		usleep(100000);
	}

	// OK, we've been HUPed. Time to shut down everything, serialize, and re-exec.
	gettimeofday(&serialize_start, NULL);

	if (input_stats_thread != NULL) {
		input_stats_thread->stop();
		delete input_stats_thread;
	}
	if (stats_thread != NULL) {
		stats_thread->stop();
		delete stats_thread;
	}
	for (size_t i = 0; i < acceptors.size(); ++i) {
		acceptors[i]->stop();
	}
	for (multimap<string, InputWithRefcount>::iterator input_it = inputs.begin();
	     input_it != inputs.end();
	     ++input_it) {
		input_it->second.input->stop();
	}
	servers->stop();

	CubemapStateProto state;
	if (stopped) {
		log(INFO, "Shutting down.");
	} else {
		log(INFO, "Serializing state and re-execing...");
		state = collect_state(
			serialize_start, acceptors, inputs, servers);
		string serialized;
		state.SerializeToString(&serialized);
		state_fd = make_tempfile(serialized);
		if (state_fd == -1) {
			exit(1);
		}
	}
	delete servers;

	access_log->stop();
	delete access_log;
	shut_down_logging();

	if (stopped) {
		exit(0);
	}

	// OK, so the signal was SIGHUP. Check that the new config is okay, then exec the new binary.
	if (!dry_run_config(argv0_canon, config_filename_canon)) {
		open_logs(config.log_destinations);
		log(ERROR, "%s --test-config failed. Restarting old version instead of new.", argv[0]);
		hupped = false;
		shut_down_logging();
		goto start;
	}
	 
	char buf[16];
	sprintf(buf, "%d", state_fd);

	for ( ;; ) {
		execlp(argv0_canon, argv0_canon, config_filename_canon, "--state", buf, NULL);
		open_logs(config.log_destinations);
		log_perror("execlp");
		log(ERROR, "re-exec of %s failed. Waiting 0.2 seconds and trying again...", argv0_canon);
		shut_down_logging();
		usleep(200000);
	}
}
