/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/stream_test_system_main.hpp
 *
 * @brief [LEVEL: beta] Macro to start a @ref
 * diagnostics::unittest::Stream_Test_System and to initialize the @ref
 * diagnostics::Logging_Config with a @ref
 * diagnostics::Intending_File_Logger. Provides insulation from all
 * involved interfaces.
 *
 * Clients have also the option to write their own insulation
 * wrapers. If they do so, they can use another configuration for
 * their testing while providing complilte-time independence.
 *
 * $Id: stream_test_system_main.hpp,v 1.4 2005/06/23 09:54:24 esdentem Exp $
 * 
 * @author Christian Schallhart
 *
 * @test none
 */
#ifndef DIAGNOSTICS__UNITTEST__STREAM_TEST_SYSTEM_MAIN_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__UNITTEST__STREAM_TEST_SYSTEM_MAIN_HPP__INCLUDE_GUARD

#include <diagnostics/unittest/namespace.hpp>

#include <diagnostics/frame/set_initial_loggers.hpp>
#include <diagnostics/macros/block_annotation.hpp>

#include <diagnostics/unittest/build_test_suite.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

/**
 * @class Stream_Test_System_Main
 *
 * @brief This class provides two methods, one for setting an initial
 * logger (a @ref File_Logger on "test/diagnostics.log") and a main
 * function which starts a @ref Stream_Test_System based on the
 * command-line arguments, build-level of the client executable and
 * given test-suite. The main reason for this wrapper is insulation --
 * any change in the interfaces of the file-logger or of the @ref
 * Stream_Test_System should force us to recompile a client. 
 */
struct Stream_Test_System_Main
{
    static void set_initial_loggers(::std::vector<Logger *> & loggers);
    static int main(int const argc,char const * const * const argv,int const build_level,
					Test_Suite* (*build_test_suite_ptr)());
};

UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;


/**
 * @brief starts a test environment using the methods of @ref
 * diagnostics::unittest::Stream_Test_System_Main
 *
 */
#define STREAM_TEST_SYSTEM_MAIN_ALL \
DIAGNOSTICS_NAMESPACE_BEGIN; \
void set_initial_loggers(::std::vector<Logger *> & loggers) \
{ \
   ::diagnostics::unittest::Stream_Test_System_Main::set_initial_loggers(loggers); \
} \
DIAGNOSTICS_NAMESPACE_END;\
\
int main(int argc,char ** argv) \
{ \
    int result; \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_PROD,::diagnostics::TYPE_PROCEDURE_ENTER,::diagnostics::TYPE_PROCEDURE_EXIT, \
                                 0,::std::string("procedure=\"") + DIAGNOSTICS_FUNC_NAME + "\" argv[0]=\"" + argv[0]+"\""); \
    result= ::diagnostics::unittest::Stream_Test_System_Main::main(argc,argv,DEBUG__LEVEL__, \
                                                                   & ::diagnostics::unittest::build_test_suite); \
    return result; \
} void dummy_procedure_for_semicolon_do_not_call()

#ifdef __MINGW32__
#  define STREAM_TEST_SYSTEM_MAIN \
int _CRT_glob = 0; \
STREAM_TEST_SYSTEM_MAIN_ALL
#else
#  define STREAM_TEST_SYSTEM_MAIN \
STREAM_TEST_SYSTEM_MAIN_ALL
#endif

#endif
// vim:ts=4:sw=4
