#include "config.h"
#include "virlist.h"
#include <qlistview.h>
#include <dirent.h>
#include "dirview.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <qpixmap.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <qtabwidget.h>
#include <ctype.h>
#include <qfileinfo.h>
#include <qpopupmenu.h>
#include <qmessagebox.h>
#include "playlist.h"
#ifdef HAVE_QOBJECTLIST_H
#include <qobjectlist.h>
#else
#ifdef HAVE_QOBJCOLL_H
#include <qobjcoll.h>
#else
#error "Your QT has neither qobjcoll.h nor qobjectlist.h"
#endif
#endif
#include "scanthread.h"
#include "dragsort.h"

#include "dir.xpm"
#include "file.xpm"
#include "up.xpm"

void VirList::resizeEvent(QResizeEvent *e)
{
    lv->resize(e->size().width()-10, e->size().height()-10);
    lv->setColumnWidth(0, e->size().width()-174);
    lv->setColumnWidth(1, 70);
    lv->setColumnWidth(2, 70);
    lv->repaint();
}

VirList::VirList(QWidget *parent, Playlist *pl, const char *name, WFlags f) : List(parent, pl, name, f)
{
	p_dir=new QPixmap((const char **)dir_xpm);
	p_file=new QPixmap((const char **)file_xpm);
	p_up=new QPixmap((const char **)up_xpm);
	dir="";
	lv=new DragSortList(this);
	lv->setSorting(-1, true);
	lv->move(5, 5);
	lv->addColumn(dir);
    lv->addColumn("BPM");
    lv->addColumn("Time");
	lv->setColumnWidthMode(0, QListView::Manual);
    lv->setAllColumnsShowFocus(true);
    lv->setColumnAlignment(1, AlignRight);
    lv->setColumnAlignment(2, AlignRight);
	lv->setSelectionMode(QListView::Extended);

	connect(lv, SIGNAL(doubleClicked(QListViewItem *)), this, SLOT(lv_doubleClicked(QListViewItem *)));
	connect(lv, SIGNAL(rightButtonPressed(QListViewItem *, const QPoint&, int)), this, SLOT(lv_rightClicked(QListViewItem *, const QPoint&, int)));
}

void VirList::load()
{
	if(file == "" || file == QString::null)
		return;

	QFileInfo info(file);
	if(info.exists())
	{
		FILE *fp=fopen(file, "r");
		if(!fp)
			return;

		char buf[2048];
		QString runtime;
		while(fgets(buf, sizeof(buf), fp) && !feof(fp))
		{
			if(!memcmp(buf, "[playlist]", 10))
			{
				setType(PLS);
				continue;
			}

			if(!memcmp(buf, "NumberOfEntries=", 16) && type() == PLS)
				continue;

			if(!memcmp(buf, "#EXTM3U", 7))
				setType(M3U);
			if(buf[0] == '#' && s_type == M3U)
			{
				if(!memcmp(buf, "#EXTINF:", 8))
				{
					int secs=atoi(buf+8);
					if(secs)
						runtime.sprintf("%d:%d", secs/60, secs%60);
					else
						runtime=QString::null;
				}
				else
					runtime=QString::null;
			}
			if(buf[0] == '#')
				continue;
			char *x=strchr(buf, '\n');
			if(x)
				*x='\0';
			if(!memcmp(buf, "File", 4) && type() == PLS)
			{
				int fnum=atoi(buf+4);
				QString check;
				check.sprintf("File%d=", fnum);
				if(!memcmp(buf, (const char *)check, check.length()))
					memmove(buf, buf+check.length(), strlen(buf)-check.length()+1);
			}
			addFile(buf, runtime, false);
			runtime=QString::null;
		}
		fclose(fp);
	}
	else
	{
		int fd=open(file, O_CREAT | O_WRONLY, 0644);
		if(fd >= 0)
			close(fd);
	}
	save();
}

VirList::~VirList()
{
	save();
	pl->scanThread()->removeList(this);
}

void VirList::lv_doubleClicked(QListViewItem *it)
{
	pl->doubleClick();
}

void VirList::setDir(QString i_dir)
{
}

QString VirList::getDir()
{
	QListViewItem *it=lv->currentItem();
	if(!it)
		return "/";
	QFileInfo info(it->text(3));
	return info.dirPath();
}

void VirList::setFile(QString i_file)
{
	file=i_file;
	load();
}

void VirList::lv_rightClicked(QListViewItem *it, const QPoint& pos, int col)
{
	if(!it)
		return;
	QPopupMenu *menu=new QPopupMenu(this);
	menu->insertItem("Load left", pl, SLOT(left_clicked()));
	menu->insertItem("Load right", pl, SLOT(right_clicked()));
	menu->insertSeparator();
	menu->insertItem("Remove", this, SLOT(removeItem()));
	VirList *autolist=0;
    QObjectList *ch=parent()->queryList("VirList");
    if(ch)
    {
        if(ch->count())
        {
            menu->insertSeparator();
            QPopupMenu *submenu=new QPopupMenu(menu);

            QObjectListIt i(*ch);
			QObject *obj;
            while((obj=i.current()) != 0)
            {
                ++i;
                VirList *list=(VirList *)obj;
				if(list->isA("PlayerList"))
					autolist=list;
                int id=submenu->insertItem(list->getTitle(), this, SLOT(moveToList(int)));
                submenu->setItemParameter(id, (long)list);
            }
            menu->insertItem("Send to", submenu);
        }
    }
	if(autolist)
	{
		int id=menu->insertItem("Set Auto list", this, SLOT(setAutoList(int)));
		menu->setItemParameter(id, (long)autolist);
	}
	menu->insertItem("Select all", this, SLOT(selectAll()));
    delete ch;

	menu->popup(pos);
}

void VirList::removeItem()
{
	QList<QListViewItem> l;
	l.setAutoDelete(true);
	QListViewItem *it=lv->firstChild();
    while(it)
    {
        if(it->isSelected())
            l.append(it);
        it=it->nextSibling();
    }
    if(l.count() == 0)
        return;
	if(QMessageBox::warning(this, "Remove", "Really remove item(s) from playlist?", "Yes", "No", 0, 0, 1) == 1)
	{
		l.setAutoDelete(true);
		return;
	}

    for(it=l.first();it;it=l.next())
        pl->scanThread()->removeFile(((DirViewItem *)it)->scanItem());
    l.clear();

	save();
}

void VirList::save()
{
	if(file == "" || file == QString::null)
		return;

	int file_num=1;
	int total_files=0;
	FILE *fp=fopen(file, "w");

	if(!fp)
	{
		QMessageBox::warning(this, "Error", "Cannot save file", "OK", 0, 0, 0, 0);
		return;
	}

	if(s_type == M3U)
	{
		fprintf(fp, "#EXTM3U\n");
	}
	QListViewItem *it=lv->firstChild();
	if(type() == PLS)
	{
		while(it)
		{
			if(it->text(3).left(7) != "cdda://")
				total_files++;
			it=it->nextSibling();
		}
		it=lv->firstChild();
		fprintf(fp, "[playlist]\n");
		fprintf(fp, "NumberOfEntries=%d\n", total_files);
	}
	while(it)
	{
		if(it->text(3).left(7) != "cdda://")
		{
			if(s_type == M3U)
			{
				int runtime=0;
				QString rt=it->text(2);
				if(rt != "" && rt != QString::null)
				{
					int pos=rt.find(":");
					if(pos != -1)
					{
						int mins=rt.mid(0, pos).toInt();
						int secs=rt.mid(pos+1).toInt();
						runtime=mins*60+secs;
					}
				}
				QString dname=it->text(0);
				int pos=dname.findRev(".");
				if(pos != -1)
				{
					dname=dname.mid(0, pos);
				}
				fprintf(fp, "#EXTINF:%d,%s\n", runtime, (const char *)dname);
			}
			if(type() == PLS)
				fprintf(fp, "File%d=%s\n", file_num++, (const char *)it->text(3));
			else
				fprintf(fp, "%s\n", (const char *)it->text(3));
		}
		it=it->nextSibling();
	}
	fclose(fp);
}

void VirList::addFile(QString a_file, bool s)
{
	addFile(a_file, QString::null, s);
}

void VirList::addFile(QString a_file, QString runtime, bool s)
{
	DirViewItem *it=(DirViewItem *)lv->firstChild();
	DirViewItem *insert_after=0;
	while(it)
	{
//		if(it->text(3) == a_file)
//			return;
		insert_after=it;
		it=(DirViewItem *)it->nextSibling();
	}
	QFileInfo info(a_file);
	it=new DirViewItem(lv, info.fileName(), false);
	if(insert_after)
		it->moveItem(insert_after);
	it->setPixmap(0, *p_file);
	it->setText(3, a_file);
	if(runtime != QString::null)
	{
		it->setText(2, runtime);
	}
	else
	{
		pl->scanThread()->addFile(this, a_file, it);
		it->setScanItem(ScanItem(pl->scanThread(), this, a_file, it));
	}
	if(s)
		save();
}

void VirList::moveToList(int arg)
{
    List *list=(List *)arg;
    if(!list)
        return;

    QListViewItem *it=lv->firstChild();
	lv->setEnabled(false);
    while(it)
    {
        if(it->isSelected())
        {
            QString path=it->text(3);
            list->addFile(path);
        }
        it=it->nextSibling();
    }
	lv->setEnabled(true);
}

void VirList::setAutoList(int arg)
{
    List *list=(List *)arg;
    if(!list)
        return;

	if(QMessageBox::warning(this, "Overwrite warning", "Are you sure you want to replace the contents of the Auto list with the contents of this list? Contents of the Auto list will be lost!", "Yes", "No", 0, 0, 1) != 0)
		return;

	list->lv->clear();
	pl->scanThread()->removeList(list);

    QListViewItem *it=lv->firstChild();
	lv->setEnabled(false);
    while(it)
    {
		QString path=it->text(3);
		list->addFile(path);
        it=it->nextSibling();
    }
	lv->setEnabled(true);
}

VirList::Type VirList::type()
{
	return s_type;
}

void VirList::setType(VirList::Type t)
{
	s_type=t;
}
