"This module provides plotting functionality (wrapper for Viper)."

# Copyright (C) 2008 Joachim B. Haga
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN. If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Martin Sandve Alnaes, 2008.
# Modified by Anders Logg, 2008-2010.
#
# First added:  2008-03-05
# Last changed: 2010-12-08

import os
import dolfin.cpp as cpp
from donothing import DoNothing
from dolfin.functions.function import Function
import ufl

__all__ = ['Viper', 'plot', 'update', 'interactive', 'save_plot', 'figure']

def make_viper_object(object, mesh=None):
    for plottype,classes in viper_dolfin.plottable:
        if isinstance(object, classes):
            return object

    if isinstance(object, cpp.DirichletBC):
        bc = object
        V = bc.function_space()
        v = Function(V)
        bc.apply(v.vector())
        return v

    # Try projecting function or expression
    from dolfin.fem.projection import project
    try:
        u = project(object, mesh=mesh)
        cpp.info("Object cannot be plotted directly, projecting to piecewise linears.")
        return u
    except:
        raise RuntimeError, ("Don't know how to plot given object and projection failed: " + str(object))


# Intelligent plot command that handles projections and different objects
# (aliased as 'plot')
def dolfin_plot(object, *args, **kwargs):
    """
    Plot given object using Viper.

    *Arguments*
        object
            a :py:class:`Mesh <dolfin.cpp.Mesh>`, a :py:class:`MeshFunction
            <dolfin.cpp.MeshFunction>`, a :py:class:`Function
            <dolfin.functions.function.Function>` or a
            :py:class:`FiniteElement <ufl.FiniteElement>`.

    *Examples of usage*
        In the simplest case, to plot only e.g. a mesh, simply use

        .. code-block:: python

            mesh = UnitSquare(4,4)
            plot(mesh)

        Use the ``title`` argument to specify title of the plot

        .. code-block:: python

            plot(mesh, tite="Finite element mesh")

        Specify the mesh when plotting a Function

        .. code-block:: python

            plot(v, mesh=mesh, title="My Function")

        It is also possible to plot an element

        .. code-block:: python

            element = FiniteElement("BDM", tetrahedron, 3)
            plot(element)

        A more advanced example

        .. code-block:: python

            plot(u,
                 mode = "displacement",
                 mesh = mesh,
                 wireframe = True,
                 interactive = True,         # hold plot on screen
                 axes = True,                # include axes
                 basename = "displacement",  # default plotfile name
                 rescale = False)

    """

    # Plot element
    if isinstance(object, ufl.FiniteElementBase):
        import ffc
        return ffc.plot(object, *args, **kwargs)

    # Check expression
    if isinstance(object, cpp.Expression) and "mesh" not in kwargs:
        raise TypeError, "expected a mesh when plotting an expression."

    mesh = kwargs.get('mesh')
    return viper_dolfin.plot(make_viper_object(object, mesh=mesh), *args, **kwargs)

# Check DOLFIN_NOPLOT
do_nothing = False
if os.environ.has_key('DOLFIN_NOPLOT'):
    cpp.info("DOLFIN_NOPLOT set, plotting disabled.")
    do_nothing = True
else:

    # Check for Viper
    try:
        from viper import viper_dolfin
        for x in __all__:
            exec ('from viper.viper_dolfin import %s' % x)
        plot = dolfin_plot
    except ImportError, details:
        cpp.warning(str(details))
        cpp.warning("Unable to import Viper, plotting disabled.")
        do_nothing= True

# Ignore all plot calls
if do_nothing:
    for x in __all__:
        exec('%s = DoNothing' % x)
