/* Copyright (C) 2002 MySQL AB & tommy@valley.ne.jp.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; version 2
   of the License.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 51 Franklin Place - Suite 330, Boston,
   MA 02110-1301, USA */

/* This file is for binary pseudo charset, created by bar@mysql.com */


#include <config.h>
#include <drizzled/definitions.h>
#include <drizzled/internal/m_string.h>
#include <drizzled/charset_info.h>

#include <algorithm>

using namespace std;

namespace drizzled
{

void my_hash_sort_bin(const CHARSET_INFO * const,
                      const unsigned char *key, size_t len,
                      uint32_t *nr1, uint32_t *nr2);


static unsigned char ctype_bin[]=
{
  0,
  32, 32, 32, 32, 32, 32, 32, 32, 32, 40, 40, 40, 40, 40, 32, 32,
  32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32,
  72, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
  132,132,132,132,132,132,132,132,132,132, 16, 16, 16, 16, 16, 16,
  16,129,129,129,129,129,129,  1,  1,  1,  1,  1,  1,  1,  1,  1,
  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 16, 16, 16, 16,
  16,130,130,130,130,130,130,  2,  2,  2,  2,  2,  2,  2,  2,  2,
  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2, 16, 16, 16, 16, 32,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
};


/* Dummy array for toupper / tolower / sortorder */

static unsigned char bin_char_array[] =
{
    0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
   16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
   32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
   48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
   64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
   80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95,
   96, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,
  112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,
  128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
  144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
  160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
  176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,
  192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,
  208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,
  224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,
  240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255
};


bool my_coll_init_8bit_bin(CHARSET_INFO *cs,
                           cs_alloc_func)
{
  cs->max_sort_char=255;
  return false;
}

int my_strnncoll_binary(const CHARSET_INFO * const,
                        const unsigned char *s, size_t slen,
                        const unsigned char *t, size_t tlen,
                        bool t_is_prefix)
{
  size_t len= min(slen,tlen);
  int cmp= memcmp(s,t,len);
  return cmp ? cmp : static_cast<int>((t_is_prefix ? len : slen) - tlen);
}


size_t my_lengthsp_binary(const CHARSET_INFO * const,
                          const char *, size_t length)
{
  return length;
}


/*
  Compare two strings. Result is sign(first_argument - second_argument)

  SYNOPSIS
    my_strnncollsp_binary()
    cs			Chararacter set
    s			String to compare
    slen		Length of 's'
    t			String to compare
    tlen		Length of 't'

  NOTE
   This function is used for real binary strings, i.e. for
   BLOB, BINARY(N) and VARBINARY(N).
   It compares trailing spaces as spaces.

  RETURN
  < 0	s < t
  0	s == t
  > 0	s > t
*/

int my_strnncollsp_binary(const CHARSET_INFO * const cs,
                          const unsigned char *s, size_t slen,
                          const unsigned char *t, size_t tlen,
                          bool)
{
  return my_strnncoll_binary(cs,s,slen,t,tlen,0);
}


int my_strnncoll_8bit_bin(const CHARSET_INFO * const,
                          const unsigned char *s, size_t slen,
                          const unsigned char *t, size_t tlen,
                          bool t_is_prefix)
{
  size_t len= min(slen,tlen);
  int cmp= memcmp(s,t,len);
  return cmp ? cmp : static_cast<int>((t_is_prefix ? len : slen) - tlen);
}


/*
  Compare two strings. Result is sign(first_argument - second_argument)

  SYNOPSIS
    my_strnncollsp_8bit_bin()
    cs			Chararacter set
    s			String to compare
    slen		Length of 's'
    t			String to compare
    tlen		Length of 't'
    diff_if_only_endspace_difference
		        Set to 1 if the strings should be regarded as different
                        if they only difference in end space

  NOTE
   This function is used for character strings with binary collations.
   The shorter string is extended with end space to be as long as the longer
   one.

  RETURN
  < 0	s < t
  0	s == t
  > 0	s > t
*/

int my_strnncollsp_8bit_bin(const CHARSET_INFO * const,
                            const unsigned char *a, size_t a_length,
                            const unsigned char *b, size_t b_length,
                            bool diff_if_only_endspace_difference)
{
  const unsigned char *end;
  size_t length;
  int res;

#ifndef VARCHAR_WITH_DIFF_ENDSPACE_ARE_DIFFERENT_FOR_UNIQUE
  diff_if_only_endspace_difference= 0;
#endif

  end= a + (length= min(a_length, b_length));
  while (a < end)
  {
    if (*a++ != *b++)
      return a[-1] - b[-1];
  }
  res= 0;
  if (a_length != b_length)
  {
    int swap= 1;
    /*
      Check the next not space character of the longer key. If it's < ' ',
      then it's smaller than the other key.
    */
    if (diff_if_only_endspace_difference)
      res= 1;                                   /* Assume 'a' is bigger */
    if (a_length < b_length)
    {
      /* put shorter key in s */
      a_length= b_length;
      a= b;
      swap= -1;					/* swap sign of result */
      res= -res;
    }
    for (end= a + a_length-length; a < end ; a++)
    {
      if (*a != ' ')
	return (*a < ' ') ? -swap : swap;
    }
  }
  return res;
}


/* This function is used for all conversion functions */

size_t my_case_str_bin(const CHARSET_INFO * const, char *)
{
  return 0;
}


size_t my_case_bin(const CHARSET_INFO * const, char *,
                   size_t srclen, char *, size_t)
{
  return srclen;
}


int my_strcasecmp_bin(const CHARSET_INFO * const,
                      const char *s, const char *t)
{
  return strcmp(s,t);
}


uint32_t my_mbcharlen_8bit(const CHARSET_INFO * const, uint32_t)
{
  return 1;
}


int my_mb_wc_bin(const CHARSET_INFO * const,
                 my_wc_t *wc, const unsigned char *str,
                 const unsigned char *end)
{
  if (str >= end)
    return MY_CS_TOOSMALL;

  *wc=str[0];
  return 1;
}


int my_wc_mb_bin(const CHARSET_INFO * const, my_wc_t wc,
                 unsigned char *str, unsigned char *end)
{
  if (str >= end)
    return MY_CS_TOOSMALL;

  if (wc < 256)
  {
    str[0]= wc;
    return 1;
  }
  return MY_CS_ILUNI;
}


void my_hash_sort_8bit_bin(const CHARSET_INFO * const,
                           const unsigned char *key, size_t len,
                           uint32_t *nr1, uint32_t *nr2)
{
  const unsigned char *pos = key;

  /*
     Remove trailing spaces. We have to do this to be able to compare
    'A ' and 'A' as identical
  */
  key= internal::skip_trailing_space(key, len);

  for (; pos < key ; pos++)
  {
    nr1[0]^= (((nr1[0] & 63) + nr2[0]) * *pos) + (nr1[0] << 8);
    nr2[0]+=3;
  }
}


void my_hash_sort_bin(const CHARSET_INFO * const,
                      const unsigned char *key, size_t len,
                      uint32_t *nr1, uint32_t *nr2)
{
  const unsigned char *pos = key;

  key+= len;

  for (; pos < key ; pos++)
  {
    nr1[0]^= (((nr1[0] & 63) + nr2[0]) * *pos) + (nr1[0] << 8);
    nr2[0]+=3;
  }
}


/*
  The following defines is here to keep the following code identical to
  the one in ctype-simple.c
*/

#define likeconv(s,A) (A)
#define INC_PTR(cs,A,B) (A)++


int my_wildcmp_bin(const CHARSET_INFO * const cs,
                   const char *str,const char *str_end,
                   const char *wildstr,const char *wildend,
                   int escape, int w_one, int w_many)
{
  int result= -1;			/* Not found, using wildcards */

  while (wildstr != wildend)
  {
    while (*wildstr != w_many && *wildstr != w_one)
    {
      if (*wildstr == escape && wildstr+1 != wildend)
	wildstr++;
      if (str == str_end || likeconv(cs,*wildstr++) != likeconv(cs,*str++))
	return(1);			/* No match */
      if (wildstr == wildend)
	return(str != str_end);		/* Match if both are at end */
      result=1;				/* Found an anchor char */
    }
    if (*wildstr == w_one)
    {
      do
      {
	if (str == str_end)		/* Skip one char if possible */
	  return(result);
	INC_PTR(cs,str,str_end);
      } while (++wildstr < wildend && *wildstr == w_one);
      if (wildstr == wildend)
	break;
    }
    if (*wildstr == w_many)
    {					/* Found w_many */
      unsigned char cmp;
      wildstr++;
      /* Remove any '%' and '_' from the wild search string */
      for (; wildstr != wildend ; wildstr++)
      {
	if (*wildstr == w_many)
	  continue;
	if (*wildstr == w_one)
	{
	  if (str == str_end)
	    return(-1);
	  INC_PTR(cs,str,str_end);
	  continue;
	}
	break;				/* Not a wild character */
      }
      if (wildstr == wildend)
	return(0);			/* match if w_many is last */
      if (str == str_end)
	return(-1);

      if ((cmp= *wildstr) == escape && wildstr+1 != wildend)
	cmp= *++wildstr;

      INC_PTR(cs,wildstr,wildend);	/* This is compared through cmp */
      cmp=likeconv(cs,cmp);
      do
      {
	while (str != str_end && (unsigned char) likeconv(cs,*str) != cmp)
	  str++;
	if (str++ == str_end)
	  return(-1);
	{
	  int tmp=my_wildcmp_bin(cs,str,str_end,wildstr,wildend,escape,w_one,
				 w_many);
	  if (tmp <= 0)
	    return(tmp);
	}
      } while (str != str_end && wildstr[0] != w_many);
      return(-1);
    }
  }
  return(str != str_end ? 1 : 0);
}


size_t
my_strnxfrm_8bit_bin(const CHARSET_INFO * const cs,
                     unsigned char * dst, size_t dstlen, uint32_t nweights,
                     const unsigned char *src, size_t srclen, uint32_t flags)
{
  set_if_smaller(srclen, dstlen);
  set_if_smaller(srclen, (size_t) nweights);
  if (dst != src)
    memcpy(dst, src, srclen);
  return my_strxfrm_pad_desc_and_reverse(cs, dst, dst + srclen, dst + dstlen,
                                         nweights - srclen, flags, 0);
}


uint32_t my_instr_bin(const CHARSET_INFO * const,
                      const char *b, size_t b_length,
                      const char *s, size_t s_length,
                      my_match_t *match, uint32_t nmatch)
{
  const unsigned char *str, *search, *end, *search_end;

  if (s_length <= b_length)
  {
    if (!s_length)
    {
      if (nmatch)
      {
        match->beg= 0;
        match->end= 0;
        match->mb_len= 0;
      }
      return 1;		/* Empty string is always found */
    }

    str= (const unsigned char*) b;
    search= (const unsigned char*) s;
    end= (const unsigned char*) b+b_length-s_length+1;
    search_end= (const unsigned char*) s + s_length;

skip:
    while (str != end)
    {
      if ( (*str++) == (*search))
      {
	const unsigned char *i,*j;

	i= str;
	j= search+1;

	while (j != search_end)
	  if ((*i++) != (*j++))
            goto skip;

        if (nmatch > 0)
	{
	  match[0].beg= 0;
	  match[0].end= (size_t) (str- (const unsigned char*)b-1);
	  match[0].mb_len= match[0].end;

	  if (nmatch > 1)
	  {
	    match[1].beg= match[0].end;
	    match[1].end= match[0].end+s_length;
	    match[1].mb_len= match[1].end-match[1].beg;
	  }
	}
	return 2;
      }
    }
  }
  return 0;
}


static MY_COLLATION_HANDLER my_collation_binary_handler =
{
  NULL,			/* init */
  my_strnncoll_binary,
  my_strnncollsp_binary,
  my_strnxfrm_8bit_bin,
  my_strnxfrmlen_simple,
  my_like_range_simple,
  my_wildcmp_bin,
  my_strcasecmp_bin,
  my_instr_bin,
  my_hash_sort_bin,
  my_propagate_simple
};


static MY_CHARSET_HANDLER my_charset_handler=
{
  NULL,			/* init */
  NULL,			/* ismbchar      */
  my_mbcharlen_8bit,	/* mbcharlen     */
  my_numchars_8bit,
  my_charpos_8bit,
  my_well_formed_len_8bit,
  my_lengthsp_binary,
  my_numcells_8bit,
  my_mb_wc_bin,
  my_wc_mb_bin,
  my_mb_ctype_8bit,
  my_case_str_bin,
  my_case_str_bin,
  my_case_bin,
  my_case_bin,
  my_snprintf_8bit,
  my_long10_to_str_8bit,
  my_int64_t10_to_str_8bit,
  my_fill_8bit,
  my_strntol_8bit,
  my_strntoul_8bit,
  my_strntoll_8bit,
  my_strntoull_8bit,
  my_strntod_8bit,
  my_strtoll10_8bit,
  my_strntoull10rnd_8bit,
  my_scan_8bit
};


DRIZZLED_API CHARSET_INFO my_charset_bin =
{
    63,0,0,			/* number        */
    MY_CS_COMPILED|MY_CS_BINSORT|MY_CS_PRIMARY,/* state */
    "binary",			/* cs name    */
    "binary",			/* name          */
    "",				/* comment       */
    NULL,			/* tailoring     */
    ctype_bin,			/* ctype         */
    bin_char_array,		/* to_lower      */
    bin_char_array,		/* to_upper      */
    NULL,			/* sort_order    */
    NULL,			/* contractions */
    NULL,			/* sort_order_big*/
    NULL,			/* tab_to_uni    */
    NULL,			/* tab_from_uni  */
    my_unicase_default,         /* caseinfo     */
    NULL,			/* state_map    */
    NULL,			/* ident_map    */
    1,				/* strxfrm_multiply */
    1,                          /* caseup_multiply  */
    1,                          /* casedn_multiply  */
    1,				/* mbminlen      */
    1,				/* mbmaxlen      */
    0,				/* min_sort_char */
    255,			/* max_sort_char */
    0,                          /* pad char      */
    0,                          /* escape_with_backslash_is_dangerous */
    1,                          /* levels_for_compare */
    1,                          /* levels_for_order   */
    &my_charset_handler,
    &my_collation_binary_handler
};

} /* namespace drizzled */
