# Ear Candy - Pulseaduio sound managment tool
# Copyright (C) 2008 Jason Taylor
# 
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
import logging
import ctypes
from pulseaudio.PulseAudioConnection import PulseAudioConnection
from Client import Client
from SinkInput import SinkInput
from Sink import Sink

log = logging.getLogger('EarCandyPulseAudio')
log.setLevel(logging.WARNING)

class PulseAudio(PulseAudioConnection):
    def __init__(self, Core):

        self.core = Core 
        self.prefer_sink_index = 0       
        self.prefer_source_index = 0 

        self.sink = {}          # sinks by name
        self.sinks_by_id = {}   # sinks by id
        self.clients = {}       # clients by name
        self.clients_by_id = {} # clients by id
        self.sink_inputs = {}   # sink inputs by id
        self.source_outputs = {}   # source outputs by id

        PulseAudioConnection.__init__(self, "EarCandy")

    # CONNECTION
    def pa_status_ready(self):
        # list sinks again now that sink_inputs have been loaded
        # this will trigger auto usb selection
        self.pa_context_get_sink_info_list()
        self.pa_context_get_source_info_list()
        self.pa_context_get_source_output_info_list()


    ### DEAL WITH SINKS #################################
    def pa_sink_info_cb(self, sink, user_data):
        s = None
        if not sink.name in self.sinks.keys():
            s = Sink(sink.name, 0, sink.index)
            log.info("new sink : %s %s" % (sink.description, sink.index))
            self.sinks[sink.name] = s
                      
        else:
            s = self.sinks[sink.name]

        self.sinks_by_id[sink.index] = s  
        s.description = sink.description
        s.monitor_source = sink.monitor_source
        s.monitor_source_name = sink.monitor_source_name
        s.index = sink.index
        s.volume = sink.volume

        if s.name.lower().count("usb") > 0 or s.name.lower().count("bluetooth") > 0:
            self.core.notify("New sound device", "Moving sound to new device")
            log.info("new device detected moving sink inputs : %s " % sink.description)

            self.prefer_sink_index = sink.index
            self.pa_context_set_default_sink(sink.name)
            self.move_all_sinks_inputs(sink.index)

        if self.core.pref:
            self.core.pref.update_output(s, s.priority)

    def pa_sink_remove_cb(self, index):
        sink = self.sinks_by_id[index]
        sink.index = 0
        if self.core.pref:
            self.core.pref.update_output(sink, False)
        return

    def move_all_sinks_inputs(self, sink_index):
        for sink_input in self.sink_inputs.values():
            self.pa_context_move_sink_input_by_index(sink_input.index, sink_index)

    def move_client_to_sink(self, client):
        sink_index = self.prefer_sink_index
        if client.prefer_sink:
            for sink in self.sinks_by_id.values():
                if sink.name == client.prefer_sink:
                    sink_index = sink.index
                    break
        for sink_input in client.sinks.values():
            self.pa_context_move_sink_input_by_index(sink_input.index, sink_index)

    ### DEAL WITH SINKS #################################


    ### DEAL WITH SOURCES ###############################
    def pa_source_info_cb(self, source, user_data, s):
        if not source.is_monitor_of_sink:
            if source.name.lower().count("usb") > 0 or source.name.lower().count("bluetooth") > 0:
                self.core.notify("New microphone", "Moving input to new device")
                self.prefer_source_index = source.index
                self.pa_context_set_default_source(source.name)
                self.move_all_source_outputs(source.index)

    def pa_source_output_info_cb(self, source_output, user_data):

        if self.prefer_source_index > 0 and source_output.source != self.prefer_source_index:
            self.pa_context_move_source_output_by_index(source_output.index, self.prefer_source_index)

        self.source_outputs[source_output.index] = source_output

    def move_all_source_outputs(self, source_index):
        for source_output in self.source_outputs.values():
            self.pa_context_move_source_output_by_index(source_output.index,  source_index)

    ### DEAL WITH SOURCES ###############################


    # CLIENTS    
    def pa_client_info_cb(self, client, user_data):

        if not self.clients.has_key(client.name):
            c = Client(self.core, client.name, client.pid)
            log.info("new client : %s" % c.description)
            
            self.clients[client.name] = c
        else:
            log.debug("update client : %s" % client.name)
            c = self.clients[client.name]
            c.pid = client.pid
        self.clients_by_id[client.index] = c

    def pa_client_remove_cb(self, index):
        
        if self.clients_by_id.has_key(index):
            client = self.clients_by_id[index]
            log.debug("remove client : %s" % client.name)
            del self.clients_by_id[index]

    # SINK INPUTS
    def pa_sink_input_info_cb(self, sink_input, user_data):
        target_sink = self.prefer_sink_index

        if not self.sink_inputs.has_key(sink_input.index):
            s = SinkInput(sink_input.index, sink_input.name, sink_input.volume, sink_input.sink, sink_input.media_role)
            self.sink_inputs[s.index] = s

            if self.clients_by_id.has_key(sink_input.client):
                client = self.clients_by_id[sink_input.client]
                client.sinks[s.index] = s
                s.client = client

                # Skim client role from 1st stream
                if not client.role and sink_input.media_role:
                    client.role = sink_input.media_role
                    log.info("new client media role : %s %s " % (client.name, sink_input.media_role))

            self.pa_create_monitor_stream_for_sink_input(sink_input.index, self.sinks_by_id[sink_input.sink].monitor_source, sink_input.name, s.client.description)

            log.info("new sink input : %s %s " % (s.client.description, s.name))

        else:
            log.debug("update sink input : %s %s" % (sink_input.name, sink_input.sink))
            s = self.sink_inputs[sink_input.index]
            s.volume = sink_input.volume
            s.name = sink_input.name

            # Check we have the correct sink
            if s.client.prefer_sink:
                for sink in self.sinks_by_id.values():
                    if sink.name == s.client.prefer_sink:
                        target_sink = sink.index

            if not s.sink == sink_input.sink:
                self.pa_create_monitor_stream_for_sink_input(sink_input.index, self.sinks_by_id[sink_input.sink].monitor_source, sink_input.name, s.client.description)   
                s.sink = sink_input.sink

        if target_sink > 0 and self.sink_inputs[sink_input.index].sink != target_sink:
            self.pa_context_move_sink_input_by_index(sink_input.index, target_sink)

        # recheck clients to windows
        self.core.window_watcher.check_all()

    def pa_sink_input_remove_cb(self, index):

        if self.sink_inputs.has_key(index):
            sink = self.sink_inputs[index]
            log.debug("remove sink input : %s" % sink.name)
            
            del( sink.client.sinks[index] )
            del( self.sink_inputs[index] )

            # reset volume in pa
            v = []
            for i in range(0, len(sink.volume)):
                v.append(100)
            self.pa_context_set_sink_input_volume(index, v)

            if self.core.pref:
                self.core.pref.update_client( sink.client )


    # VOLUME METER FOR SINK INPUT
    def pa_stream_request_cb(self, meter_level, user_data):
        if user_data in self.sink_inputs.keys():
            self.sink_inputs[user_data].set_meter( meter_level )
            log.debug("pa_stream_request_cb : %s %s" % (meter_level, user_data))
            if self.core.pref:
                self.core.pref.update_client( self.sink_inputs[user_data].client )

    def pa_create_monitor_stream_for_sink_input(self, index, monitor_index, name, description):
        
        log.debug("pa_create_monitor_stream_for_sink_input : %s" % index)

        # Create new stream
        ss = self.pa_sample_spec()
        ss.channels = 1
        ss.format = 5
        ss.rate = 25
        pa_stream = self.pa_stream_new(description, ss)
        
        self.pa_stream_set_monitor_stream(pa_stream, index);
        self.pa_stream_set_read_callback(pa_stream, index);
        self.pa_stream_set_suspended_callback(pa_stream);

        attr = self.pa_buffer_attr()
        attr.fragsize = 4
        attr.maxlength = 10
        attr.tlength = 0
        attr.prebuf = 0
        attr.minreq = 0

        self.pa_stream_connect_record(pa_stream, str(monitor_index), attr, 10752) 


if __name__ == '__main__':

    import gtk
    # Turn on gtk threading
    gtk.gdk.threads_init()

    pa = PulseAudio("Test")
    pa.connect()

    gtk.main()
