/* $Id: DeclarativeRegion.hpp 4323 2009-01-27 13:48:12Z potyra $ 
 *
 * DeclarativeRegion: One pocket to store symbols. Can be chained together.
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */


#ifndef __DECLARATIVE_REGION_HPP_INCLUDED
#define __DECLARATIVE_REGION_HPP_INCLUDED

#include <list>
#include <string>

#include "frontend/misc/Symbol.hpp"

namespace ast {

//! one declarative region
class DeclarativeRegion {
public:
	//! c'tor
	/**
	 * @param p link to parent declarative region, NULL in case 
	 *          it's a library declarative region.
	 */
	DeclarativeRegion(DeclarativeRegion *p);
	
	/** d'tor */
	~DeclarativeRegion();

	/** register symbol sym to declarations.
	 *  @param sym Symbol that should get registered.
	 */
	void registerSymbol(Symbol *sym);

	/** lookup name symbol by name.
	 *  @param name search for this name.
	 *  @return visible candidates for given symbol.
	 */
	std::list<Symbol*> lookup(const std::string &name) const ;

	/** check if sym is the symbol of an enclosing region 
	 *  @param sym symbol to check.
	 *  @return true, if sym declares a region which encloses
	 *          this region.
	 */
	bool isEnclosingSymbol(const Symbol &sym) const;

	/** symbols declared in this region */
	std::list<Symbol*> declarations;

	/** symbols visible through use-clauses */
	std::list<Symbol*> imports;

	/** parent declarative region */
	DeclarativeRegion *parent;

	// FIXME potyra: use a better storage facility than just an
	//       unordered list. Maybe a hashtable or s.th. like that.

	/** lookup name symbol by name, only in current DeclarativeRegion,
	 *  not considering imported symbols.
	 *  @param name search for this name
	 *  @return candidates for given symbol of local region.
	 */
	std::list<Symbol*> lookupLocal(const std::string &name) const;

private:
	/** check if this region contains a Homograph to 
	 *  symbol sym already.
	 *  @param sym check for a Homograph of this symbol.
	 *  @return the Symbol, which is a Homograph or NULL if 
	 *          no Homograph exists.
	 */
	const Symbol* containsHomograph(const Symbol &sym) const;

	/** recursively search (parent) declarative regions
	 *  that resolve the symbol with name.
	 *  @param name name to look for.
	 *  @param candidates list of Symbols which are candidates to 
	 *         resolve given name.
	 */
	void 
	lookupRec(
		const std::string &name,
		std::list<Symbol*> &candidates
		) const;

	/** check if symbol sym is visible in the context of a set of
	 *  possible homographs.
	 *  @param sym contains the symbol for which visibility should get
	 *             checked.
	 *  @param check list of symbols, which possibly hide sym.
	 *  @return true if the symbol is visible.
	 */
	bool 
	isVisible(
		const Symbol &sym, 
		const std::list<Symbol*> &check
		) const;

};

}; /* namespace ast */


#endif /* __DECLARATIVE_REGION_HPP_INCLUDED */
