/**************************************************************************\
 gatos (General ATI TV and Overlay Software)

  Project Coordinated By Insomnia (Steaphan Greene)
  (insomnia@core.binghamton.edu)

  Copyright (C) 1999 Steaphan Greene, yvind Aabling, Octavian Purdila, 
	Vladimir Dergachev and Christian Lupien.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

\**************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#include <ibtk/version.h>
#include <ibtk/iwindow.h>
#include <ibtk/ibox.h>
#include <ibtk/itextbox.h>
#include <ibtk/istatbar.h>
#include <ibtk/ibutton.h>
#include <ibtk/isbutton.h>
#include <ibtk/islider.h>
#include <ibtk/iintbox.h>
#include <ibtk/irealbox.h>
#include <ibtk/ilistbox.h>

#include "config.h"
#include "freq.h"
#include "xutils.h"
#include "version.h"

#define  NUM_I2C_MODES        6                     /* 
						     * 5 modes + probe 
						     */
/* Default pathnames for XF86Config file */
char *xfreeconfigfile[] = {
  "/etc/XF86Config",
  "/etc/X11/XF86Config"
  };

static IWindow *mwin = NULL;
static IListBox *formb = NULL;
static ITextBox *forml = NULL;
static ITextBox *buffertitle;
static ISButton *bufb1 = NULL, *bufb2 = NULL, *bufb3 = NULL;
static IIntBox *bufval = NULL;
static ITextBox *videoramtitle;
static ISButton *vramb1 = NULL, *vramb2 = NULL, *vramb3 = NULL;
static ISButton *vramb4 = NULL, *vramb5 = NULL;
static ISButton *ramprobedb;
static ITextBox *i2cmodetitle = NULL;
static IListBox *i2cmode = NULL;
static IButton *quitsaveb = NULL;
static IButton *quitabortb = NULL;
static ITextBox *statusbar = NULL;

void ResizeCB(IWindow *w, int x, int y);
void tvtypeCB(IDoDad *p, IDoDad *d, int n);
void i2cmodeCB(IDoDad *p, IDoDad *d, int n);
void quitsaveCB(IDoDad *p, IDoDad *d, int n);
void quitabortCB(IDoDad *p, IDoDad *d, int n);
void KeyMappingCB(IWindow *w, XEvent *ev);
void update_statusbar(void);
void videoramselect(IDoDad *p, IDoDad *d, int x);
void bufferselect(IDoDad *p, IDoDad *d, int x);
void chbufval(IDoDad *p, IDoDad *d);
void no_none_selected(IDoDad *p, IDoDad *d, int x);
void videoramprobed(IDoDad *p, IDoDad *d, int x);
int get_tvformat(int type);
int get_tvtypeformat(int type);
void updategui(void);
void initgui(void);
void initvar(void);
void save_configfiles(void);
void load_gatos_configfile(void);
int save_gatos_configfile(void);
void load_xfree_configfile(void);
int save_xfree_configfile(void);
int check_perms(void);
void usage(void);

int ixs, iys;  // X and Y size of main window

/* Some default values */
int videoramsize = 2048;
int buffersize = 600;
int i2cmodeselected = 0;
int tvtypeselected = 0;
int standardselected = 0;
int ramwillprobed = 1;

int xfreeline = -1;
int xfree_videoramsize = 1448; //default value if gatos.conf not exist
int xfree_videoram_line = -1;
int display_endsection = -1;
int videoramline[10]; //allow 10 video cards, because Xconfigurator
                      // seems to put many "Device" sections
char xfreeconfigname[128];
int xfree_path_overrided = 0;
char *i2c_modes[NUM_I2C_MODES];
char *xfreeconf[256]; 
int gatos_conf_exist = -1; //remember if gatos.conf exist

/* ------------------------------------------------------------------------ */

#define gerr(FMT,ARGS...)	fprintf(stderr,FMT,##ARGS)

/* ------------------------------------------------------------------------ */

int main(int argc, char **argv)  {
  int i;

  bindtextdomain(PACKAGE, GATOS_LOCALEDIR);
  textdomain(PACKAGE);

  if(!check_perms()) {
    gerr(_("gatos-conf: this program must be run as root.\n"));
    exit(1);
    }

  for(i=1; i<argc; i++) {
    if(!strcmp(argv[i], "-h")
        || !strcmp(argv[i], "-help")
        || !strcmp(argv[i], "--help"))       usage();
    else if(!strcmp(argv[i], "-X") && i+1<argc) {
      i++;                             /* Override pathname of XF86Config */
      xfreeconfigfile[3] = (char *)malloc(strlen(argv[i])+1);
      strncpy(xfreeconfigfile[3], argv[i], strlen(argv[i])+1);
      xfree_path_overrided = 1;
      }
    }

  /* Check installed ibtk version */
  check_ibtk_version();

  ixs = 480; iys = ((((NUM_STANDARDS*13+6+9)/10)*10)+90);

  mwin = new IWindow("GATOS-Configurator", ixs, iys);

  mwin->SetBufSize(ixs, iys);
  mwin->Map();
  mwin->SetResizeCallback(ResizeCB);
  mwin->SetKeyMappingCallback(KeyMappingCB);

  /* Initialize some vars */
  initvar();

  /* Initialize buttons and lists */
  initgui();

  mwin->DispatchEvents();
  }

/* ------------------------------------------------------------------------ */

void ResizeCB(IWindow *w, int x, int y)  {

  mwin->Resize(ixs, iys);
  mwin->RebuildClip();
  XFlush(mwin->GetDisplay());
  }

/* ------------------------------------------------------------------------ */

void tvtypeCB(IDoDad *p, IDoDad *d, int n)  {

  standardselected = n;
  tvtypeselected = get_tvformat(n);
  }

/* ------------------------------------------------------------------------ */

void i2cmodeCB(IDoDad *p, IDoDad *d, int n)  {

  i2cmodeselected = n;
  }

/* ------------------------------------------------------------------------ */

void quitsaveCB(IDoDad *p, IDoDad *d, int n)  {

  save_configfiles();
  mwin->Close();
  }

/* ------------------------------------------------------------------------ */

void quitabortCB(IDoDad *p, IDoDad *d, int n)  {

  mwin->Close();
  }

/* ------------------------------------------------------------------------ */

void KeyMappingCB(IWindow *w, XEvent *ev)  {

  XRefreshKeyboardMapping(&ev->xmapping);
  }

/* ------------------------------------------------------------------------ */

void update_statusbar(void)  {
  int ramforx;
  char buf[128];

  ramforx = (videoramsize - buffersize);

  sprintf(buf, _("Video RAM available for X: %dkb"), ramforx);

  if(ramforx <= 0 || ramforx >= videoramsize) {
    sprintf(buf, _("%s << problem!!"), buf);
    }

  /* Destroy status bar for update text !! */
  if(statusbar != NULL) free(statusbar);
  statusbar = new ITextBox(buf, mwin, 0, (iys - 20), ixs, 20);
  statusbar->Show();
  }

/* ------------------------------------------------------------------------ */

void videoramselect(IDoDad *p, IDoDad *d, int x)  {
 
  if(d == vramb1) { 
    videoramsize = 2048;
    }
  else if(d == vramb2) { 
    videoramsize = 4096;
    }
  else if(d == vramb3) { 
    videoramsize = 8192;
    }
  else if(d == vramb4) { 
    videoramsize = 16384;
    }
  else if(d == vramb5) { 
    videoramsize = 32768;
    }

  if(d != vramb1 && vramb1->State() == 1) vramb1->Press(1, 0, 0);
  if(d != vramb2 && vramb2->State() == 1) vramb2->Press(1, 0, 0);
  if(d != vramb3 && vramb3->State() == 1) vramb3->Press(1, 0, 0);
  if(d != vramb4 && vramb4->State() == 1) vramb4->Press(1, 0, 0);
  if(d != vramb5 && vramb5->State() == 1) vramb5->Press(1, 0, 0);

  update_statusbar();
  }

/* ------------------------------------------------------------------------ */

void bufferselect(IDoDad *p, IDoDad *d, int x)  {
 
  if(d == bufb1) { 
    buffersize = 600;
    bufval->Disable();
    }
  else if(d == bufb2) { 
    buffersize = 864;
    bufval->Disable();
    }
  else if(d == bufb3) {
    buffersize = ((IIntBox *)bufval)->GetVal();
    bufval->Enable();
    }

  if(d != bufb1 && bufb1->State() == 1) bufb1->Press(1, 0, 0);
  if(d != bufb2 && bufb2->State() == 1) bufb2->Press(1, 0, 0);
  if(d != bufb3 && bufb3->State() == 1) bufb3->Press(1, 0, 0);

  update_statusbar();
  }

/* ------------------------------------------------------------------------ */

void chbufval(IDoDad *p, IDoDad *d)  {

  buffersize = ((IIntBox *)bufval)->GetVal();
  update_statusbar();
  }
/* ------------------------------------------------------------------------ */
/*
 * Always one "button radio" must be selected.
 * Deselected all buttons in a "group" is avoided with this.
 */
void no_none_selected(IDoDad *p, IDoDad *d, int x)  {
  if(d == vramb1 && (vramb1->State() == 0
		     && vramb2->State() == 0 
		     && vramb3->State() == 0 
		     && vramb4->State() == 0
		     && vramb5->State() == 0)) vramb1->Press(1, 0, 0);
  if(d == vramb2 && (vramb1->State() == 0
		     && vramb2->State() == 0 
		     && vramb3->State() == 0 
		     && vramb4->State() == 0
		     && vramb5->State() == 0)) vramb2->Press(1, 0, 0);
  if(d == vramb3 && (vramb1->State() == 0
		     && vramb2->State() == 0 
		     && vramb3->State() == 0 
		     && vramb4->State() == 0
		     && vramb5->State() == 0)) vramb3->Press(1, 0, 0);
  if(d == vramb4 && (vramb1->State() == 0
		     && vramb2->State() == 0 
		     && vramb3->State() == 0 
		     && vramb4->State() == 0
		     && vramb5->State() == 0)) vramb4->Press(1, 0, 0);
  if(d == vramb5 && (vramb1->State() == 0
		     && vramb2->State() == 0 
		     && vramb3->State() == 0 
		     && vramb4->State() == 0
		     && vramb5->State() == 0)) vramb5->Press(1, 0, 0);

  if(d == bufb1 && (bufb1->State() == 0
		    && bufb2->State() == 0
		    && bufb3->State() == 0)) bufb1->Press(1, 0 ,0);
  if(d == bufb2 && (bufb1->State() == 0
		    && bufb2->State() == 0
		    && bufb3->State() == 0)) bufb2->Press(1, 0 ,0);
  if(d == bufb3 && (bufb1->State() == 0
  		    && bufb2->State() == 0
  		    && bufb3->State() == 0)) bufb3->Press(1, 0 ,0);
  }

/* ------------------------------------------------------------------------ */

void videoramprobed(IDoDad *p, IDoDad *d, int x)  {

  if(d == ramprobedb && ramprobedb->State() == 1) ramwillprobed = 1;
  else ramwillprobed = 0;

  update_statusbar();
  }

/* ------------------------------------------------------------------------ */

int get_tvformat(int type)  {

  return tvtype_format[type];
  }

/* ------------------------------------------------------------------------ */

int get_tvtypeformat(int type)  {
  int i;

  for(i=0; i<NUM_STANDARDS; i++) {
    if(tvtype_format[i] == type) return i;
    }

  return 0;
  }

/* ------------------------------------------------------------------------ */

void updategui(void)  {

  forml->Show();
  formb->Show();
  videoramtitle->Show();
  vramb1->Show();
  vramb1->Show();
  vramb2->Show();
  vramb3->Show();
  vramb4->Show();
  vramb5->Show();
  ramprobedb->Show();
  buffertitle->Show();
  bufb1->Show();
  bufb2->Show();
  bufb3->Show();
  bufval->Show();
  i2cmodetitle->Show();
  i2cmode->Show();
  quitsaveb->Show();
  quitabortb->Show();
  statusbar->Show();

  mwin->RebuildClip();
  XFlush(mwin->GetDisplay());
  }

/* ------------------------------------------------------------------------ */

void initgui(void)  {
  int stsz = ((NUM_STANDARDS*13+6+9)/10)*10;
  int i2csz = ((NUM_I2C_MODES*13+6+9)/10)*10;


  videoramtitle = new ITextBox(_("RAM on Card"), mwin, 10, 10, 150, 20);
  vramb1 = new ISButton(_("2Mb"),  mwin, 10,  30, 150, 20);
  vramb2 = new ISButton(_("4Mb"),  mwin, 10,  50, 150, 20);
  vramb3 = new ISButton(_("8Mb"),  mwin, 10,  70, 150, 20);
  vramb4 = new ISButton(_("16Mb"), mwin, 10,  90, 150, 20);
  vramb5 = new ISButton(_("32Mb"), mwin, 10, 110, 150, 20);
  vramb1->SetDownCallback(videoramselect); 
  vramb1->SetUpCallback(no_none_selected); 
  vramb2->SetDownCallback(videoramselect); 
  vramb2->SetUpCallback(no_none_selected); 
  vramb3->SetDownCallback(videoramselect); 
  vramb3->SetUpCallback(no_none_selected); 
  vramb4->SetDownCallback(videoramselect); 
  vramb4->SetUpCallback(no_none_selected); 
  vramb5->SetDownCallback(videoramselect); 
  vramb5->SetUpCallback(no_none_selected); 
  if(videoramsize == 2048 || videoramsize == 0) vramb1->Press(0, 0, 0);
  else if(videoramsize == 4096)  vramb2->Press(0, 0, 0);
  else if(videoramsize == 8192)  vramb3->Press(0, 0, 0);
  else if(videoramsize == 16384) vramb4->Press(0, 0, 0);
  else if(videoramsize == 32768) vramb5->Press(0, 0, 0);

  ramprobedb = new ISButton(_("Gatos will Probe RAM"), mwin, 10, 140, 150, 20);
  ramprobedb->SetDownCallback(videoramprobed); 
  ramprobedb->SetUpCallback(videoramprobed);
  if(ramwillprobed) ramprobedb->Press(0, 0 ,0);

  buffertitle = new ITextBox(_("Buffer size"),   mwin, 10, 175, 150, 20);
  bufb1 = new ISButton(_("600kb for NTSC"),      mwin, 10, 195, 150, 20);
  bufb2 = new ISButton(_("864kb for PAL/SECAM"), mwin, 10, 215, 150, 20);
  bufb3 = new ISButton(_(">>"), mwin, 10, 235, 30, 20);

  bufb1->SetDownCallback(bufferselect); 
  bufb1->SetUpCallback(no_none_selected);
  bufb2->SetDownCallback(bufferselect);
  bufb2->SetUpCallback(no_none_selected);
  bufb3->SetDownCallback(bufferselect);
  bufb3->SetUpCallback(no_none_selected);

  bufval = new IIntBox(0, mwin, 40, 235, 120, 20);
  bufval->SetMin(0);
  bufval->SetMax(32768);
  bufval->SetChangeCallback(chbufval);
  bufval->Disable();

  if(buffersize == 600) {
    bufval->SetVal(buffersize);
    bufb1->Press(0, 0, 0);
    }
  else if(buffersize == 864) {
    bufval->SetVal(buffersize);
    bufb2->Press(0, 0, 0);
    }
  else {
    bufval->SetVal(buffersize);
    bufval->Enable();
    bufb3->Press(0, 0, 0);
    }

  i2cmodetitle = new ITextBox(_("I2C mode selection"), mwin, 10, 270, 150, 20);
  i2cmode = new IListBox((char **)i2c_modes, NUM_I2C_MODES, 
			 mwin, 10, 290, 150, i2csz);
  i2cmode->SetSelCallback(i2cmodeCB); i2cmode->SetDCCallback(i2cmodeCB);
  i2cmode->Select(i2cmodeselected);

  forml = new ITextBox(_("Select Television Type"), mwin, 170, 10, 300, 20);
  formb = new IListBox((char **)tvtype_names, NUM_STANDARDS, 
		       mwin, 170, 30, 300, stsz);
  formb->SetSelCallback(tvtypeCB); formb->SetDCCallback(tvtypeCB);
  formb->Select(standardselected);


  quitsaveb = new IButton(_("Save & Quit"), mwin, 50, iys-50, 150, 20);
  quitsaveb->SetClickCallback(&quitsaveCB); 
  quitabortb = new IButton(_("Abort"), mwin, ixs-200, iys-50, 150, 20);
  quitabortb->SetClickCallback(&quitabortCB); 

  /* 
   * Remember, statusbar ITextBox 
   * is created/destroyed in update_statusbar()
   */

  updategui();
  }

/* ------------------------------------------------------------------------ */

void initvar(void)  {

  tvtype_names[0] =
    /* None   */  _("Autodetect      Autodetect video format");
  tvtype_names[1] =
    /* NTSC   */  _("NTSC            North/South American TV Antenna");
  tvtype_names[2] =
                  _("NTSC            North/South American Cable (CATV)");
  tvtype_names[3] =
                  _("NTSC            Japanese TV Antenna");
  tvtype_names[4] =
                  _("NTSC            Japanese TV Cable (CATV)");
  tvtype_names[5] =
    /* PAL    */  _("PAL-B/G         Cable");
  tvtype_names[6] =
                  _("PAL-B           Europe Antenna");
  tvtype_names[7] =
                  _("PAL-B           Australia");
  tvtype_names[8] =
                  _("PAL-B+G         New Zealand");
  tvtype_names[9] =
                  _("PAL-B+G         Italy");
  tvtype_names[10] =
                  _("PAL-I           Ireland");
  tvtype_names[11] =
                  _("PAL-I           South Africa");
  tvtype_names[12] =
                  _("PAL-I           Angola");
  tvtype_names[13] =
                  _("PAL-B+G/H/I     Many (Europe and others)");
  tvtype_names[14] =
                  _("PAL-D/K         China");
  tvtype_names[15] =
                  _("PAL-D/K+G       Romania");
  tvtype_names[16] =
                  _("PAL-D/K+?       Korea");
  tvtype_names[17] =
                  _("PAL-M           Brazil");
  tvtype_names[18] =
                  _("PAL-N           Paraguay");
  tvtype_names[19] =
                  _("PAL-N+?         Uruguay");
  tvtype_names[20] =
                  _("PAL-combN       Argentina");
  tvtype_names[21] =
    /* SECAM  */  _("SECAM-L         France");
  tvtype_names[22] =
                  _("SECAM-B+G       Italy");
  tvtype_names[23] =
                  _("SECAM-B+G       Greece, Morocco");
  tvtype_names[24] =
                  _("SECAM-D+K       Russia, Vietnam");
  tvtype_names[25] =
                  _("SECAM-K1        Togo, Niger, Congo");

  i2c_modes[0] = _("Gatos autodection");
  i2c_modes[1] = _("Register type A");
  i2c_modes[2] = _("Register type B");
  i2c_modes[3] = _("Rage PRO interface");
  i2c_modes[4] = _("Register type LG");
  i2c_modes[5] = _("MPP/ImpacTV interface");

  load_gatos_configfile();
  load_xfree_configfile();
  }

/* ------------------------------------------------------------------------ */

void save_configfiles()  {

  if(!save_gatos_configfile()) {
    gerr(_("Unable to save %s\n"), CONFIGFILE);
    }

  if(!save_xfree_configfile()) {
    gerr(_("Unable to save %s\n"), xfreeconfigname);
    }
  }

/* ------------------------------------------------------------------------ */

void load_gatos_configfile(void)  {
  FILE *gatosconffile;
  char buf[256], *ln;

  /* Try to open the gatos.conf */
  if((gatosconffile = fopen(CONFIGFILE, "r")) != NULL) {

    gatos_conf_exist = 1;
    ln = fgets(buf, 256, gatosconffile) ;
    while(ln != NULL) {
      if(!strncasecmp(ln, "#videoram", 9)) {
	while(*ln == ' ' && *ln != '\n' && *ln != 0) ++ln;
	if(*ln != '\n' && *ln != 0) ++ln;
	while (*ln != ' ' && *ln != '\t' && *ln != '\n' && *ln != 0) ++ln ;
	if(*ln != '\n' && *ln != 0)  {
	  videoramsize = strtol(ln, &ln, 10) ;
	  ramwillprobed = 1;	
	  }
        }
      else if(!strncasecmp(ln, "videoram", 8)){
	while(*ln == ' ' && *ln != '\n' && *ln != 0) ++ln;
	if(*ln != '\n' && *ln != 0) ++ln;
	while (*ln != ' ' && *ln != '\t' && *ln != '\n' && *ln != 0) ++ln ;
	if(*ln != '\n' && *ln != 0)  {
	  videoramsize = strtol(ln, &ln, 10) ;
	  ramwillprobed = 0;
	  }
        }
      else if(!strncasecmp(ln, "buffermem", 9)){
	while(*ln == ' ' && *ln != '\n' && *ln != 0) ++ln;
	if(*ln != '\n' && *ln != 0) ++ln;
	while (*ln != ' ' && *ln != '\t' && *ln != '\n' && *ln != 0) ++ln ;
	if(*ln != '\n' && *ln != 0)  {
	  buffersize = strtol(ln, &ln, 10) ;
	  }
        }
      else if(!strncasecmp(ln, "format", 6)){
	while(*ln == ' ' && *ln != '\n' && *ln != 0) ++ln;
	if(*ln != '\n' && *ln != 0) ++ln;
	while (*ln != ' ' && *ln != '\t' && *ln != '\n' && *ln != 0) ++ln ;
	if(*ln != '\n' && *ln != 0)  {
	  tvtypeselected = strtol(ln, &ln, 10) ;
	  while((*ln == ' ' || *ln == '\t' || *ln == '#') 
		 && *ln != '\n' && *ln != 0) ++ln;
	  if(*ln != '\n' && *ln != 0)  {
	    if(!strncasecmp(ln, "Standard", 8)){
	      while(*ln != ' ' && *ln != '=' && *ln != '\t' 
		     && *ln != '\n' && *ln != 0) ++ln;
	      while((*ln == ' ' || *ln == '=' || *ln == '\t') 
		     && *ln != '\n' && *ln != 0) ++ln;
	      while ((*ln == ' ' || *ln == ':' ) && *ln != ' ' 
		      && *ln != '\t' && *ln != '\n' && *ln != 0) ++ln ;
	      if(*ln != '\n' && *ln != 0)  {
		standardselected = strtol(ln, &ln, 10) ;
	        }
	      }
	    }
	  }
        }
      else if(!strncasecmp(ln, "#i2c_mode", 9)){
	while(*ln == ' ' && *ln != '\n' && *ln != 0) ++ln;
	if(*ln != '\n' && *ln != 0) ++ln;
	while (*ln != ' ' && *ln != '\t' && *ln != '\n' && *ln != 0) ++ln ;
	if(*ln != '\n' && *ln != 0)  {
	  i2cmodeselected = 0;
	  }
        }
      else if(!strncasecmp(ln, "i2c_mode", 8)){
	while(*ln == ' ' && *ln != '\n' && *ln != 0) ++ln;
	if(*ln != '\n' && *ln != 0) ++ln;
	while (*ln != ' ' && *ln != '\t' && *ln != '\n' && *ln != 0) ++ln ;
	if(*ln != '\n' && *ln != 0)  {
	  i2cmodeselected = strtol(ln, &ln, 10) ;
	  }
        }
      ln = fgets(buf, 256, gatosconffile) ;
      }
    fclose(gatosconffile);
    }
  else {
    gerr(_("Unable to open %s\n"), CONFIGFILE);
    }
  }

/* ------------------------------------------------------------------------ */

int save_gatos_configfile(void)  {
  FILE *gcfg;
  char bck_name[256];
  char strtmp[80];

  /* Don't try to make a backup file if gatos.conf not exist before */
  if(gatos_conf_exist == 1) {
    snprintf(bck_name, sizeof(bck_name), "%s.old", CONFIGFILE);

    /* Make a backup copy */
    if(-1==rename(CONFIGFILE, bck_name)) 
      gerr(_("Can't make backup file %s\n"), bck_name);
    }

  /* Try to open the gatos.conf */
  if((gcfg = fopen(CONFIGFILE, "w")) != NULL) {
    fputs("# Sample config file, install as /usr/local/lib/gatos.conf\n", 
	  gcfg);
    fputs("# and edit/check videoram, buffermem and format.\n", gcfg);
    fputs("\n", gcfg);
    fputs("# Change to (full) amount of video RAM on your card !\n", gcfg);
    if(ramwillprobed == 0) {
      sprintf(strtmp, 
	      "videoram\t%d\t\t# Total size of video memory in kilobytes.\n",
	      videoramsize);
      }
    else {
      sprintf(strtmp, 
	      "#videoram\t%d\t\t# Total size of video memory in kilobytes.\n",
	      videoramsize);
      }
    fputs(strtmp, gcfg);
    fputs("\n", gcfg);
    fputs("# Change to amount of video RAM you've reserved for GATOS !\n", 
	  gcfg);
    fputs("# See also XF86Config.sample.\n", gcfg);

    /* check the reserved memory user defined */
    if(buffersize <= 0 || buffersize >= videoramsize) {
      gerr("Wrong buffer size value, check your choices.\n");
      buffersize = 600;
      }

    sprintf(strtmp,
	    "buffermem\t%d\t\t# Size of reserved video memory in kilobytes.\n",
	    buffersize);
    fputs(strtmp, gcfg);
    fputs("\n", gcfg);
    fputs("# TV signal format.	Country\n", gcfg);
    fputs("#\n", gcfg);
    fputs("#   1	NTSC-M		US and many others\n", gcfg);
    fputs("#   2	NTSC-Japan	Japan\n", gcfg);
    fputs("#   3	PAL-B		Many (Europe)\n", gcfg);
    fputs("#   3	PAL-D		China\n", gcfg);
    fputs("#   3	PAL-G		Many (Europe)\n", gcfg);
    fputs("#   3	PAL-H		Belgium\n", gcfg);
    fputs("#   3	PAL-I		Great Britain and others\n", gcfg);
    fputs("#   4	PAL-M		Brazil (NTSC-like PAL format)\n", 
	  gcfg);
    fputs("#   5	PAL-N		Paraguay, Uruguay\n", gcfg);
    fputs("#   6	SECAM		East Europe, France, Middle East\n", 
	  gcfg);
    fputs("#   7	PAL-Ncomb	Argentina\n", gcfg);
    fputs("#\n", gcfg);

    if(tvtypeselected == 0) {
      sprintf(strtmp,
	      "#format\t\t%d\t# Standard=%d:%s.\n", tvtypeselected, 
	      standardselected, tvtype_names[standardselected]);
      }
    else {
      sprintf(strtmp,
	      "format\t\t%d\t# Standard=%d:%s.\n", tvtypeselected, 
	      standardselected, tvtype_names[standardselected]);
      }

    fputs(strtmp, gcfg);
    fputs("\n", gcfg);
    fputs("# Uncomment and change if automatic I2C mode detection fails:\n", 
	  gcfg);
    fputs("#   1: Register type A: DAC_CNTL+GEN_TEST_CNTL Registers\n", gcfg);
    fputs("#   2: Register type B: GP_IO Register\n", gcfg);
    fputs("#   3: Rage PRO interface (I2C_CNTL_{0,1} registers)\n", gcfg);
    fputs("#   4: Register type LG: GP_IO Register\n", gcfg);
    fputs("#   5: MPP/ImpacTV interface\n", gcfg);
    if(i2cmodeselected == 0) {
      sprintf(strtmp,"#i2c_mode\t?\n");
      }
    else {
      sprintf(strtmp,"i2c_mode\t%d\n", i2cmodeselected);
      }
    fputs(strtmp, gcfg);
    fclose(gcfg);
    return 1;
    }
  else {
    gerr(_("Unable to open %s\n"), CONFIGFILE);
    return -1;
    }

  return 0;
  }

/* ------------------------------------------------------------------------ */

void load_xfree_configfile(void)  {
  FILE *xfreeconffile;
  char buf[256], *ln;
  int endtmp = 0;
  int i, j, k = 0, xfreecfgfound = 0;

  for(i=0; i<256; i++)
    xfreeconf[i] = NULL;


  /* Now, try to open the XF86Config */
  if(xfree_path_overrided) {     // If XF86Config was defined by using -X parm
    strcpy(xfreeconfigname, xfreeconfigfile[3]);
    if((xfreeconffile = fopen(xfreeconfigname, "r")) != NULL) {
      xfreecfgfound = 1;
      }
    }
  else {                                      // Try to find in defaults paths
    for(j=0; j<2; j++) {
      strcpy(xfreeconfigname, xfreeconfigfile[j]);
      if((xfreeconffile = fopen(xfreeconfigname, "r")) != NULL) {
	xfreecfgfound = 1;
	j = 1;
        }
      }
    }

  /* XF86Config found, good. */
  if(xfreecfgfound) {
    xfreeline = 0;
    ln = fgets(buf, 256, xfreeconffile) ;
    while(ln != NULL) {
      xfreeconf[xfreeline] = (char *)malloc(strlen(ln)+1);
      strncpy(xfreeconf[xfreeline], ln, strlen(ln)+1);
      while((*ln == ' ' || *ln == '\t') && *ln != '\n' && *ln != 0) ++ln;

      if(!strncasecmp(ln, "VideoRam", 8)) {
	while((*ln == ' ' || *ln == '\t') 
	       && *ln != '\t' && *ln != '\n' && *ln != 0) ++ln;
	if(*ln != '\n' && *ln != 0) ++ln;
	while (*ln != ' ' && *ln != '\t' && *ln != '\n' && *ln != 0) ++ln ;
  	if(*ln != '\n' && *ln != 0)  {
  	  xfree_videoramsize = strtol(ln, &ln, 10) ;
	  xfree_videoram_line = xfreeline;
	  videoramline[k++] = xfreeline;
	  endtmp = 1;
	  }
        }
      else if(!strncasecmp(ln, "Section", 7)) {
	while (*ln != ' ' && *ln != '\t' && *ln != '\n' && *ln != 0) ++ln;
  	while(*ln == ' ' || *ln == '\t') ++ln;
	if(!strncasecmp(ln, "\"Device\"", 8)) {
	  endtmp = 1;
	  }
        }
      else if(!strncasecmp(ln, "EndSection", 10)
	       && endtmp == 1) {
	display_endsection = xfreeline;
	endtmp = 2;
        }
      
      ln = fgets(buf, 256, xfreeconffile) ;
      xfreeline++;
      }
    fclose(xfreeconffile);
    }
  else {
    if(xfree_path_overrided)
      gerr(_("Unable to open %s\n"), xfreeconfigfile[3]);
    else
      gerr(_("Unable to open %s or %s\n"), 
	   xfreeconfigfile[0], xfreeconfigfile[1]);
    }
  }

/* ------------------------------------------------------------------------ */

int save_xfree_configfile(void)  {
  FILE *xfcfg;
  char bck_name[256];
  char strtmp[80];
  int i, j = 0;

  snprintf(bck_name, sizeof(bck_name), "%s.old", xfreeconfigname);

  /* Make a backup copy */
  if(-1==rename(xfreeconfigname, bck_name)) 
    gerr(_("Can't make backup file %s\n"), bck_name);

  /* Try to open the XF86Config */
  if(xfreeline != -1) { // XF86Config wasn't open/loaded
    if((xfcfg = fopen(xfreeconfigname, "w")) != NULL) {

      for(i=0; i<xfreeline; i++) {
	if(xfree_videoram_line != -1) {
	  if(i == videoramline[j]) {
	    sprintf(strtmp,
		    "   VideoRam        %d\n", (videoramsize - buffersize));
	    fputs(strtmp, xfcfg);
	    i++;
	    j++;
            }
	  }
	else if(i == display_endsection) {
	  sprintf(strtmp,
		  "   VideoRam        %d\n", (videoramsize - buffersize));
	  fputs(strtmp, xfcfg);
          }
	fputs(xfreeconf[i], xfcfg); 
        }
      fclose(xfcfg);
      return 1;
      }
    else {
      gerr(_("Unable to open %s\n"), xfreeconfigname);
      return -1;
      }
    }
  else {
    gerr(_("Unable to open %s\n"), xfreeconfigname);
    return -1;
    }

  return 0;
  }

/* ------------------------------------------------------------------------ */
/* 
 * Check if effective user is root
 */
int check_perms(void)  {
  return (geteuid() == 0);
  }

/* ------------------------------------------------------------------------ */

void usage(void)  {

  gerr(_("Usage: gatos-conf [options]\n"));
  gerr(_("\noptions:\n"));
  gerr(_("  -X:              Set pathname of XF86Config file.\n"));
  gerr(_("                    ex: gatos-conf -X /usr/local/etc/XF86Config\n"));
  gerr(_("  -h,\n"));
  gerr(_("  --help:          Display this help text.\n"));
  exit(0);
  }

/* ------------------------------------------------------------------------ */
