#!/usr/bin/python
# Copyright (C) 2007-2012 - Curtis Hovey <sinzui.is at verizon.net>
# This software is licensed under the GNU General Public License version 2
# (see the file COPYING).
"""A simple doctest runner.

This test app can be run from test.py which sets the default options.
"""

import curses
import doctest
import optparse
import os
import re
import sys
import unittest
from collections import defaultdict
try:
    from unittest.runner import _WritelnDecorator
    _WritelnDecorator != '# Supress redefintion warning.'
except ImportError:
    # Hack support for running the tests in Python 2.6-.
    from unittest import _WritelnDecorator

    class FakeRunner:

        def __init__(self, writelin_decorator):
            self._WritelnDecorator = writelin_decorator

    unittest.runner = FakeRunner(_WritelnDecorator)

from testing import Dummy, SignalTester, literal, proof


class Env(object):
    """The test environment properties."""

    @classmethod
    def get(cls, key, default):
        """Return the key value when exists, or the default."""
        if key in Env.__dict__:
            return Env.__dict__[key]
        else:
            return default


class TPut(object):
    """Terminal colours (tput) utility."""
    _colours = [
        'black', 'blue', 'green', 'white', 'red', 'magenta', 'yellow', 'grey']

    def __init__(self):
        try:
            self.bold = os.popen('tput bold').read()
            self.reset = os.popen('tput sgr0').read()
            for i in range(8):
                cmd = 'tput setf %d' % i
                colour = self._colours[i]
                self.__dict__[colour] = os.popen(cmd).read()
                self.__dict__['bold' + colour] = (
                    self.bold + self.__dict__[colour])
        except IOError:
            # The default values of the styles are safe for all streams.
            self.bold = ''
            self.reset = ''
            for i in range(8):
                colour = self._colours[i]
                self.__dict__[colour] = ''
                self.__dict__['bold' + colour] = ''


class XTermWritelnDecorator(_WritelnDecorator):
    """Decorate lines with xterm bold and colors when suported."""
    _test_terms = re.compile(r'^(File|Failed|Expected|Got)(.*)$', re.M)
    _test_rule = re.compile(r'^(--[-]+)$', re.M)

    def __init__(self, stream):
        """Initialize the stream and setup colors."""
        _WritelnDecorator.__init__(self, stream)
        self.tput = TPut()
        curses.initscr()
        self.has_colors = curses.has_colors()
        curses.endwin()

    def write(self, arg):
        """Write bolded and coloured lines."""
        if not self.has_colors:
            text = arg
        elif arg.startswith('Doctest:') or arg.startswith('Test:'):
            text = '%s%s%s' % (self.tput.blue, arg, self.tput.reset)
        elif arg.startswith('Ran'):
            text = '%s%s%s' % (self.tput.bold, arg, self.tput.reset)
        elif arg.startswith('ERROR') or arg.startswith('FAIL:'):
            text = '%s%s%s' % (self.tput.boldred, arg, self.tput.reset)
        elif arg.endswith('FAIL'):
            text = '%s%s%s' % (self.tput.red, arg, self.tput.reset)
        elif arg.startswith('ok'):
            text = '%s%s%s' % (self.tput.blue, arg, self.tput.reset)
        elif arg.startswith('--') or arg.startswith('=='):
            text = '%s%s%s' % (self.tput.grey, arg, self.tput.reset)
        elif arg.startswith('Traceback'):
            term = r'%s\1\2%s' % (self.tput.red, self.tput.reset)
            text = self._test_terms.sub(term, arg)
            rule = r'%s\1%s' % (self.tput.grey, self.tput.reset)
            text = self._test_rule.sub(rule, text)
        else:
            text = arg
        self.stream.write(text)


def parse_args():
    """Parse the command line arguments and return the options."""
    parser = optparse.OptionParser(
        usage="usage: %prog [%options]")
    parser.add_option(
        "-v", "--verbosity", dest="verbosity", default=0, action='count',
        help="The verbosity of the ouput")
    parser.add_option(
        "-t", "--test", dest="test_pattern", type="string",
        help="A regular expression pattern used to select the testst to run.")
    parser.set_defaults(
        display=Env.get('t', 't'), verbosity=Env.get('verbosity', 1),
        test_pattern='.*')
    return parser.parse_args()


def setup_env(params=None):
    """Setup the test environment.

    This function merges the command line args with Env customizations
    and the default values.
    """
    Env.dir_re = re.compile(params.get('dir_re', '(sourcecode)'))
    if 'paths' in params:
        [sys.path.insert(0, os.path.abspath(path)) for path in params['paths']]
    if 'verbosity' in params:
        Env.verbosity = params['verbosity']

    (options, args) = parse_args()
    Env.verbosity = options.verbosity
    Env.write_decorator = XTermWritelnDecorator
    if len(args) >= 1:
        Env.test_pattern = args[0]
    else:
        Env.test_pattern = options.test_pattern


def get_globs():
    """Return a dictionary of test objects and functions for doctests."""
    return dict(
        Dummy=Dummy, SignalTester=SignalTester,
        literal=literal, proof=proof)


def find_tests(root_dir, skip_dir_re='sourcecode', test_pattern='.*'):
    """Generate a list of matching test files below a directory."""
    file_re = re.compile(r'.*(%s).*' % test_pattern)
    test_re = re.compile(r'(.*\.doctest|test_.*\.py)$')
    for path, subdirs, files in os.walk(root_dir):
        subdirs[:] = [dir for dir in subdirs
                      if skip_dir_re.match(dir) is None]
        if path.endswith('tests'):
            for file_ in files:
                file_path = os.path.join(path, file_)
                if os.path.islink(file_path) or not test_re.match(file_):
                    continue
                if file_re.match(file_path):
                    yield os.path.join(path, file_)


def project_dir():
    """The project directory for this script."""
    script_path = os.path.abspath(sys.argv[0])
    return '/'.join(script_path.split('/')[0:-1])


def setUp(doctest):
    """Gedit has gettext compiled into builtins."""
    import __builtin__
    from gettext import gettext
    __builtin__.__dict__['_'] = gettext


def tearDown(doctest):
    """Remove the gedit's gettext from builtins."""
    import __builtin__
    del __builtin__.__dict__['_']


def main(params=None):
    """Run the specified tests or all.

    Uses an option command line argument that is a regulat expression to
    select a subset of tests to run.
    """
    setup_env(params)
    os.chdir(project_dir())
    os.environ['use_fake_gedit'] = 'true'
    # Format the output.
    unittest.runner._WritelnDecorator = Env.write_decorator
    suite = unittest.TestSuite()
    test_loader = unittest.defaultTestLoader
    doctest.set_unittest_reportflags(doctest.REPORT_NDIFF)
    option_flags = doctest.ELLIPSIS | doctest.NORMALIZE_WHITESPACE
    globs = get_globs()
    doctests = defaultdict(list)
    for test in find_tests('./', Env.dir_re, Env.test_pattern):
        if test.endswith('.doctest'):
            top_dir = test[2:test.find('/', 2, ) - 0]
            doctests[top_dir].append(test)
        else:
            if test.startswith('./testing/'):
                test_module = test[2:-3].replace('/', '.')
            else:
                test_module = test[10:-3].replace('/', '.')
            suite.addTest(test_loader.loadTestsFromName(test_module))
    for top_dir in doctests:
        suite.addTest(doctest.DocFileSuite(
            module_relative=False, setUp=setUp, tearDown=tearDown,
            globs=globs, optionflags=option_flags, *doctests[top_dir]))
    unittest.TextTestRunner(verbosity=Env.verbosity).run(suite)


if __name__ == '__main__':
    main()
