// GENPO - the GENeral Purpose Organ
// Copyright (C) 2003,2004,2008 - Steve Merrony 

/* This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/> or
    write to the Free Software Foundation, 51 Franklin Street, Fifth Floor, 
    Boston, MA 02110-1301, USA
*/
#include <iostream>
using namespace std;

#include <alsa/asoundlib.h>
				 
#include <QAction>
#include <QApplication>
#include <QFileDialog>
#include <QMenuBar>
#include <QMessageBox>
#include <QStatusBar>
#include <QTextStream>
				 
#include "application.h"
#include "customEvents.h"
#include "configParser.h"
#include "console.h"

#include "fileopen.xpm"

// constructor
ApplicationWindow::ApplicationWindow( volatile Console *shared_console )
  : QMainWindow( )
{
  console = (Console *)shared_console;

  /////  First, sort out the menu /////

  QPixmap openIcon; // , saveIcon, printIcon;
  openIcon = QPixmap( fileopen );
  
  fileMenu = menuBar()->addMenu( tr("&File") );
  midiMenu = menuBar()->addMenu( tr("&MIDI") );
  helpMenu = menuBar()->addMenu( tr("&Help") );
  
  openAction  	= new QAction( QIcon( "fileopen.xpm" ), tr("&Open") , this );
  openAction->setShortcut( tr( "Ctrl+O" ) );
  connect( openAction, SIGNAL( triggered() ), this, SLOT( choose() ) );
  reloadAction	= new QAction( tr("&Reload") , this );
  reloadAction->setShortcut( tr( "Ctrl+R" ) );
  connect( reloadAction, SIGNAL( triggered() ), this, SLOT( reload() ) );
  quitAction	= new QAction( tr("&Quit") , this );
  connect( quitAction, SIGNAL( triggered() ), qApp, SLOT(closeAllWindows()) );
  panicAction	= new QAction( tr("&Panic") , this );
  panicAction->setShortcut( tr( "Ctrl+P" ) );
  connect( panicAction, SIGNAL( triggered() ), this, SLOT( midiPanic() ) );
  aboutAction 	= new QAction( tr("&About"), this );
  connect( aboutAction, SIGNAL( triggered() ), this, SLOT( about() ) );
  
  fileMenu->addAction( openAction );
  fileMenu->addAction( reloadAction );
  fileMenu->addSeparator();
  fileMenu->addAction( quitAction );
  midiMenu->addAction( panicAction );
  helpMenu->addAction( aboutAction );
  
  central = new QWidget( this );
  setCentralWidget( central );
  grid = new QGridLayout( central );  
  
  // create widgets we'll need later
  desc = new QLabel( "", central );
  stopGroup = new QButtonGroup( central );  // a single invisible group for ALL the stops
  stopGroup->setExclusive( false );
  pistonGroup = 	new QButtonGroup( central );
  toePistonGroup = 	new QButtonGroup( central );
  couplerGroup = 	new QButtonGroup( central );
  couplerGroup->setExclusive( false );
  
  gcButton = new QPushButton( "G.C. ( )", central );

  seq = &(shared_console->seq);
  op_a = &(shared_console->out_port_a);
  op_b = &(shared_console->out_port_b);
  
  resize( 750, 700 );

  if (console->orgFileName != QString::null) {
    // an organ filename was passed into the program - load it
    load( console->orgFileName );
  }
}


ApplicationWindow::~ApplicationWindow()
{

}


void ApplicationWindow::choose()
{
	QString fn = QFileDialog::getOpenFileName( this, "Load an organ configuration", "", "Organs (*.org)" );
  if ( !fn.isEmpty() )
    load( fn );
  else
    statusBar()->showMessage( "Loading aborted", 2000 );
}


void ApplicationWindow::load( const QString &fileName )
{
	// clean out anything that might already be there
	generalCancel();

	for ( int m = 0; m < console->num_divisions; m++ ) {
		delete divisionLabels[m];
		for ( int s = 0; s < console->divisions[m].num_stops; s++ )   delete stopButtons[m][s];
		for ( int p = 0; p < console->divisions[m].num_pistons; p++ ) {
			delete pistonButtons[m][p];
		}
		if ( console->divisions[m].num_pistons > 0 ) {
			delete pvbox[m];
		}
		if ( console->divisions[m].num_couplers > 0 ) {
			for (int c = 0; c < console->divisions[m].num_couplers; c++ ) delete couplerCheckBoxes[m][c];
			delete cvbox[m];
		}
	}
	if ( console->num_toePistons > 0 ) {
		for (int tp = 0; tp < console->num_toePistons; tp ++ ) delete toePistonButtons[tp];
		delete tphbox;
	}
		
	QFile f( fileName );

	if ( !f.open( QIODevice::ReadOnly ) ) {
		return;
	}
	else {
		QTextStream ts( &f );
		ConfigParser handler;
		QXmlInputSource source( &f );
		QXmlSimpleReader reader;
		reader.setContentHandler( &handler );
		reader.parse( source );

		const char accel_keys[4][10] = {
			{ 'z', 'x', 'c', 'v', 'b', 'n', 'm', ',', '.', '/' },
			{ 'a', 's', 'd', 'f', 'g', 'h', 'j', 'k', 'l', ';' },
			{ 'q', 'w', 'e', 'r', 't', 'y', 'u', 'i', 'o', 'p' },
			{ '1', '2', '3', '4', '5', '6', '7', '8', '9', '0' }
		};
		handler.copyConfig( console );

		desc->setText( console->organName + " (" + console->description + ")" );
		grid->addWidget( desc, 0, 0, 1, -1 , Qt::AlignHCenter ); 	// 1 row, but full width

		QString midiChan;
		int num_pistons = 0;
		int num_couples = 0;
		int num_toe_pistons = 0;

		// console names

		for ( int m = 0; m < console->num_divisions; m++ ) {
			divisionLabels[m] = new QLabel( "X", central );
			divisionLabels[m]->setText( console->divisions[m].name );
			divisionLabels[m]->setAlignment( Qt::AlignHCenter );
			grid->addWidget( divisionLabels[m], 1, m );
		}

		// stops

		for ( int m = 0; m < console->num_divisions; m++ ) {
			for ( int s = 0; s < console->divisions[m].num_stops; s++ ) {
				stopButtons[m][s] = new QPushButton( central );
				stopButtons[m][s]->setMinimumHeight( 15 );
				// insert into the invisible group with a hashed ID
				stopGroup->addButton( stopButtons[m][s], m * MAX_STOPS + s );
				stopButtons[m][s]->setObjectName( console->divisions[m].stops[s].label );
				stopButtons[m][s]->setCheckable( true );

				if ((( console->acceleratorStyle == "QWERTY" ) || ( console->acceleratorStyle != "None" ) ) &&
				        ( m < 4 ) &&
				        ( s < 10 ) ) {  // default accelerators for 1st 10 stops on 1st 4 divisions
					stopButtons[m][s]->setText( console->divisions[m].stops[s].name + "   (&" + accel_keys[m][s] + ")" );
				}
				else {
					stopButtons[m][s]->setText( console->divisions[m].stops[s].name );
				}

				// colouring - stop colour overrides manual colour
				stopPalettes[m][s] = new QPalette();
				if ( console->divisions[m].stops[s].colour != "lightgrey" ) {
					stopPalettes[m][s]->setColor( QPalette::Button, console->divisions[m].stops[s].colour );
				}
				else {
					if ( console->divisions[m].colour != "lightgrey" ) {
						stopPalettes[m][s]->setColor( QPalette::Button, console->divisions[m].colour );
					}
				}
				stopButtons[m][s]->setPalette( *stopPalettes[m][s] );
				
				grid->addWidget( stopButtons[m][s], s + 3, m );
			}
		}

		// one Slot-Handler for all stops
		stopGroup->disconnect( SIGNAL( buttonClicked( int ) ), this , SLOT( changeStop( int ) ) );
		connect( stopGroup, SIGNAL( buttonClicked( int ) ), this, SLOT( changeStop( int ) ) );

		// pistons
		for ( int m = 0; m < console->num_divisions; m++ ) {
			if ( console->divisions[m].num_pistons > 0 ) {
				pvbox[m] = new QGridLayout( ); // no parent!
				for ( int p = 0; p < console->divisions[m].num_pistons; p++ ) {
					pistons[num_pistons].manual = m;
					pistons[num_pistons].piston = p;
					pistonButtons[m][p] = new QPushButton( );
					num_pistons++;
					// insert into the invisible group with a hashed ID
					pistonGroup->addButton( pistonButtons[m][p], m * MAX_PISTONS + p );
					pistonButtons[m][p]->setText( console->divisions[m].pistons[p].label );
					pistonButtons[m][p]->setCheckable( false );
					//pistonButtons[m][p]->setMinimumHeight( 16 );
					pistonButtons[m][p]->setFixedSize( 30, 18 );
					pistonButtons[m][p]->show();
					//QToolTip::add( pistonButtons[m][p], "(Huh?)" );
					pvbox[m]->addWidget((QWidget *)pistonButtons[m][p], p / 3, p % 3 );
					// store the widget ID in the shared structure
					console->divisions[m].pistons[p].widgetID = pistonButtons[m][p];
				}
				grid->addLayout( pvbox[m], 2 + MAX_STOPS, m );
			}
		}
		pistonGroup->disconnect( SIGNAL( buttonClicked( int ) ), this , SLOT( pistonChanged( int ) ) );
		connect( pistonGroup, SIGNAL( buttonClicked( int ) ), this, SLOT( pistonChanged( int ) ) );
		
		// couplers
		for ( int m = 0; m < console->num_divisions; m++ ) {
			if ( console->verboseMode ) cout << "Division " << m << ", couplers: " << console->divisions[m].num_couplers << "\n";
			if ( console->divisions[m].num_couplers > 0 ) {
				cvbox[m] = new QVBoxLayout( );
				for ( int c = 0; c < console->divisions[m].num_couplers; c++ ) {
					couples[num_couples].manual = m;
					couples[num_couples].coupler = c;
					couplerCheckBoxes[m][c] = new QCheckBox( ); 
					couplerCheckBoxes[m][c]->setMinimumHeight( 14 );
					num_couples++;
					// insert into the invisible group with a hashed ID
					couplerGroup->addButton( couplerCheckBoxes[m][c], m * MAX_COUPLERS + c );
					couplerCheckBoxes[m][c]->setText( console->divisions[m].couplers[c].name );
					couplerCheckBoxes[m][c]->setObjectName( console->divisions[m].couplers[c].label );
					cvbox[m]->addWidget(( QWidget * )couplerCheckBoxes[m][c] );
				}

				grid->addLayout( cvbox[m], MAX_STOPS + MAX_PISTONS, m );
			}
		}
		couplerGroup->disconnect( SIGNAL( buttonClicked( int ) ), this , SLOT( couplerChanged( int ) ) );
		connect( couplerGroup, SIGNAL( buttonClicked( int ) ), this, SLOT( couplerChanged( int ) ) );
		
		// Toe Pistons
		if ( console->num_toePistons > 0 ) {
			tphbox = new QHBoxLayout();
			for ( int tp = 0; tp < console->num_toePistons; tp++ ) {
				toePistons[tp].toePiston_ix = tp;
				toePistonButtons[tp] = new QPushButton( );
				num_toe_pistons++;
				// insert into the invisible group with a hashed ID
				toePistonGroup->addButton( toePistonButtons[tp], tp );
				toePistonButtons[tp]->setText( console->toePistons[tp].label );
				toePistonButtons[tp]->setMinimumHeight( 16 );
				toePistonButtons[tp]->setCheckable( false );
				toePistonButtons[tp]->show();
				tphbox->addWidget(( QWidget * )toePistonButtons[tp] );
				console->toePistons[tp].widgetID = toePistonButtons[tp];
			}

			grid->addLayout( tphbox, MAX_STOPS + MAX_PISTONS + MAX_COUPLERS,  0, 1, -1 );
		}
		toePistonGroup->disconnect( SIGNAL( buttonClicked( int ) ), this , SLOT( toePistonChanged( int ) ) );
		connect( toePistonGroup, SIGNAL( buttonClicked( int ) ), this, SLOT( toePistonChanged( int ) ) );
		
		// General Cancel
		grid->addWidget( gcButton, MAX_STOPS + MAX_PISTONS + MAX_COUPLERS + 1, 0 );
		gcButton->setShortcut( Qt::Key_Space );
		gcButton->disconnect( SIGNAL( clicked() ), this, SLOT( generalCancel() ) );
		connect( gcButton, SIGNAL( clicked() ), this, SLOT( generalCancel() ) );

		console->orgFileName = fileName;
		grid->setVerticalSpacing( 3 );
		//    setCaption( APP_NAME + ": " + console.organName );
		statusBar()->showMessage( "Loaded configuration: " + console->organName, 4000 );
	}
}


void ApplicationWindow::closeEvent( QCloseEvent* ce ) {
  ce->accept();
  if (console->verboseMode) cout << "Normal user exit\n";
}

void ApplicationWindow::reload() {

	if (console->orgFileName != QString::null) {
    // remove child widgets	
    for (int m = 0; m<console->num_divisions; m++) {
      for (int s = 0; s<console->divisions[m].num_stops; s++) {
        stopButtons[m][s]->hide();
      }
    }
    load( console->orgFileName );
  }
  else {
    statusBar()->showMessage( "No configuration (.org) file loaded - cannot reload!" );
  }
}

void ApplicationWindow::midiPanic() {

  for (int mc = 0; mc < console->num_out_ports * MIDI_CHANNELS_PER_PORT ; mc++) {
    silenceMidiChannel( mc );
  }
}

void ApplicationWindow::silenceMidiChannel( int mc ) {

  snd_seq_event_t ano_event;
  int rc;

  snd_seq_ev_clear( &ano_event );
  if (mc <MIDI_CHANNELS_PER_PORT) {
  	snd_seq_ev_set_controller( &ano_event, mc, MIDI_CTL_ALL_NOTES_OFF,0 );
  } 
  else {
	  snd_seq_ev_set_controller( &ano_event, mc - 16, MIDI_CTL_ALL_NOTES_OFF,0 );
  } 
  
  snd_seq_ev_set_subs( &ano_event );
  snd_seq_ev_set_direct( &ano_event );
  if (mc <MIDI_CHANNELS_PER_PORT) {
    snd_seq_ev_set_source( &ano_event, op_a->my_port );
  } 
  else {
    snd_seq_ev_set_source( &ano_event, op_b->my_port );
  }

  rc = snd_seq_event_output_direct( seq->seq_handle, &ano_event );
  if (rc < 0) {
    cout << "Error: sending sequencer controller event (" << snd_strerror( rc ) << ")\n";
  }

}

void ApplicationWindow::about()
{
  QMessageBox::about( this, "About GENPO",
		      "<center><b>GENPO</b><br><br>"
					"Version: "  APP_VERSION "<br><br>"
		      		"&copy; 2003 - 2008 Steve Merrony<br><br>"
					"Please see<br>"
		  			"<a href='http://genpo.sourceforge.net/'>http://genpo.sourceforge.net/</a><br>"
				  	"for more information</center>" );
}

void ApplicationWindow::changeStop( int stopIDhash ) {

  int m, s;

  m = stopIDhash / MAX_STOPS;  // manual number
  s = stopIDhash % MAX_STOPS;  // stop number in that manual

  if (stopButtons[m][s]->isChecked()) {
    // handle single-stop mode
    if (console->singleStopMode) {
      generalCancel();
      this->update();
    }
    pullStop( m, s );
  }
  else {
    pushStop( m, s );
  }

} 

void ApplicationWindow::changePiston( int m, int pc ) {

	// do nothing if there are no pistons for this manual
	//if ( (m<=console->num_divisions) && (console->divisions[m].num_pistons > 0) && (pc <=console->divisions[m].num_pistons)) {
	if ( m <= console->num_divisions )
		if ( console->divisions[m].num_pistons > 0 )
			if ( pc <= console->divisions[m].num_pistons ) {
				//dumpVoices();
				// clear all stops on this manual
				for ( int s = 0; s < console->divisions[m].num_stops; s++ ) {
					if ( stopButtons[m][s]->isChecked() ) {
						stopButtons[m][s]->setChecked( FALSE );
						pushStop( m, s );
					}
				}

				//dumpVoices();

				// pull the stops for this piston
				for ( int p = 0; p < console->divisions[m].pistons[pc].numPistonStops; p++ ) {
					for ( int s = 0; s < console->divisions[m].num_stops; s++ ) {
						if ( QString::compare( console->divisions[m].pistons[pc].pistonStops[p].stopLabel,
						                       console->divisions[m].stops[s].label
						                     ) == 0 ) {
							stopButtons[m][s]->setChecked( TRUE );
							pullStop( m, s );
						}
					}
				}
			}

}

void ApplicationWindow::pistonChanged( int pistonIDhash ) {

  int m, p;
  
  m = pistonIDhash / MAX_PISTONS;
  p = pistonIDhash % MAX_PISTONS;
  
  changePiston( m, p );

  // clear the button
  pistonButtons[m][p]->setChecked( false ); 

  //printf( "Done\t" );
  //dumpVoices();

}

void ApplicationWindow::changeToePiston( int tp_ix ) {
  
  generalCancel();
  
  // pull the stops for this piston
  for (int p = 0; p < console->toePistons[tp_ix].numPistonStops; p++) {
    for (int m = 0; m < console->num_divisions; m++) {
      for (int s = 0; s < console->divisions[m].num_stops; s++) {
	if  ( QString::compare( console->toePistons[tp_ix].pistonStops[p].stopLabel,
	    console->divisions[m].stops[s].label
			    ) == 0 ) {
	      stopButtons[m][s]->setChecked( TRUE );
	      pullStop( m, s );
			    }
      }
    }
  }

}

void ApplicationWindow::toePistonChanged( int tpix ) {

  changeToePiston( tpix );

  // clear the button
  toePistonButtons[tpix]->setChecked(FALSE); 

  //printf( "Done\t" );
  //dumpVoices();

}

void ApplicationWindow::couplerChanged( int couplerIDhash ) {

	int m, c;
	
	m = couplerIDhash / MAX_COUPLERS;
	c = couplerIDhash % MAX_COUPLERS;
	
  //couple_t *c = (couple_t *)coupler;

  if (console->verboseMode) cout << "Coupler " << c <<" on manual " << m << " changed\n";

  console->divisions[m].couplers[c].active = !console->divisions[m].couplers[c].active;

}

void ApplicationWindow::generalCancel() {

	// stops and coupler boxes
	for ( int m = 0; m < console->num_divisions; m++ ) {
		for ( int s = 0; s < console->divisions[m].num_stops; s++ ) {
			stopButtons[m][s]->setChecked( false );
			pushStop( m, s );
		}

		for ( int c = 0; c < console->divisions[m].num_couplers; c++ ) {
			couplerCheckBoxes[m][c]->setChecked( FALSE );
			console->divisions[m].couplers[c].active = FALSE;
		}
	}
}

void ApplicationWindow::pullStop( int m, int s ) { // activate a stop

  int rc, man_channel, new_channel;
  snd_seq_event_t pc_event;
  QString msg;
  
  console->divisions[m].stops[s].pulled = TRUE;

  man_channel = console->divisions[m].midiChannel;
  //  dumpVoices();
  // find 1st free slot for a new voice
  new_channel = 0;
  while (new_channel<=console->num_out_ports*MIDI_CHANNELS_PER_PORT && console->voices[new_channel].active) new_channel++;

  //printf( "Candidate channel for new voice: %d\n", new_channel );
  if (new_channel >= console->num_out_ports*MIDI_CHANNELS_PER_PORT) {
    stopButtons[m][s]->setChecked( FALSE );
    if (console->num_out_ports == 0)
      msg = "Warning: Too many stops pulled - GENPO thinks no synthesizers are connected!";
    else
      msg = "Warning: Too many stops pulled (16 per connected synthesizer are allowed).";
    statusBar()->showMessage( msg );
    cout << qPrintable( msg ) << endl;
  }
  else {
    statusBar()->clearMessage();
    console->voices[new_channel].active = TRUE;
    console->voices[new_channel].manual_channel = man_channel;
    console->voices[new_channel].velocity = console->divisions[m].stops[s].velocity;
    console->voices[new_channel].synth = 0;  // *** JUST using 0 for now ***<<<<<<<<<
    console->voices[new_channel].bank = console->divisions[m].stops[s].midiBank;
    console->voices[new_channel].voice = console->divisions[m].stops[s].midiVoice;

    // store the assigned voice in the stop
    console->divisions[m].stops[s].kbdMidiChannel = man_channel;
    console->divisions[m].stops[s].voiceIndex = new_channel;

    // send a MIDI event to change the program for this voice (channel)
    snd_seq_ev_clear( &pc_event );

    if ( new_channel < MIDI_CHANNELS_PER_PORT) {
      snd_seq_ev_set_controller( &pc_event, new_channel, 0, console->divisions[m].stops[s].midiBank );
      snd_seq_ev_set_subs( &pc_event );
      snd_seq_ev_set_direct( &pc_event );
      snd_seq_ev_set_source( &pc_event, op_a->my_port );
    }
    else {
      snd_seq_ev_set_controller( &pc_event, new_channel - MIDI_CHANNELS_PER_PORT, 0, console->divisions[m].stops[s].midiBank );
      snd_seq_ev_set_subs( &pc_event );
      snd_seq_ev_set_direct( &pc_event );
      snd_seq_ev_set_source( &pc_event, op_b->my_port );
    }

    rc = snd_seq_event_output_direct( seq->seq_handle, &pc_event );
    if (rc < 0) {
      cout << "Error: sending sequencer controller event (" << snd_strerror( rc ) << ")\n";
    }   

	if (new_channel < MIDI_CHANNELS_PER_PORT)
    	snd_seq_ev_set_pgmchange( &pc_event, new_channel, console->divisions[m].stops[s].midiVoice );
	else
		snd_seq_ev_set_pgmchange( &pc_event, new_channel - 16, console->divisions[m].stops[s].midiVoice );
		
    rc = snd_seq_event_output_direct( seq->seq_handle, &pc_event );
    if (rc < 0) {
      cout << "Error: sending sequencer pgm change event (" << snd_strerror( rc ) << ")\n";
    }

    if (console->verboseMode) cout << "Voice " << console->divisions[m].stops[s].midiVoice << " set on channel " << new_channel << "\n";

  }
}

void ApplicationWindow::pushStop( int m, int s ) {  // turn off a stop

  int old_voice = console->divisions[m].stops[s].voiceIndex;

  // this if should not be required it stops a crash that shouldn't happen <===========================
  if (old_voice >= 0 && old_voice < console->num_out_ports*MIDI_CHANNELS_PER_PORT) {
    console->divisions[m].stops[s].pulled = FALSE;

    console->voices[old_voice].active = FALSE;
    console->voices[old_voice].manual_channel = FREE_CHANNEL;
    console->divisions[m].stops[s].kbdMidiChannel = FREE_CHANNEL;
    console->divisions[m].stops[s].kbdMidiChannel = FREE_CHANNEL;
  
    // turn all notes off on the Voice
    silenceMidiChannel( old_voice );
  }
  if (console->verboseMode) cout << "PushStop( " << m << ", " << s <<" )\n";

}

void ApplicationWindow::dumpVoices() {

  for (int c = 0; c < console->num_out_ports*MIDI_CHANNELS_PER_PORT; c++) {
    cout << c << ":" << console->voices[c].active << "/" << console->voices[c].manual_channel << "/" << console->voices[c].voice << ", ";
  }
  cout << "\n";
}

