#include "vektor.h"
#include "printer.h"
#include "parser.h"
#include "gfanapplication.h"
#include "minkowskisum.h"
#include "newtonpolytope.h"
#include "buchberger.h"
#include "wallideal.h"
#include "lp.h"
#include "tropical.h"
#include "division.h"
#include "bergman.h"
#include "tropical2.h"
#include "dimension.h"
#include "timer.h"
#include "log.h"

class TropicalTraverseApplication : public GFanApplication
{
  //  SimpleOption optionPerformanceTimer;
  SimpleOption optionSymmetry;
  SimpleOption optionNoIncidencePrinting;
  //  SimpleOption optionInitialIdealPrinting;
  //  SimpleOption optionLargeDimensionalPrinting;
public:
  bool includeInDefaultInstallation()
  {
    return true;
  }
  const char *helpText()
  {
    return "This program computes a polyhedral representation of the tropical variety of a homogeneous prime ideal $I$. Let $d$ be the Krull dimension of $I$ and let $\\omega$ be a relative interior point of $d$-dimensional Groebner cone contained in the tropical variety. The input for this program is a pair of marked reduced Groebner bases with respect to the term order represented by $\\omega$, tie-broken in some way. The first one is for the initial ideal $in_\\omega(I)$ the second one for $I$ itself. The pair is the starting point for a traversal of the $d$-dimensional Groebner cones contained in the tropical variety. If the ideal is not prime but with the tropical variety still being pure $d$-dimensional the program will only compute a codimension $1$ connected component of the tropical variety. By default the output is a list of properties of the ideal together with a polyhedral representation of the variety including cone descriptions in terms of ray incidence lists. Note that there are other quicker representations if you are not interested in this detailed description, see --noincidence.\n";
  }
  TropicalTraverseApplication():
    //    optionPerformanceTimer("-T","Enable performance timer"),
    optionSymmetry("--symmetry","Do computations up to symmetry and group the output accordingly. If this option is used the program will read in a list of generators for a symmetry group after the pair of Groebner bases have been read. Two advantages of using this option is that the output is nicely grouped and that the computation can be done faster. For best performance you should also specify --noincidence."),
    optionNoIncidencePrinting("--noincidence","Disables printing of incidence information. This will avoid extracting the full fan from its orbits and avoid computation of the lower dimensional faces. The output is a list of orbits of maximal cones and a list of orbits of codimension 1 cones.")
    // optionInitialIdealPrinting("--initialideals","Print the initial ideals during incidence printing ONLY without --noincidence."),
    //    optionLargeDimensionalPrinting("--largedimensional","Print the full dimensional cones and the codimension 1 cones computed in the traversal.")
  {
    registerOptions();
  }
  char *name()
  {
    return "_tropicaltraverse";
  }
  int main()
  {
    PolynomialSetList tropical;

    //    lpSetSolver("cddgmp");
    FileParser P(Stdin);

    AsciiPrinter p(Stdout);
    PolynomialSet coneGroebnerBasis=P.parsePolynomialSetWithRing();
    PolynomialSet idealGroebnerBasis=P.parsePolynomialSet(coneGroebnerBasis.getRing());
    coneGroebnerBasis.changeNumberOfVariables(idealGroebnerBasis.numberOfVariablesInRing());
    
    SymmetryGroup s(idealGroebnerBasis.numberOfVariablesInRing());
    if(optionSymmetry.getValue())
      {
	IntegerVectorList generators=P.parseIntegerVectorList();
	assertSymmetriesMatch(generators,idealGroebnerBasis);
	s.computeClosure(generators);
	log2 s.print(Stderr);
	log2 fprintf(Stderr,"\n");
      }
    

    int n=idealGroebnerBasis.numberOfVariablesInRing();
    PolyhedralCone hspace=homogeneitySpace(idealGroebnerBasis);
    IntegerVectorList hv=hspace.dualCone().getEquations();
    int h=hv.size();
    int d=krullDimension(idealGroebnerBasis);

    log1 fprintf(Stdout,"Ambient dimension: %i\n",n);
    log1 fprintf(Stdout,"Dimension of homogeneity space: %i\n",h);
    log1 fprintf(Stdout,"Dimension of tropical variety: %i\n",d);
    //    fprintf(Stdout,"Homogeneity space:\n");
    BergmanFan f=bergman(coneGroebnerBasis,idealGroebnerBasis,&s);
    f.computeMultiplicities();
    log1 fprintf(Stdout,"Is simplicial: %s\n",f.isSimplicial()?"true":"false");
    log1 fprintf(Stdout,"Order of input symmetry group: %i\n",s.elements.size());
    log1 fprintf(Stdout,"Number of maximal cones: %i\n",f.numberOfMaximalCones());
    log1 fprintf(Stdout,"Modulo the homogeneity space:\n");
    log1 p.printVectorList(hv);


    if(optionNoIncidencePrinting.getValue())
      f.print(p);
    else
      {
	log1 fprintf(Stderr,"Converting representation to a polyhedral complex modulo symmetry...\n");
	PolyhedralFan p1=f.toPolyhedralFan();
	log1 fprintf(Stderr,"Done converting representation to a polyhedral complex modulo symmetry.\n");
	
	p1.removeAllLowerDimensional();
	

	AsciiPrinter Q(Stdout);
	//	f.print(Q);
	//	p1.print(&Q);
	
	p1.printWithIndices(&Q,true,&s,optionSymmetry.getValue());
      }
    /*    if(optionInitialIdealPrinting.getValue())
      {
	//	    AsciiPrinter p(Stderr);
	PolyhedralFan p1=f.toPolyhedralFan();
	int n=p1.getAmbientDimension();
	IncidenceList a=p1.getIncidenceList(&s);
	fprintf(Stderr,"Computing rays...\n");
	IntegerVectorList rays=p1.getRaysInPrintingOrder(&s);
	fprintf(Stderr,"Done computing rays.\n");
	p.printString("Rays:\n");
	p.printVectorList(rays);
	vector<IntegerVector> rays2(rays.size());
	int K=0;
	for(IntegerVectorList::const_iterator k=rays.begin();k!=rays.end();k++)
	  rays2[K++]=*k;

	for(IncidenceList::const_iterator j=a.begin();j!=a.end();j++)
	  {
	    p.printInteger(j->first);
	    for(IntegerVectorList::const_iterator i=j->second.begin();i!=j->second.end();i++)
	      {
		p.printVector(*i);
		IntegerVector v(n);
		for(int t=0;t<i->size();t++)
		  {
		    v+=rays2[(*i)[t]];
		  }
		p.printVector(v);
		PolynomialSet g2=idealGroebnerBasis;
		buchberger(&g2,WeightReverseLexicographicTermOrder(v));
		g2=initialFormsAssumeMarked(g2,v);
		g2=saturatedIdeal(g2);
		p.printPolynomialSet(g2);
		    
	      }
	  }
	  }*/
    
    //    if(optionPerformanceTimer.getValue())Timer::printList();
    return 0;
  }
};

static TropicalTraverseApplication theApplication;
