/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.jmh;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import com.gs.collections.impl.list.Interval;
import com.gs.collections.impl.list.mutable.FastList;
import org.junit.Assert;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.GenerateMicroBenchmark;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.TearDown;
import org.openjdk.jmh.annotations.Warmup;

@State(Scope.Thread)
@BenchmarkMode(Mode.Throughput)
@OutputTimeUnit(TimeUnit.SECONDS)
public class AnySatisfyTest
{
    private static final int SIZE = 1_000_000;
    private static final int BATCH_SIZE = 10_000;
    private final List<Integer> integersJDK = new ArrayList<>(Interval.oneTo(SIZE));
    private final FastList<Integer> integersGSC = new FastList<>(Interval.oneTo(SIZE));

    private ExecutorService executorService;

    @Setup
    public void setUp()
    {
        this.executorService = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors());
    }

    @TearDown
    public void tearDown() throws InterruptedException
    {
        this.executorService.shutdownNow();
        this.executorService.awaitTermination(1L, TimeUnit.SECONDS);
    }

    @GenerateMicroBenchmark
    public void short_circuit_middle_serial_lazy_jdk()
    {
        Assert.assertTrue(this.integersJDK.stream().anyMatch(each -> each > SIZE / 2));
    }

    @GenerateMicroBenchmark
    public void process_all_serial_lazy_jdk()
    {
        Assert.assertFalse(this.integersJDK.stream().anyMatch(each -> each < 0));
    }

    @Warmup(iterations = 20)
    @Measurement(iterations = 10)
    @GenerateMicroBenchmark
    public void short_circuit_middle_serial_eager_gsc()
    {
        Assert.assertTrue(this.integersGSC.anySatisfy(each -> each > SIZE / 2));
    }

    @Warmup(iterations = 20)
    @Measurement(iterations = 10)
    @GenerateMicroBenchmark
    public void process_all_serial_eager_gsc()
    {
        Assert.assertFalse(this.integersGSC.anySatisfy(each -> each < 0));
    }

    @GenerateMicroBenchmark
    public void short_circuit_middle_serial_lazy_gsc()
    {
        Assert.assertTrue(this.integersGSC.asLazy().anySatisfy(each -> each > SIZE / 2));
    }

    @GenerateMicroBenchmark
    public void process_all_serial_lazy_gsc()
    {
        Assert.assertFalse(this.integersGSC.asLazy().anySatisfy(each -> each < 0));
    }

    @GenerateMicroBenchmark
    public void short_circuit_middle_serial_eager_scala()
    {
        AnySatisfyScalaTest.short_circuit_middle_serial_eager_scala();
    }

    @GenerateMicroBenchmark
    public void process_all_serial_eager_scala()
    {
        AnySatisfyScalaTest.process_all_serial_eager_scala();
    }

    @GenerateMicroBenchmark
    public void short_circuit_middle_serial_lazy_scala()
    {
        AnySatisfyScalaTest.short_circuit_middle_serial_lazy_scala();
    }

    @GenerateMicroBenchmark
    public void process_all_serial_lazy_scala()
    {
        AnySatisfyScalaTest.process_all_serial_lazy_scala();
    }

    @GenerateMicroBenchmark
    public void short_circuit_middle_parallel_lazy_jdk()
    {
        Assert.assertTrue(this.integersJDK.parallelStream().anyMatch(each -> each == SIZE / 2 - 1));
    }

    @GenerateMicroBenchmark
    public void process_all_parallel_lazy_jdk()
    {
        Assert.assertFalse(this.integersJDK.parallelStream().anyMatch(each -> each < 0));
    }

    @GenerateMicroBenchmark
    public void short_circuit_middle_parallel_lazy_gsc()
    {
        Assert.assertTrue(this.integersGSC.asParallel(this.executorService, BATCH_SIZE).anySatisfy(each -> each == SIZE / 2 - 1));
    }

    @GenerateMicroBenchmark
    public void process_all_parallel_lazy_gsc()
    {
        Assert.assertFalse(this.integersGSC.asParallel(this.executorService, BATCH_SIZE).anySatisfy(each -> each < 0));
    }

    @GenerateMicroBenchmark
    public void short_circuit_middle_parallel_lazy_scala()
    {
        AnySatisfyScalaTest.short_circuit_middle_parallel_lazy_scala();
    }

    @GenerateMicroBenchmark
    public void process_all_parallel_lazy_scala()
    {
        AnySatisfyScalaTest.process_all_parallel_lazy_scala();
    }
}
