//*****************************************************************************
//                             PnlNgSpiceAC.cpp                               *
//                            ------------------                              *
// Started     : 04/08/2004                                                   *
// Last Update : 13/04/2010                                                   *
// Copyright   : (C) 2004 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "ngspice/panels/PnlNgSpiceAC.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( PnlNgSpiceAC, PnlAnaBase )

  EVT_RADIOBOX( ID_RBX_SCALE,   PnlNgSpiceAC::OnScale   )
  EVT_CHOICE  ( ID_CHO_SRCNAME, PnlNgSpiceAC::OnSrcName )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.

PnlNgSpiceAC::PnlNgSpiceAC( wxWindow * poWin ) : PnlAnaBase( poWin )
{
  bSetSimrEng( eSIMR_NGSPICE );
  bSetAnaType( eCMD_AC );

  Create( );  // Create the analysis panel
  bClear( );  // Clear all object attributes
}

//*****************************************************************************
// Destructor.

PnlNgSpiceAC::~PnlNgSpiceAC( )
{
}

//*****************************************************************************
// Create the display objects.

void  PnlNgSpiceAC::Create( void )
{
  int  ix, iy;

  // Disable the checkboxes for the parameters NG-Spice cannot calculate
  m_oCbxCurrent.Disable( );
  m_oCbxPower  .Disable( );
  m_oCbxResist .Disable( );

  // Set the frequency sweep parameter labels
  m_oSbxSwpPars.GetSize( &ix, &iy );
  m_oSbxSwpPars.SetSize(  ix, 167 );
  m_oSbxSwpPars.SetLabel( wxT(" AC Sweep ") );
  m_oPnlStart  .bSetName( wxT("Start Frequency") );
  m_oPnlStop   .bSetName( wxT("Stop Frequency") );

  // Set sweep parameter spin control parameters and units
  m_oPnlStart.bSetUnitsType( eUNITS_FREQ );
  m_oPnlStop .bSetUnitsType( eUNITS_FREQ );
  m_oPnlStep .bSetVarType( SpinCtrl::eVAR_INT );
  m_oPnlStep .bShowUnits( FALSE );

  // Create the scale controls but disable the scale option NG-Spice doesn't
  // support
  PnlAnaBase::CreateScale( );
  m_oRbxScale.Enable( eSCALE_LOG, FALSE );

  PnlAnaBase::CreateCpxPrt( );  // Create the simulation parameter complex part check boxes
  PnlAnaBase::CreateSrc( );     // Create input source controls
  PnlAnaBase::CreateTemp( );    // Create the analysis temperature controls
}

//*****************************************************************************
// Initialize the step scale.

void  PnlNgSpiceAC::InitScale( void )
{
  switch( m_oRbxScale.GetSelection( ) )
  {
    case eSCALE_LIN :
      m_oPnlStep.bSetName( wxT("Steps in Total") );
      m_oPnlStep.bSetParms( 10, 1, 100000, 1, 100 );
      break;

    case eSCALE_DEC :
      m_oPnlStep.bSetName( wxT("Steps / Decade") );
      m_oPnlStep.bSetParms( 10, 1, 10000, 1, 100 );
      break;

    case eSCALE_OCT :
      m_oPnlStep.bSetName( wxT("Steps / Octave") );
      m_oPnlStep.bSetParms( 10, 1, 10000, 1, 100 );
      break;

    default :
      break;
  }
}

//*****************************************************************************
// Set the state of the step scale radio box.
//
// Argument List :
//   eScale - The enumerated scale specifier
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlNgSpiceAC::bSetScale( eScaleType eScale )
{
  if( eScale<eSCALE_FST || eScale>eSCALE_LST )      return( FALSE );
#if wxCHECK_VERSION( 2,8,0 )
  if( m_oRbxScale.GetCount( ) < (uint) eScale + 1 ) return( FALSE );
#else
  if( m_oRbxScale.GetCount( ) < (int)  eScale + 1 ) return( FALSE );
#endif

  m_oRbxScale.SetSelection( (int) eScale );

  InitScale( );

  return( TRUE );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlNgSpiceAC::bClear( void )
{
  bool  bRtn=TRUE;

  // Clear the base class
  if( ! PnlAnaBase::bClear( ) )   bRtn = FALSE;

  // Set the sweep parameters to their defaults
  m_oPnlStart.bSetValue( (float) 1.0 );
  m_oPnlStep .bSetValue( (float) 10.0 );
  m_oPnlStart.bSetUnits( wxT("kHz") );
  m_oPnlStop .bSetUnits( wxT("kHz") );

  // Set default scale value
  if( ! bSetScale( eSCALE_DEC ) ) bRtn = FALSE;

  // Set the complex part check box default values
  m_oCbxMag  .SetValue( TRUE );
  m_oCbxPhase.SetValue( FALSE );
  m_oCbxReal .SetValue( FALSE );
  m_oCbxImag .SetValue( FALSE );
  m_oCbxMagDb.SetValue( TRUE );

  return( bRtn );
}

//*****************************************************************************
// Load information from a simulation object.
//
// Argument List :
//   roSim - A simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlNgSpiceAC::bLoad( SimnNgSpice & roSimn )
{
  bool  bRtn=TRUE;

  // Load components into the source choice box
  PnlAnaBase::LoadSrcNames( roSimn.m_oaCpnts, wxT("VI") );

  // Set the source component
  if( ! PnlAnaBase::bSetSrcCpnt( roSimn.m_oCpntSwpSrc ) )    bRtn = FALSE;

  // Set the sweep values
  if( ! m_oPnlStart.bSetValue( roSimn.m_oCmdAC.m_osStart ) ) bRtn = FALSE;
  if( ! m_oPnlStop .bSetValue( roSimn.m_oCmdAC.m_osStop  ) ) bRtn = FALSE;
  if( ! m_oPnlStep .bSetValue( roSimn.m_oCmdAC.m_osStep  ) ) bRtn = FALSE;
  if( m_oPnlStart.dfGetValue( ) == 0.0 )
        m_oPnlStart.bSetUnits( m_oPnlStop.rosGetUnits( ) );

  // Set the step scale (do this before setting the sweep step)
  m_oRbxScale.SetSelection( roSimn.m_oCmdAC.m_eScale );

  if( roSimn.eGetAnaType( ) == eCMD_AC )
  {
    // Set the complex parts to derive
    m_oCbxMag  .SetValue( roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_MAG   ] );
    m_oCbxPhase.SetValue( roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_PHASE ] );
    m_oCbxReal .SetValue( roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_REAL  ] );
    m_oCbxImag .SetValue( roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_IMAG  ] );
    m_oCbxMagDb.SetValue( roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_MAGDB ] );
  }

  // Set the analysis temperature
  if( ! m_oPnlTemp.bSetValue( roSimn.m_oCmdOPT.m_osTEMP ) )  bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Save information to a simulation object.
//
// Argument List :
//   roSimn - A simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlNgSpiceAC::bSave( SimnNgSpice & roSimn )
{
  wxString  os1;
  float     f1;
  size_t    sz1;
  bool      b1;

  m_osErrMsg.Empty( );

  // Check the start frequency (can't set a frequency of zero so set one as
  // small as possible, 1.4e-45 is as small as I can go as a float but it will
  // be converted to a string in engineering units so use 1.0e-15 = 1.0f)
  ConvertType::bStrToFlt( roSimn.m_oCmdAC.m_osStart, &f1 );
  if( f1 == 0.0 ) roSimn.m_oCmdAC.m_osStart = wxT("1.0e-15");

  // Set the sweep values
  roSimn.m_oCmdAC.m_osStart = m_oPnlStart.rosGetValue( );
  roSimn.m_oCmdAC.m_osStop  = m_oPnlStop .rosGetValue( );
  roSimn.m_oCmdAC.m_osStep  = m_oPnlStep .rosGetValue( );

  // Set the sweep scale
  roSimn.m_oCmdAC.m_eScale = (eScaleType) m_oRbxScale.GetSelection( );

  // Set the sweep source component
  if( m_oChoSrcName.GetStringSelection( ) == wxT("None") )
    SetErrMsg( wxT("No source component has been selected.") );
  else if( m_oPnlSrcLvl.dfGetValue( ) == 0.0 )
    SetErrMsg( wxT("Signal source component value of zero is not permitted.") );
  else
  {
    os1 = m_oChoSrcName.GetStringSelection( );
    roSimn.m_oCpntSwpSrc = roSimn.NetList::roGetCpnt( os1 );
    os1 = wxT("AC ") + m_oPnlSrcLvl.rosGetValue( );
    roSimn.m_oCpntSwpSrc.bSetValue( os1 );
  }

  // Set the analysis type
  roSimn.m_oCmdPR.m_eAnaType = eCMD_AC;

  // Store the parameters to derive
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_VLT ] = m_oCbxVoltage.GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_CUR ] = m_oCbxCurrent.GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_PWR ] = m_oCbxPower  .GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_RES ] = m_oCbxResist .GetValue( );

  // Store the complex parts of the parameters to derive
  roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_MAG   ] = m_oCbxMag  .GetValue( );
  roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_PHASE ] = m_oCbxPhase.GetValue( );
  roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_REAL  ] = m_oCbxReal .GetValue( );
  roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_IMAG  ] = m_oCbxImag .GetValue( );
  roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_MAGDB ] = m_oCbxMagDb.GetValue( );

  // Set the analysis temperature
  roSimn.m_oCmdOPT.m_osTEMP = m_oPnlTemp.rosGetValue( );

  // Create the command strings
  roSimn.m_oCmdAC.bFormat( );
  roSimn.m_oCmdPR.bFormat( );

  // Check for errors
  if( ! roSimn.m_oCmdAC.bIsValid( ) )
    SetErrMsg( roSimn.m_oCmdAC.rosGetErrMsg( ) );
  if( ! roSimn.m_oCmdPR.bIsValid( ) )
    SetErrMsg( roSimn.m_oCmdPR.rosGetErrMsg( ) );
  for( sz1=eCPX_MAG, b1=FALSE; sz1<=eCPX_IMAG; sz1++ )
    if( roSimn.m_oCmdPR.m_bCpxPrts[ sz1 ] ) b1 = TRUE;
  if( ! b1 ) SetErrMsg( wxT("No complex parts have been selected.") );

  return( bIsOk( ) );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// Step scale radio box event handler.
//
// Argument List :
//   roEvtCmd - An object holding information about the event

void  PnlNgSpiceAC::OnScale( wxCommandEvent & roEvtCmd )
{
  InitScale( );
}

//*****************************************************************************
// Source component choice box event handler.
//
// Argument List :
//   roEvtCmd - An object holding information about the event

void  PnlNgSpiceAC::OnSrcName( wxCommandEvent & roEvtCmd )
{
  wxString  os1;

  // Set the units type
  os1 = m_oChoSrcName.GetStringSelection( );
  m_oPnlSrcLvl.bSetUnitsType( Component::eGetUnitsType( os1 ) );

  // Set the source value
  if( m_oChoSrcName.GetStringSelection( ) == wxT("None") )
    m_oPnlSrcLvl.bSetValue( (double) 0.0 );
  else if( m_oPnlSrcLvl.dfGetValue( ) == 0.0 )
    m_oPnlSrcLvl.bSetValue( (double) 1.0 );

  // Permit the base class to process this event as well
  roEvtCmd.Skip( );
}

//*****************************************************************************
