/*
    Copyright 2011-2016 Thibaut Paumard

    This file is part of Gyoto.

    Gyoto is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Gyoto is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Gyoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "GyotoRegister.h"
#include "GyotoUtils.h"
#include "GyotoAstrobj.h"
#include "GyotoMetric.h"
#include "GyotoSpectrum.h"
#include "GyotoSpectrometer.h"
#include "GyotoConverters.h"

#include <dlfcn.h>
#include <cstdlib>
#include <string>
#include <iostream>
#include <vector>

using namespace Gyoto;
using namespace std;

typedef void GyotoInitFcn();

void Gyoto::loadPlugin(char const*const name, int nofail) {
  string dlfile = "libgyoto-" ;
  dlfile += name ;
  dlfile += "." ;
  dlfile += GYOTO_PLUGIN_SFX ;
  string dlfunc = "__Gyoto";
  dlfunc += name ;
  dlfunc += "Init";
  void* handle = NULL;
  GyotoInitFcn* initfcn = NULL;
  char * err = NULL;

  GYOTO_DEBUG << "Loading plug-in: " << name <<endl;
  GYOTO_DEBUG << "Trying to dlopen " << dlfile << "...\n";
  handle = dlopen(dlfile.c_str(), RTLD_LAZY | RTLD_GLOBAL);

  std::vector<std::string> plug_path;
  plug_path.push_back(GYOTO_PKGLIBDIR "/");
  plug_path.insert(plug_path.begin(), plug_path[0] + GYOTO_SOVERS "/");
# if defined GYOTO_LOCALPKGLIBDIR
  plug_path.insert(plug_path.begin(), GYOTO_LOCALPKGLIBDIR "/");
  plug_path.insert(plug_path.begin(), plug_path[0] + GYOTO_SOVERS "/");
# endif
  std::vector<std::string>::iterator cur = plug_path.begin();
  std::vector<std::string>::iterator end = plug_path.end();
  std::string dlfull= dlfile;
  while (!handle && cur != end) {
    dlfull = *cur + dlfile;
    GYOTO_DEBUG << "Trying to dlopen " << dlfull << "...\n";
    handle = dlopen(dlfull.c_str(), RTLD_LAZY | RTLD_GLOBAL);
    ++cur;
  }

  if (handle) {
    GYOTO_DEBUG << "Successfully loaded " << dlfull << ".\n";
  } else {
    GYOTO_DEBUG << "Failed loading " << dlfull << ".\n";
    if (nofail) {
      if (verbose() >= GYOTO_DEFAULT_VERBOSITY)
	cerr << "WARNING: unable to load optional plug-in " << dlfile << endl;
      return;
    }
    if ( (err=dlerror()) ) throwError(err);
    throwError((string("Failed to load plug-in ")+dlfile).c_str());
  }

  GYOTO_DEBUG << "Searching plug-in init function " << dlfunc << endl;
  initfcn = (GyotoInitFcn*)dlsym(handle, dlfunc.c_str());
  if ( (err=dlerror()) || !initfcn) {
    dlfunc = "__GyotoPluginInit";
    initfcn = (GyotoInitFcn*)dlsym(handle, dlfunc.c_str());
  }
  if ( (err=dlerror()) ) throwError(err);
  GYOTO_DEBUG << "Calling plug-in init function " << dlfunc << endl;
  (*initfcn)();
  GYOTO_DEBUG << "Done." << endl;
}

void Gyoto::Register::init(char const *  cpluglist) {

  // Clean registers
  Metric::initRegister();
  Astrobj::initRegister();
  Spectrum::initRegister();
  // This cleans and fills Spectometer::Register_
  Spectrometer::initRegister();

  // Init units system
  Units::Init();

  // Init built-in plug-ins
#ifdef GYOTO_BUILTIN_STDPLUG
  __GyotostdplugInit();
#endif
#ifdef GYOTO_BUILTIN_LORENEPLUG
  __GyotoloreneInit();
#endif

  // Load DL plug-ins

  if (!cpluglist) cpluglist = getenv("GYOTO_PLUGINS");
  if (!cpluglist) cpluglist = GYOTO_DEFAULT_PLUGINS;

  std::string pluglist = cpluglist;

  if (pluglist.length()) {
    size_t first=0, last=0;
    string curplug="";
    int nofail=0;
    while (pluglist.length()) {
      last=pluglist.find(",");
      nofail=0;
      curplug=pluglist.substr(0, last);
      if (debug())
	cerr << "DEBUG: first: " << first << ", last: " << last
	     << ", pluglist: |" << pluglist << "|"
	     << ", curplug: |" << curplug << "|" << endl;
      if (last <= pluglist.length()) pluglist=pluglist.substr(last+1);
      else pluglist="";
      if (!curplug.compare(0, 7, "nofail:")) {
	curplug = curplug.substr(7);
	nofail=1;
      }

      Gyoto::loadPlugin(curplug.c_str(), nofail);
      nofail=0;
    }
  }

  if (debug()) Register::list();

}

Register::Entry::Entry(std::string name,
		       Gyoto::SmartPointee::Subcontractor_t* subcontractor,
		       Register::Entry* next)
  : name_(name), subcontractor_(subcontractor), next_(next)
{}

Register::Entry::~Entry() { if (next_) delete next_; }


Gyoto::SmartPointee::Subcontractor_t*
Register::Entry::getSubcontractor(std::string name, int errmode) {
# if GYOTO_DEBUG_ENABLED
  GYOTO_IF_DEBUG
    GYOTO_DEBUG_EXPR(name);
    GYOTO_DEBUG_EXPR(errmode);
  GYOTO_ENDIF_DEBUG
# endif
  if (name_==name) return subcontractor_;
  if (next_) return next_ -> getSubcontractor(name, errmode);
  if (errmode) return NULL;
  throwError ("Unregistered kind: "+name);
  return NULL; // will never get there, avoid compilation warning
}

void Gyoto::Register::list() {
  Register::Entry* entry = NULL;

  cout <<
"Gyoto will look for plug-ins first in the run-time linker default locations\n"
"(typically includes directories listed in e.g. $LD_LIBRARY_PATH), then in the\n"
"following locations:" << endl;

# if defined GYOTO_LOCALPKGLIBDIR
  cout << "    " << GYOTO_LOCALPKGLIBDIR "/" GYOTO_SOVERS "/" << endl;
  cout << "    " << GYOTO_LOCALPKGLIBDIR "/" << endl;
# endif
  cout << "    " << GYOTO_PKGLIBDIR "/" GYOTO_SOVERS "/" << endl;
  cout << "    " << GYOTO_PKGLIBDIR "/" << endl << endl;

  cout << "List of available Metrics:" << endl;
  for (entry = Metric::Register_; entry; entry = entry -> next_)
    cout << "    " << entry -> name_ << endl;
  
  cout << "List of available Astrobjs:" << endl;
  for (entry = Astrobj::Register_; entry; entry = entry -> next_)
    cout << "    " << entry -> name_ << endl;
  
  cout << "List of available Spectra:" << endl;
  for (entry = Spectrum::Register_; entry; entry = entry -> next_)
    cout << "    " << entry -> name_ << endl;
    
  
  cout << "List of available Spectrometers:" << endl;
  for (entry = Spectrometer::Register_; entry; entry = entry -> next_)
    cout << "    " << entry -> name_ << endl;
    
}
