/*
 *  Copyright (C) 2010 Tuomo Penttinen, all rights reserved.
 *
 *  Author: Tuomo Penttinen <tp@herqq.org>
 *
 *  This file is part of Herqq UPnP (HUPnP) library.
 *
 *  Herqq UPnP is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Herqq UPnP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Herqq UPnP. If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef HCLIENTSERVICE_ADAPTER_H_
#define HCLIENTSERVICE_ADAPTER_H_

#include <HUpnpCore/HAsyncOp>

#include <QtCore/QObject>

namespace Herqq
{

namespace Upnp
{

class HClientServiceAdapterPrivate;

/*!
 * This is an abstract base class for HUPnP A/V helper service classes.
 *
 * \headerfile hav_service.h HClientServiceAdapter
 *
 * \ingroup hupnp_av_common
 *
 * \remarks This class is not thread-safe.
 */
class H_UPNP_CORE_EXPORT HClientServiceAdapter :
    public QObject
{
Q_OBJECT
H_DISABLE_COPY(HClientServiceAdapter)
H_DECLARE_PRIVATE(HClientServiceAdapter)

protected:

    HClientServiceAdapterPrivate* h_ptr;

    /*!
     * Creates a new instance.
     *
     * \param parent specifies the parent \c QObject.
     */
    HClientServiceAdapter(const HResourceType& serviceType);

    //
    // \internal
    //
    HClientServiceAdapter(
        const HResourceType& serviceType, HClientServiceAdapterPrivate& dd);

    /*!
     * Prepares the instance to be associated with the specified HClientService.
     *
     * \param device specifies the device to be associated.
     *
     * \return \e true if the instance was successfully prepared to be used
     * with the specified HDevice. If \e false is returned the specified HDevice
     * will not be associated with this instance.
     */
    virtual bool prepareService(HClientService* service);

    /*!
     *
     */
    void setDeviceModelInfoProvider(const HDeviceModelInfoProvider&);

    /*!
     *
     */
    const HDeviceModelInfoProvider* deviceModelInfoProvider() const;

    /*!
     *
     */
    HClientAction* getAction(const QString&, qint32* rc = 0) const;

public:

    /*!
     * Destroys the instance.
     *
     * Destroys the instance.
     */
    virtual ~HClientServiceAdapter() = 0;

    /*!
     *
     */
    enum ValidationType
    {
        /*!
         *
         */
        MinimalValidation,

        /*!
         *
         */
        FullValidation
    };

    /*!
     * Associates the specified \c HService with this instance.
     *
     * \param service specifies the HService to be associated with this instance.
     *
     * \return \e true in case the \c service was successfully associated with
     * this instance.
     *
     * \sa service()
     */
    bool setService(HClientService* service, ValidationType vt = FullValidation);

    /*!
     * Returns the \c HService associated with this instance.
     *
     * \return the \c HService associated with this instance, if any.
     * A null pointer is returned in case setService() has not been called
     * successfully.
     *
     * \sa setService()
     */
    HClientService* service() const;

    /*!
     * Indicates if the HService associated with this instance implements
     * an action with the specified name.
     *
     * \param name specifies the name of the action to be checked.
     *
     * \return \e true in case the associated HService implements the
     * specified action.
     */
    bool implementsAction(const QString& name) const;

    /*!
     * Indicates if the HService associated with this instance implements
     * a state variable with the specified name.
     *
     * \param name specifies the name of the state variable to be checked.
     *
     * \return \e true in case the associated HService implements the
     * specified state variable.
     */
    bool implementsStateVariable(const QString& name) const;

    /*!
     * Indicates if the object is ready to be used.
     *
     * \return \e true if the object is ready to be used.
     */
    bool isReady() const;

    /*!
     * Returns the service type this adapter abstracts.
     *
     * \return the service type this adapter abstracts.
     */
    const HResourceType& serviceType() const;
};

}
}

#endif /* HCLIENTSERVICE_ADAPTER_H_ */
