/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icalculatorbase.h"


#include "icalculatorkit.h"
#include "icalculatorparser.h"
#include "ierror.h"

//
//  Templates
//
#include "iarraytemplate.h"
#include "ibuffertemplate.h"


using namespace iCalculatorKit;


iCalculatorBase::iCalculatorBase(const iCalculatorBase *driver) : mVariables(driver==0 ? mVariableData : driver->mVariables)
{
	mParser = 0;
	mErrorPosition = -1;

	mTrapFPE = (driver == 0) ? false : driver->GetTrapFPE();

	mAllowUndeterministicCalls = false;

	mResult = 0;
	mStatus = 0;
	mResultVariable = 0;
}


iCalculatorBase::~iCalculatorBase()
{
	this->Reset();
	if(mParser != 0) delete mParser;
}


void iCalculatorBase::Reset()
{
	mStatus = 0;
	mResult = 0;
	mCode.Clear();
	mUsedVars.Clear();

	while(mDataStack.Size() > 0) delete mDataStack.RemoveLast();
	while(mCallStack.Size() > 0) delete mCallStack.RemoveLast();

	while(mLiterals.Size() > 0) delete mLiterals.RemoveLast();
}


void iCalculatorBase::SetResultVariable(iCalculatorKit::Variable *var)
{
	if(mResultVariable != var)
	{
		mResultVariable = var;
		if(mStatus > 1) mStatus = 1; // make sure we re-link if this is changed
	}
}


void iCalculatorBase::SetResultVariable(iCalculatorKit::Variable &var)
{
	this->SetResultVariable(&var);
}


void iCalculatorBase::CopyVariables(const iCalculatorBase *other)
{
	if(other != 0)
	{
		int i;
		for(i=0; i<other->mVariables.Size(); i++)
		{
			mVariableData.Add(other->mVariables[i]);  //  this insures the correct name scope resolution
		}
	}
}


void iCalculatorBase::SetTrapFPE(bool s)
{
	if(mTrapFPE != s)
	{
		mTrapFPE = s;
		mStatus = 0; // make sure we re-compile if this is changed
	}
}


void iCalculatorBase::AddVariable(const iCalculatorKit::Variable *var)
{
	if(var!=0 && !var->Name().IsEmpty())
	{
		mVariableData.Add(var);
	}
}


void iCalculatorBase::RemoveVariable(const iString &name)
{
	if(!name.IsEmpty())
	{
		int i;
		for(i=mVariableData.MaxIndex(); i>=0; i--)  // variables are always looped over in the reverse order to insure proper name resolution
		{
			if(mVariableData[i]->Name() == name)
			{
				mVariableData.Remove(i);
				return;  //  remove one at a time
			}
		}
	}
}


void iCalculatorBase::SetError(const iString &message, int pos)
{
	mErrorMessage = message;
	mErrorPosition = pos;

	this->SetErrorExtra(message,pos);

	this->Reset();
}


//
//  Overwritable method for extending
//
iCalculatorParser* iCalculatorBase::CreateParser() const
{
	return new iCalculatorParser(false);  //  Create the default parser
}


void iCalculatorBase::SetErrorExtra(const iString &, int)
{
	//
	//  Nothing to do here
	//
}


iString iCalculatorBase::ReportRunTimeErrorExtra(int, const iCalculatorKit::Function *)
{
	return "unknown error";
}


//
//  Punctuation symbols
//
const iString& iCalculatorBase::SymbolSeparator() const
{
	if(mParser == 0)
	{
		mParser = this->CreateParser(); IERROR_ASSERT(mParser);
	}
	return mParser->SymbolSeparator();
}


const iString& iCalculatorBase::SymbolOpeningBraket() const
{
	if(mParser == 0)
	{
		mParser = this->CreateParser(); IERROR_ASSERT(mParser);
	}
	return mParser->SymbolOpeningBraket();
}


const iString& iCalculatorBase::SymbolClosingBraket() const
{
	if(mParser == 0)
	{
		mParser = this->CreateParser(); IERROR_ASSERT(mParser);
	}
	return mParser->SymbolClosingBraket();
}


const iString& iCalculatorBase::SymbolOpeningParenthesis() const
{
	if(mParser == 0)
	{
		mParser = this->CreateParser(); IERROR_ASSERT(mParser);
	}
	return mParser->SymbolOpeningParenthesis();
}


const iString& iCalculatorBase::SymbolClosingParenthesis() const
{
	if(mParser == 0)
	{
		mParser = this->CreateParser(); IERROR_ASSERT(mParser);
	}
	return mParser->SymbolClosingParenthesis();
}


bool iCalculatorBase::IsValidVariableName(const iString &name) const
{
	if(name.IsEmpty()) return false;

	if(mParser == 0)
	{
		mParser = this->CreateParser(); IERROR_ASSERT(mParser);
	}

	int i;
	bool ok = true;
	if(!mParser->IsVariableFirstLetter(name[0])) ok = false;
	for(i=1; ok && i<name.Length(); i++) if(!mParser->IsVariableLetter(name[i]))
	{
		ok = false;
		break;
	}
	return ok;
}


//
//  Main workers
//
bool iCalculatorBase::Evaluate(const iString &expr)
{
	if(!this->Compile(expr)) return false;
	if(!this->Link()) return false;
	return this->Execute();
}


bool iCalculatorBase::Compile(const iString &expr)
{
	if(mParser == 0)
	{
		mParser = this->CreateParser(); IERROR_ASSERT(mParser);
	}

	if(!mParser->Parse(expr))
	{
		mCode.Clear();
		this->SetError(mParser->GetErrorMessage(),mParser->GetErrorPosition());
		return false;
	}

	if(mParser->GetPseudoCode() != mCode) // Need to re-compile.
	{
		this->Reset();	
		mErrorMessage.Clear();
		mErrorPosition = -1;

		//
		//  First, create the stack.
		//
		mCode = mParser->GetPseudoCode();

		int i = 0;
		while(i < mCode.Length())
		{
			if(!this->CompileToken(i))
			{
				return false;
			}
		}

		if(mDataStack.Size() < 1)
		{
			this->SetError("Usage error: cannot evaluate empry expression.",0);
			return false;
		}

	}

	mStatus = 1;

	return true;
}


bool iCalculatorBase::CompileToken(int &index)
{
	iString w;
	int i2, i3, i1 = index;
	Handle *h;
	Variable *var;
	Function *fun;

	//
	//  Add one handle
	//
	if(mCode.Find(mParser->TokenBeg(),i1) != i1)
	{
		this->SetError("Internal error: pseudo-code is corrupted.",i1);
		return false;
	}
	i1 += mParser->TokenBeg().Length();

	//
	//  A literal
	//
	if(mCode.Find(mParser->TokenNum(),i1) == i1)
	{
		i1 += mParser->TokenFun().Length();
		i2 = mCode.Find(mParser->TokenEnd(),i1);
		if(i2 == -1)
		{
			this->SetError("Internal error: pseudo-code is corrupted.",i1);
			return false;
		}
		w = mCode.Part(i1,i2-i1);
		var = this->CreateLiteral(w);
		if(var == 0)
		{
			this->SetError("Internal error: unable to convert string {"+w+"} into a literal value, or there is not enough memory to proceed.",i1);
			return false;
		}
		h = new Handle(var);
		if(h == 0)
		{
			this->SetError("There is not enough memory to proceed.",-1);
			return false;
		}
		mLiterals.Add(var);
		mDataStack.Add(h);
		index = i2 + mParser->TokenEnd().Length();
		return true;
	}

	//
	//  A variable
	//
	if(mCode.Find(mParser->TokenVar(),i1) == i1)
	{
		i1 += mParser->TokenFun().Length();
		i2 = mCode.Find(mParser->TokenEnd(),i1);
		if(i2 == -1)
		{
			this->SetError("Internal error: pseudo-code is corrupted.",i1);
			return false;
		}
		w = mCode.Part(i1,i2-i1);
		h = new Handle(w);
		if(h == 0)
		{
			this->SetError("There is not enough memory to proceed.",-1);
			return false;
		}
		mDataStack.Add(h);
		index = i2 + mParser->TokenEnd().Length();
		if(mUsedVars.IsEmpty()) mUsedVars += w; else mUsedVars += "," + w;
		return true;
	}

	//
	//  A function
	//
	if(mCode.Find(mParser->TokenFun(),i1) == i1)
	{
		i1 += mParser->TokenFun().Length();
		i2 = mCode.Find(mParser->TokenBeg(),i1);
		i3 = mCode.Find(mParser->TokenEnd(),i1);
		if(i2==-1 && i3==-1)
		{
			this->SetError("Internal error: pseudo-code is corrupted.",i1);
			return false;
		}			
		if(i2==-1 || i3<i2)
		{
			//
			//  A function without arguments
			//
			i2 = i3;
		}			
		w = mCode.Part(i1,i2-i1);
		fun = this->CreateFunction(w,mDataStack.Size());
		if(fun == 0)
		{
			this->SetError("Syntax error: {"+w+"} is not a name of a valid function.",-1);
			return false;
		}
		h = new Handle(fun->Result());
		if(h == 0)
		{
			this->SetError("There is not enough memory to proceed.",-1);
			return false;
		}
		mCallStack.Add(fun);
		mDataStack.Add(h);

		//
		//  Compile function arguments
		//
		iString w1;
		int narg = 0;
		while(mParser->FindCompleteTokenToRight(mCode,w1,i2-1) >= i2)
		{
            if(!this->CompileToken(i2)) return false;
			narg++;
		}

		//
		//  Verify the number of arguments for this function
		//
		if(fun->NumArguments()>-1 && narg!=fun->NumArguments())
		{
			this->SetError("Syntax error: incorrect number of arguments for function {"+w+"}. This function only accepts "+iString::FromNumber(fun->NumArguments())+" arguments.",i2);
			return false;
		}
		if(fun->NumArguments() == -1) fun->SetNumArguments(narg);

		if(i2 != mCode.Find(mParser->TokenEnd(),i2))
		{
			this->SetError("Internal error: pseudo-code is corrupted.",i2);
			return false;
		}
		index = i2 + mParser->TokenEnd().Length();
		return true;
	}

	this->SetError("Internal error: pseudo-code is corrupted.",index);
	return false;
}


bool iCalculatorBase::Link()
{
	if(mStatus < 1)
	{
		this->SetError("Usage error: trying to link the expression before it is compiled.",-1);
		return false;
	}

	if(mStatus >= 2) return true; // already linked

	//
	//  Make sure that all variable handles are satisfied.
	//
	int i, id;
	const Variable *var;
	for(id=0; id<mDataStack.Size(); id++) if(mDataStack[id]->Self() == 0)
	{
		var = 0;
		for(i=mVariables.MaxIndex(); var==0 && i>=0; i--)  // variables are always looped over in the reverse order to insure proper name resolution
		{
			if(mVariables[i]->Name() == mDataStack[id]->Name()) var = mVariables[i];
		}
		if(var == 0)
		{
			this->SetError("Link error: variable {"+mDataStack[id]->Name()+"} has not been defined.",-1);
			return false;
		}
		mDataStack[id]->Attach(var);
	}

	//
	//  Now check that all dimensions and usages are consistent.
	//  For that, we need to use a temporary value stack that changes as we "execute".
	//
	mBuffer.Extend(mDataStack.Size());
	mSavedBuffer.Extend(mDataStack.Size());
	for(id=0; id<mDataStack.Size(); id++) mSavedBuffer[id] = mDataStack[id]->Self();
	memcpy(mBuffer,mSavedBuffer,mDataStack.Size()*sizeof(Variable*));

	//
	//  Now go down the stack, as if we were evaluating the expression.
	//
	int ic, iptr, narg;
	iString err;
	iBuffer<int> dims;
	iBuffer<int> uses;
	id = mDataStack.MaxIndex();
	for(ic=mCallStack.MaxIndex(); ic>=0; ic--)
	{
		iptr = mCallStack[ic]->DataLocation();
		narg = mCallStack[ic]->NumArguments();
		if(iptr+narg > id)  //  Under normal circumstances, this should never happen.
		{
			this->SetError("Internal error: the data stack is corrupted.",ic);
			return false;
		}
		iptr++;
		for(i=0; i<narg; i++)
		{
			dims[i] = mBuffer[iptr+i]->Dim();
			uses[i] = mBuffer[iptr+i]->Use();
		}
		if(!mCallStack[ic]->CheckArguments(dims,uses,err))
		{
			if(err.IsEmpty())
			{
				this->SetError("Link error: dimensions and/or usage patterns of arguments for function {"+mDataStack[iptr-1]->Name()+"} are not compatible.",-1);
			}
			else
			{
				this->SetError("Link error for function {"+mDataStack[iptr-1]->Name()+"}: "+err+".",-1);
			}
			return false;
		}
		//
		//  We are fine, compress the stack
		//
		for(i=iptr+narg; i<=id; i++) mBuffer[i-narg] = mBuffer[i];
		id -= narg;
	}

	//
	//  At the end, we should end up with a single value in the stack.
	//
	if(id > 0)
	{
		this->SetError("Internal error: the result of the execution is not a single value.",id);
		return false;
	}

	if(mResultVariable != 0)
	{
		if(mResultVariable->Dim() != mBuffer[0]->Dim())
		{
			this->SetError("Usage error: the dimension of the result variable (" + iString::FromNumber(mResultVariable->Dim()) + ") is not compatible with the dimension of the expression (" + iString::FromNumber(mBuffer[0]->Dim())+").",-1);
			return false;
		}
		if(mResultVariable->Use() != mBuffer[0]->Use())
		{
			this->SetError("Usage error: the usage pattern of the result variable (" + iString::FromNumber(mResultVariable->Use()) + ") is not compatible with the usage pattern of the expression (" + iString::FromNumber(mBuffer[0]->Use())+").",-1);
			return false;
		}
	}

	mResult = mBuffer[0];

	mStatus = 2;
	return true;
}


bool iCalculatorBase::Execute()
{
	if(mStatus < 2)
	{
		this->SetError("Usage error: trying to execute the expression before it is compiled and linked.",-1);
		return false;
	}

	if(mStatus==3 && !mAllowUndeterministicCalls && mVariables.Size()==0) return true;  //  the expression has not been changed, we dot not permit non-deterministic calls (like rand()), and we have no variables, so the result can not change

	//
	//  Create the run-time stack.
	//
	memcpy(mBuffer,mSavedBuffer,mDataStack.Size()*sizeof(Variable*));

	//
	//  Now go down the stack and evaluate the expression.
	//
	int i, id, ic, iptr, narg;
	id = mDataStack.MaxIndex();
	iCalculatorKit::Function* const *callStack = mCallStack.Data();  //  cache for speed
	iCalculatorKit::Variable* *dataStack = mBuffer;  //  cache for speed
#ifdef I_DEBUG
//	this->PrintStack(0,0,id,dataStack);
#endif
	for(ic=mCallStack.MaxIndex(); ic>=0; ic--)
	{
		iptr = callStack[ic]->DataLocation() + 1;
		narg = callStack[ic]->NumArguments();
		i = callStack[ic]->Evaluate(dataStack+iptr);
		if(i != 0)
		{
			this->ReportRunTimeError(i,callStack[ic]);
			return false;
		}
		//
		//  We are fine, compress the stack
		//
		for(i=iptr+narg; i<=id; i++) dataStack[i-narg] = dataStack[i];
		id -= narg;
#ifdef I_DEBUG
//		this->PrintStack(iptr,narg,id,dataStack);
#endif
	}
	if(mResultVariable != 0)
	{
		if(!mResultVariable->Copy(*mResult))
		{
			this->SetError("Run-time error: unable to copy the result into the result variable;",-1);
			return false;
		}
	}
	mStatus = 3;
	return true;
}


void iCalculatorBase::ReportRunTimeError(int ierr, const Function* caller)
{
	iString err;

	switch(ierr)
	{
	case 1:
		{
			err = "indexing vector outside the bounds";
			break;
		}
	case 2:
		{
			err = "floating-point exception (division by zero, invalid function argument, etc)";
			break;
		}
	default:
		{
			err = this->ReportRunTimeErrorExtra(ierr,caller);
			break;
		}
	}
	this->SetError("Run-time error in function {"+caller->Result()->Name()+"} : "+err,-1);
}

