/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iobject.h"


#include "ierror.h"
#include "ierrorstatus.h"
#include "iobjecthelp.h"
#include "ipiecewisefunction.h"
#include "irangecollection.h"


//
//  Templates
//
#include "iarraytemplate.h"


#define I_CHECK_DUPLICATES


//
//  *****************************************************************
//
//  iObjectType implementation
//
//  *****************************************************************
//
iObjectType::iObjectType(ClassId c, const char *fullname, const char *shortname)
{
	mClass = c;
	mFullName = fullname;
	mShortName = shortname;
	mHelp = new iObjectTypeHelp(this);
}


iObjectType::iObjectType()
{
	mClass = _Any;
	mHelp = 0;
}


iObjectType::~iObjectType()
{
	if(mHelp != 0) delete mHelp;
}


bool iObjectType::IsHidden() const
{
	//
	//  If the type short name start with -, it is hidden.
	//
	return (mShortName[0] == '-');
}


bool iObjectType::IsMatch(const iString &p) const
{
	return p==mFullName || p==mShortName;
}


bool iObjectType::IsMatch(const iObjectType &p) const
{
	return p.mFullName == mFullName;
}


const iString& iObjectType::FullName() const
{
	return mFullName;
}


const iString& iObjectType::ShortName() const
{
	return mShortName;
}


bool iObjectType::operator==(const iObjectType &s) const
{
	return (mFullName == s.mFullName);
}


bool iObjectType::operator<(const iObjectType &s) const
{
	return mClass<s.mClass || (mClass==s.mClass && mFullName<s.mFullName);
}


//
//  iObjectTypeRegistry implementation
//
class iObjectTypePointer
{

public:

	iObjectTypePointer()
	{ 
		mPointer = 0; 
	}

	iObjectTypePointer(const iObjectType *p)
	{ 
		mPointer = p; 
	}

	const iObjectType* operator->() const
	{ 
		return mPointer;
	}
	
	const iObjectType* GetPointer() const
	{ 
		return mPointer;
	}
	
	void operator=(const iObjectTypePointer &p)
	{
		mPointer = p.mPointer;
	}

	bool operator==(const iObjectTypePointer &p) const
	{
		return (mPointer->FullName() == p.mPointer->FullName());
	}

	bool operator<(const iObjectTypePointer &p) const
	{
		return (mPointer->FullName() < p.mPointer->FullName());
	}

private:

	const iObjectType *mPointer;
};


int iObjectTypeRegistry::mTraversalLocation = 0;
iObjectType::ClassId iObjectTypeRegistry::mTraversalClass = iObjectType::_Any;


iArray<iObjectTypePointer>& iObjectTypeRegistry::List()
{
	static iOrderedArray<iObjectTypePointer> list(30);
	return list;
}


const iObjectType* iObjectTypeRegistry::CreateType(iObjectType::ClassId c, const char *fp, const char *sp)
{
	iObjectType *tmp = new iObjectType(c,fp,sp); IERROR_ASSERT(tmp);
#if defined(I_DEBUG) && defined (I_CHECK_DUPLICATES)
	const iObjectType *k;
	if((k=iObjectTypeRegistry::FindType(tmp->mFullName,true))!=0 || (k=iObjectTypeRegistry::FindType(tmp->mShortName,true))!=0)
	{
		IERROR_LOW("Duplicate type.");
	}
#endif
	//
	//  Register the type
	//
	List().Add(tmp);
	return tmp;
}


const iObjectType* iObjectTypeRegistry::FindType(const iString &str, bool hidden)
{
	int i, n;
	const iObjectType *p;
	iArray<iObjectTypePointer> &list = List();

	n = list.Size();
	for(i=0; i<n; i++)
	{
		p = list[i].GetPointer(); // dereference for speed
		if(str==p->mFullName || str==p->mShortName)
		{
			//
			//  Is it hidden?
			//
			if(hidden || !p->IsHidden()) return p; else return 0;
		}
	}
	return 0;
}


void iObjectTypeRegistry::InitTraversal(iObjectType::ClassId c)
{
	mTraversalLocation = 0;
	mTraversalClass = c;
}


const iObjectType* iObjectTypeRegistry::GetNextType(bool hidden)
{
	iArray<iObjectTypePointer> &list = List();
	int n = list.Size();

	if(mTraversalClass == iObjectType::_Any)
	{
		while(!hidden && mTraversalLocation<n && list[mTraversalLocation]->IsHidden()) mTraversalLocation++;
		if(mTraversalLocation < n)
		{
			return list[mTraversalLocation++].GetPointer();
		}
	}
	else
	{
		while(mTraversalLocation < n)
		{
			if(list[mTraversalLocation]->Class()==mTraversalClass && (hidden || !list[mTraversalLocation]->IsHidden())) return list[mTraversalLocation++].GetPointer();
			mTraversalLocation++;
		}
	}
	return 0;
}


//
//  *****************************************************************
//
//  iObjectKey implementation
//
//  *****************************************************************
//
const char iObjectKey::mLastLetterOfDelimiter = iObjectKey::Delimiter()[iObjectKey::Delimiter().Length()-1];
const char iObjectKey::mLastLetterOfPrefixSeparator = iObjectKey::PrefixSeparator()[iObjectKey::PrefixSeparator().Length()-1];
bool iObjectKey::mUseShortKeys = false;


iObjectKey::iObjectKey(const iObjectType *type, const char *fk, const char *sk, Arg a, int d, const iObjectType* parent)
{
	this->Define(type,fk,sk,a,d,parent);
}


void iObjectKey::Define(const iObjectType *type, const char *fk, const char *sk, Arg a, int d, const iObjectType* parent)
{
	mArg = a;
	mDim = d;
	//
	//  Ignore the prefix for proper downcasting
	//
	if(type != 0)
	{
		mFullKey = fk;
		mSearchableFullKey = fk + FieldSeparator();
		mSearchableIndexedFullKey = fk + LeftBracket();
		mPrefixedFullKey = type->FullName() + PrefixSeparator() + fk;
		mShortKey = sk;
		mSearchableShortKey = sk + FieldSeparator();
		mSearchableIndexedShortKey = sk + LeftBracket();
		mPrefixedShortKey = type->ShortName() + PrefixSeparator() + sk;
	}

	mType = type;
	mParent = parent;

	mHelp = new iObjectKeyHelp(this);
}

	
iObjectKey::~iObjectKey()
{
	if(mHelp != 0) delete mHelp;
}


bool iObjectKey::IsHidden() const
{
	//
	//  If the key short name start with -, it is hidden.
	//
	return mType->IsHidden() || (mShortKey[0] == '-');
}


const iString& iObjectKey::PrefixSeparator()
{
	static const iString s(":");
	return s;
}


const iString& iObjectKey::FieldSeparator()
{
	static const iString s("/");
	return s;
}


const iString& iObjectKey::SubSeparator()
{
	static const iString s(";");
	return s;
}


const iString& iObjectKey::SubSubSeparator()
{
	static const iString s(",");
	return s;
}


const iString& iObjectKey::Delimiter()
{
	static const iString s(" ");
	return s;
}


const iString& iObjectKey::LeftBracket()
{
	static const iString s("[");
	return s;
}


const iString& iObjectKey::RightBracket()
{
	static const iString s("]");
	return s;
}


const iString& iObjectKey::SpecialSeparator()
{
	static const iString s("&&");
	return s;
}


const iObjectType& iObjectKey::Type() const
{
	IERROR_ASSERT(mType);
	return *mType;
}


const iObjectKey::Arg iObjectKey::Argument() const
{
	return mArg;
}


const int iObjectKey::Dimension() const
{
	return mDim;
}


const iString iObjectKey::PrefixedFullName(int index) const
{
	if(index < 0) return mPrefixedFullKey; else return mPrefixedFullKey + LeftBracket() + iString::FromNumber(index+1) + RightBracket();
}


const iString iObjectKey::PrefixedShortName(int index) const
{
	if(index < 0) return mPrefixedShortKey; else return mPrefixedShortKey + LeftBracket() + iString::FromNumber(index+1) + RightBracket();
}


const iString iObjectKey::UnprefixedFullName(int index) const
{
	if(index < 0) return mFullKey; else return mFullKey + LeftBracket() + iString::FromNumber(index+1) + RightBracket();
}


const iString iObjectKey::UnprefixedShortName(int index) const
{
	if(index < 0) return mShortKey; else return mShortKey + LeftBracket() + iString::FromNumber(index+1) + RightBracket();
}


const iString iObjectKey::PrefixedKey(int index) const
{
	if(mUseShortKeys) return this->PrefixedShortName(index); else return this->PrefixedFullName(index);
}


const iString iObjectKey::UnprefixedKey(int index) const
{
	if(mUseShortKeys) return this->UnprefixedShortName(index); else return this->UnprefixedFullName(index);
}


const iString iObjectKey::HelpEntryKey() const
{
	if(mType != 0)
	{
		if(mParent == 0) return mType->ShortName() + "." + mShortKey; else return mParent->ShortName() + "." + mShortKey;
	}
	else return "";
}


const iString iObjectKey::PackingKey(int index, bool prefixed) const
{
	if(prefixed) return this->PrefixedKey(index); else return this->UnprefixedKey(index);
}


const iString iObjectKey::RetypedKey(const iObjectType &type, int index) const
{
	if(mUseShortKeys)
	{
		return type.ShortName() + iObjectKey::PrefixSeparator() + this->UnprefixedShortName(index);
	}
	else
	{
		return type.FullName() + iObjectKey::PrefixSeparator() + this->UnprefixedFullName(index);
	}
}


bool iObjectKey::operator==(const iObjectKey &s) const
{
	return (mPrefixedFullKey == s.mPrefixedFullKey);
}


bool iObjectKey::operator<(const iObjectKey &s) const
{
	return (mPrefixedFullKey < s.mPrefixedFullKey);
}


int iObjectKey::FindFullMatch(const iString &str, const iString &pat) const
{
	int ind = str.Find(pat);
	//
	//  We have to be careful with keys - they may produce a match on just a piece
	//  of another key. So, check that this is the complete key and not a part of 
	//  another one. I.e., the letter just before the key must be a last letter of either 
	//  the delimiter or the prefix separator.
	//
	while(ind>0 && str[ind-1]!=mLastLetterOfDelimiter && str[ind-1]!=mLastLetterOfPrefixSeparator) ind = str.Find(pat,ind+1);
	return ind;
}


int iObjectKey::FindInString(const iString &str, int &index) const
{
	//
	//  Search for the un-indexed key first
	//
	int ind = this->FindFullMatch(str,mSearchableFullKey);
	if(ind == -1) 
	{
		ind = this->FindFullMatch(str,mSearchableShortKey); 
	}
	if(ind >= 0)  // key found
	{
		index = -1;
		return ind;
	}
	//
	//  repeat for the indexed key
	//
	ind = this->FindFullMatch(str,mSearchableIndexedFullKey);
	if(ind == -1) ind = this->FindFullMatch(str,mSearchableIndexedShortKey); 
	if(ind == -1) //not found
	{
		index = -1;
		return ind;
	}
	//
	//  Get the index
	//
	bool ok;
	iString s = str.Part(ind);
	int i1 = s.Find(LeftBracket());
	int i2 = s.Find(RightBracket());
	if(i1+2 > i2) // wrong order
	{
		index = -1;
		return -1;
	}
	index = s.Part(i1+1,i2-i1-1).ToInt(ok) - 1;
	if(!ok || index<0) // syntax error
	{
		index = -1;
		return -1;
	}
	return ind;
}


//
//  iObjectKeyRegistry implementation
//
class iObjectKeyPointer
{

public:

	iObjectKeyPointer()
	{ 
		mPointer = 0; 
	}

	iObjectKeyPointer(const iObjectKey *p)
	{ 
		mPointer = p; 
	}

	const iObjectKey* operator->() const
	{ 
		return mPointer;
	}
	
	const iObjectKey* GetPointer() const
	{ 
		return mPointer;
	}
	
	void operator=(const iObjectKeyPointer &p)
	{
		mPointer = p.mPointer;
	}

	bool operator==(const iObjectKeyPointer &p) const
	{
		return (mPointer->PrefixedFullName() == p.mPointer->PrefixedFullName());
	}

	bool operator<(const iObjectKeyPointer &p) const
	{
		return (mPointer->PrefixedFullName() < p.mPointer->PrefixedFullName());
	}

private:

	const iObjectKey *mPointer;
};


int iObjectKeyRegistry::mTraversalLocation = 0;
bool iObjectKeyRegistry::mTraversalWithInherited = true;
const iObjectType* iObjectKeyRegistry::mTraversalType = 0;


iArray<iObjectKeyPointer>& iObjectKeyRegistry::List()
{
	static iOrderedArray<iObjectKeyPointer> list(100);
	return list;
}


iArray<iObjectKey*>& iObjectKeyRegistry::TempKeys()
{
	static iArray<iObjectKey*> list;
	return list;
}


const iObjectKey* iObjectKeyRegistry::CreateKey(const iObjectType &type, const char *fk, const char *sk, iObjectKey::Arg a, int d, const iObjectType* parent, const iObjectKey* helpOwner)
{
	iObjectKey *tmp = new iObjectKey(&type,fk,sk,a,d,parent);
	IERROR_ASSERT(tmp);
#if defined(I_DEBUG) && defined (I_CHECK_DUPLICATES)
	const iObjectKey *k;
	if((k=iObjectKeyRegistry::FindKey(tmp->mPrefixedFullKey,true))!=0 || (k=iObjectKeyRegistry::FindKey(tmp->mPrefixedShortKey,true))!=0) 
	{
		IERROR_LOW("Duplicate key.");
	}
#endif
	//
	//  Assign help
	//
	if(helpOwner != 0) tmp->mHelp = new iObjectKeyHelp(helpOwner);
	//
	//  Register the key
	//
	List().Add(tmp);
	return tmp;
}


const iObjectKey& iObjectKeyRegistry::GetTempKey(const iObjectType &type, const iString &fk, int id)
{
	iArray<iObjectKey*> &tmp = TempKeys();
	//
	//  If any key is requested (id < 0), return the first unused one
	//
	if(id < 0)
	{
		for(id=0; id<tmp.Size(); id++) 
		{
			if(tmp[id]->mFullKey.IsEmpty()) break;
		}
	}	
	//
	//  If all the keys are in use, create new ones
	//
	while(id >= tmp.Size())
	{
		tmp.Add(new iObjectKey(&type,fk.ToCharPointer(),"-tmp",iObjectKey::_Any,0));
	}
	//
	//  Change the existing one to a proper value, unless it is the same one
	//
	if(!(tmp[id]->Type()==type) || tmp[id]->mFullKey!=fk)
	{
		tmp[id]->Define(&type,fk.ToCharPointer(),"-tmp",iObjectKey::_Any,0);
	}
	return *tmp[id];
}


void iObjectKeyRegistry::ReleaseTempKey(const iObjectKey &key)
{
	int i;
	iArray<iObjectKey*> &tmp = TempKeys();
	//
	//  Find the key and remove its full name
	//
	for(i=0; i<tmp.Size(); i++)
	{
		if(*tmp[i] == key)
		{
			tmp[i]->mFullKey.Clear();
			break;
		}
	}
}


const iObjectKey* iObjectKeyRegistry::FindKey(const iString &str0, bool hidden)
{
	int i, n;
	const iObjectKey *p;
	iArray<iObjectKeyPointer> &list = List();

	iString str = str0.Section(iObjectKey::LeftBracket(),0,0);

	n = list.Size();
	for(i=0; i<n; i++)
	{
		p = list[i].GetPointer();  // dereference for speed
		if(str==p->mPrefixedFullKey || str==p->mPrefixedShortKey)
		{
			//
			//  Is it hidden?
			//
			if(hidden || !p->IsHidden()) return p; else return 0;
		}
	}
	return 0;
}


void iObjectKeyRegistry::InitTraversal(const iObjectType *type, bool inherited)
{
	mTraversalLocation = 0;
	mTraversalWithInherited = inherited;
	mTraversalType = type;
}


const iObjectKey* iObjectKeyRegistry::GetNextKey(bool hidden)
{
	iArray<iObjectKeyPointer> &list = List();
	int n = list.Size();

	if(mTraversalType == 0)
	{
		while(!hidden && mTraversalLocation<n && list[mTraversalLocation]->IsHidden()) mTraversalLocation++;
		if(mTraversalLocation < n)
		{
			return list[mTraversalLocation++].GetPointer();
		}
	}
	else
	{
		const iObjectType &type = *mTraversalType;
		while(mTraversalLocation < n)
		{
			if(list[mTraversalLocation]->Type()==type && (hidden || !list[mTraversalLocation]->IsHidden()) && (mTraversalWithInherited || !list[mTraversalLocation]->IsInherited())) return list[mTraversalLocation++].GetPointer();
			mTraversalLocation++;
		}
	}
	return 0;
}


//
//  *****************************************************************
//
// iObject implementation
//
//  *****************************************************************
//
const iString iObject::mSlash = "@@";
int iObject::mMissingKeysStack = 0;


void iObject::ReportMissingKeys(bool s)
{
	if(s) mMissingKeysStack--; else mMissingKeysStack++;
	if(mMissingKeysStack < 0) mMissingKeysStack = 0;
}


void iObject::ReportAMissingKey(const iObjectKey &key)
{
	if(mMissingKeysStack == 0) IERROR_LOW((iString("Missing key: ")+key.PrefixedFullName()).ToCharPointer());
}


iObject::iObject(const iString &name) : mErrorStatus(name)
{
	mUnPackedSomething = false;
	mCacheInvalid = true;
}


iObject::~iObject()
{
}


void iObject::ClearCache()
{
	mCacheInvalid = true;
}


void iObject::PackCompleteState(iString &s) const
{
	this->PackState(s);
}


void iObject::UnPackCompleteState(const iString &s)
{
	this->UnPackState(s);
}


void iObject::PackState(iString &s) const
{
	if(mCacheInvalid || mCache.IsEmpty())
	{
		s.Init(10000);
		this->PackStateBody(s);
		mCache = s;
		mCacheInvalid = false;
	}
	else
	{
		s = mCache;
	}
}


void iObject::UnPackState(const iString &s)
{
	mUnPackedSomething = false;
	this->UnPackStateBody(s);
}


void iObject::CopyState(const iObject *p)
{
	iString s;
	//
	//  Read the parameters
	//
	p->PackCompleteState(s);
	//
	//  Set the parameters of the new instance
	//
	this->UnPackCompleteState(s);
}


bool iObjectKey::IsArgumentCompatible(Arg a) const
{
	 // offset int and int have compatible types
	return mArg==_Any || a==mArg || (a==_OffsetInt && mArg==_Int) || (a==_Int && mArg==_OffsetInt);
}


bool iObjectKey::IndexHelper(Arg a, int n, int index, int &imin, int &imax) const
{
	if(!this->IsArgumentCompatible(a))
	{
		// incompatible arguments, exit without doing anything.
		IERROR_LOW("Incompatible key argument.");
		return true;
	}

	if(n<0 || (mDim!=0 && ((index<0 && n!=mDim) || (index>=0 && n!=1 && index>=mDim))))  //  dim = 0 disables bound checking 
	{
		// incorrect key index, exit without doing anything.
		IERROR_LOW("Incorrect key index.");
		return true;
	}

	if(index < 0)
	{
		if(n == 0)
		{
			IERROR_LOW("Incorrect key index.");
			return true;
		}
		imin = 0;
		imax = n;
	}
	else
	{
		if(mDim == 1)
		{
			IERROR_LOW("Incorrect key index.");
			return true;
		}
		imin = index;
		imax = index + 1;
	}
	return false;
}


//
//  Packing helpers
//
void iObject::PackValue(iString &s, const iObjectKey &key, const int* val, int num, int index, bool prefixed) const
{
	int i, i1, i2, offset = 0;
	if(key.IndexHelper(iObjectKey::_Int,num,index,i1,i2)) return;
	s += key.PackingKey(index,prefixed);
	if(key.Argument() == iObjectKey::_OffsetInt) offset = 1;
	for(i=i1; i<i2; i++) 
	{
		s += key.FieldSeparator() + iString::FromNumber(val[i]+offset);
	}
	s += key.FieldSeparator() + key.Delimiter();
}


void iObject::PackValue(iString &s, const iObjectKey &key, const bool* val, int num, int index, bool prefixed) const
{
	int i, i1, i2;
	if(key.IndexHelper(iObjectKey::_Bool,num,index,i1,i2)) return;
	s += key.PackingKey(index,prefixed);
	for(i=i1; i<i2; i++) 
	{
		s += key.FieldSeparator() + iString::FromNumber(val[i]?1:0);
	}
	s += key.FieldSeparator() + key.Delimiter();
}


void iObject::PackValue(iString &s, const iObjectKey &key, const float* val, int num, int index, bool prefixed) const
{
	int i, i1, i2;
	if(key.IndexHelper(iObjectKey::_Float,num,index,i1,i2)) return;
	s += key.PackingKey(index,prefixed);
	for(i=i1; i<i2; i++) 
	{
		s += key.FieldSeparator() + iString::FromNumber(val[i]);
	}
	s += key.FieldSeparator() + key.Delimiter();
}


void iObject::PackValue(iString &s, const iObjectKey &key, const double* val, int num, int index, bool prefixed) const
{
	int i, i1, i2;
	if(key.IndexHelper(iObjectKey::_Double,num,index,i1,i2)) return;
	s += key.PackingKey(index,prefixed);
	for(i=i1; i<i2; i++) 
	{
		s += key.FieldSeparator() + iString::FromNumber(val[i],"%.16lg");
	}
	s += key.FieldSeparator() + key.Delimiter();
}


void iObject::PackValue(iString &s, const iObjectKey &key, const iColor* val, int num, int index, bool prefixed) const
{
	int i, i1, i2;
	if(key.IndexHelper(iObjectKey::_Color,num,index,i1,i2)) return;
	s += key.PackingKey(index,prefixed);
	for(i=i1; i<i2; i++) 
	{
		s += key.FieldSeparator() + iString::FromNumber(val[i].Red()) + key.SubSubSeparator() + iString::FromNumber(val[i].Green()) + key.SubSubSeparator() + iString::FromNumber(val[i].Blue());
	}
	s += key.FieldSeparator() + key.Delimiter();
}


void iObject::PackValue(iString &s, const iObjectKey &key, const iString* val, int num, int index, bool prefixed) const
{
	iString s1;
	int i, i1, i2;
	if(key.IndexHelper(iObjectKey::_String,num,index,i1,i2)) return;
	s += key.PackingKey(index,prefixed);
	for(i=i1; i<i2; i++) 
	{
		s1 = val[i];
		s1.Replace(key.FieldSeparator(),mSlash);
		s += key.FieldSeparator() + s1;
	}
	s += key.FieldSeparator() + key.Delimiter();
}


//
//  Unpacking helpers
//
bool iObject::UnPackValue(const iString &s, const iObjectKey &key, int* val, int num)
{
	bool ok;
	int index, ind = key.FindInString(s,index);
	if(ind >=0)
	{
		int i, i1, i2, offset = 0;;
		ok = true;
		iString tmp = s.Part(ind);
		if(key.IndexHelper(iObjectKey::_Int,num,index,i1,i2))
		{
			return false;
		}
		if(key.Argument() == iObjectKey::_OffsetInt) offset = 1;
		int* v = new int[num+1]; IERROR_ASSERT(v); // num+1 is because arrays go from 1 to num
		for(i=i1; ok && i<i2; i++)
		{
			v[i] = tmp.Section(key.FieldSeparator(),i+1-i1,i+1-i1).ToInt(ok) - offset;
		}
		if(ok) memcpy(val+i1,v+i1,(i2-i1)*sizeof(int)); else IERROR_LOW("Invalid key value.");
		delete [] v;
	}
	else ok = false;
#ifdef I_CHECK1
	if(!ok) ReportAMissingKey(key);
#endif
	if(ok) mUnPackedSomething = true;
	return ok;
}


bool iObject::UnPackValue(const iString &s, const iObjectKey &key, bool* val, int num)
{
	bool ok;
	int index, ind = key.FindInString(s,index);
	if(ind >=0)
	{
		int i, i1, i2;
		ok = true;
		iString tmp = s.Part(ind);
		if(key.IndexHelper(iObjectKey::_Bool,num,index,i1,i2))
		{
			return false;
		}
		bool* v = new bool[num+1]; IERROR_ASSERT(v);
		for(i=i1; ok && i<i2; i++)
		{
			v[i] = (tmp.Section(key.FieldSeparator(),i+1-i1,i+1-i1).ToInt(ok) != 0);
		}
		if(ok) memcpy(val+i1,v+i1,(i2-i1)*sizeof(bool)); else IERROR_LOW("Invalid key value.");
		delete [] v;
	}
	else ok = false;
#ifdef I_CHECK1
	if(!ok) ReportAMissingKey(key);
#endif
	if(ok) mUnPackedSomething = true;
	return ok;
}


bool iObject::UnPackValue(const iString &s, const iObjectKey &key, float* val, int num)
{
	bool ok;
	int index, ind = key.FindInString(s,index);
	if(ind >=0)
	{
		int i, i1, i2;
		ok = true;
		iString tmp = s.Part(ind);
		if(key.IndexHelper(iObjectKey::_Float,num,index,i1,i2))
		{
			return false;
		}
		float* v = new float[num+1]; IERROR_ASSERT(v);
		for(i=i1; ok && i<i2; i++)
		{
			v[i] = tmp.Section(key.FieldSeparator(),i+1-i1,i+1-i1).ToFloat(ok);
		}
		if(ok) memcpy(val+i1,v+i1,(i2-i1)*sizeof(float)); else IERROR_LOW("Invalid key value.");
		delete [] v;
	}
	else ok = false;
#ifdef I_CHECK1
	if(!ok) ReportAMissingKey(key);
#endif
	if(ok) mUnPackedSomething = true;
	return ok;
}


bool iObject::UnPackValue(const iString &s, const iObjectKey &key, double* val, int num)
{
	bool ok;
	int index, ind = key.FindInString(s,index);
	if(ind >=0)
	{
		int i, i1, i2;
		ok = true;
		iString tmp = s.Part(ind);
		if(key.IndexHelper(iObjectKey::_Double,num,index,i1,i2))
		{
			return false;
		}
		double* v = new double[num+1]; IERROR_ASSERT(v);
		for(i=i1; ok && i<i2; i++)
		{
			v[i] = tmp.Section(key.FieldSeparator(),i+1-i1,i+1-i1).ToDouble(ok);
		}
		if(ok) memcpy(val+i1,v+i1,(i2-i1)*sizeof(double)); else IERROR_LOW("Invalid key value.");
		delete [] v;
	}
	else ok = false;
#ifdef I_CHECK1
	if(!ok) ReportAMissingKey(key);
#endif
	if(ok) mUnPackedSomething = true;
	return ok;
}


bool iObject::UnPackValue(const iString &s, const iObjectKey &key, iColor* val, int num)
{
	bool ok, ok1;
	iString s1;
	int r, g, b, index, ind = key.FindInString(s,index);
	if(ind >=0)
	{
		int i, i1, i2;
		ok = true;
		iString tmp = s.Part(ind);
		if(key.IndexHelper(iObjectKey::_Color,num,index,i1,i2))
		{
			return false;
		}
		iColor* v = new iColor[num+1]; IERROR_ASSERT(v);
		for(i=i1; ok && i<i2; i++)
		{
			s1 = tmp.Section(key.FieldSeparator(),i+1-i1,i+1-i1);
			r = s1.Section(key.SubSubSeparator(),0,0).ToInt(ok1); ok = ok && ok1;
			g = s1.Section(key.SubSubSeparator(),1,1).ToInt(ok1); ok = ok && ok1;
			b = s1.Section(key.SubSubSeparator(),2,2).ToInt(ok1); ok = ok && ok1;
			v[i] = iColor(r,g,b);
		}
		if(ok) memcpy(val+i1,v+i1,(i2-i1)*sizeof(iColor)); else IERROR_LOW("Invalid key value.");
		delete [] v;
	}
	else ok = false;
#ifdef I_CHECK1
	if(!ok) ReportAMissingKey(key);
#endif
	if(ok) mUnPackedSomething = true;
	return ok;
}


bool iObject::UnPackValue(const iString &s, const iObjectKey &key, iString* val, int num)
{
	bool ok;
	int index, ind = key.FindInString(s,index);
	if(ind >=0)
	{
		int i, i1, i2;
		ok = true;
		iString tmp = s.Part(ind);
		if(key.IndexHelper(iObjectKey::_String,num,index,i1,i2))
		{
			return false;
		}
		for(i=i1; ok && i<i2; i++)
		{
			val[i] = tmp.Section(key.FieldSeparator(),i+1-i1,i+1-i1);
			val[i].Replace(mSlash,key.FieldSeparator());
		}
	}
	else ok = false;
#ifdef I_CHECK1
	if(!ok) ReportAMissingKey(key);
#endif
	if(ok) mUnPackedSomething = true;
	return ok;
}


void iObject::PackValuePiecewiseFunction(iString &s, const iObjectKey &key, const iPiecewiseFunction * const *f, int num, int index, bool prefixed) const
{
	int i, j, n, i1, i2;
	if(f == 0) return;

	if(key.IndexHelper(iObjectKey::_Any,num,index,i1,i2)) return;
	s += key.PackingKey(index,prefixed);
	for(i=i1; f[i]!=0 && i<i2; i++) 
	{
		n = f[i]->N();
		s += key.FieldSeparator() + iString::FromNumber(f[i]->Min()) + key.SubSubSeparator() + iString::FromNumber(f[i]->Max());
		s += key.SubSeparator() + iString::FromNumber(f[i]->X(0)) + key.SubSubSeparator() + iString::FromNumber(f[i]->Y(0));
		s += key.SubSeparator() + iString::FromNumber(f[i]->X(n-1)) + key.SubSubSeparator() + iString::FromNumber(f[i]->Y(n-1));
		for(j=1; j<n-1; j++)
		{
			s += key.SubSeparator() + iString::FromNumber(f[i]->X(j)) + key.SubSubSeparator() + iString::FromNumber(f[i]->Y(j));
		}
	}
	s += key.FieldSeparator() + key.Delimiter();
}


bool iObject::UnPackValuePiecewiseFunction(const iString &s, const iObjectKey &key, iPiecewiseFunction **f, int num)
{
	bool ok;
	int index, ind = key.FindInString(s,index);
	if(ind >=0)
	{
		int i, j, i1, i2;
		ok = true;
		iString tmp = s.Part(ind);
		if(key.IndexHelper(iObjectKey::_Any,num,index,i1,i2))
		{
			return false;
		}
		iPiecewiseFunction* v = iPiecewiseFunction::New(); IERROR_ASSERT(v);
		iString tmp1, tmp2;
		float x1, x2;
		bool ok1;
		for(i=i1; ok && i<i2; i++)
		{
			tmp1 = tmp.Section(key.FieldSeparator(),i+1-i1,i+1-i1);
			tmp2 = tmp1.Section(key.SubSeparator(),0,0);
			x1 = tmp2.Section(key.SubSubSeparator(),0,0).ToFloat(ok1); ok = ok && ok1;
			x2 = tmp2.Section(key.SubSubSeparator(),1,1).ToFloat(ok1); ok = ok && ok1;
			if(ok) v->SetMinMax(x1,x2);
			j = 1;
			while(ok && !(tmp2=tmp1.Section(key.SubSeparator(),j,j)).IsEmpty())
			{
				x1 = tmp2.Section(key.SubSubSeparator(),0,0).ToFloat(ok1); ok = ok && ok1;
				x2 = tmp2.Section(key.SubSubSeparator(),1,1).ToFloat(ok1); ok = ok && ok1;
				if(ok) 
				{
					if(j < 3) v->MovePoint(j-1,x1,x2); else v->AddPoint(x1,x2);
				}
				j++;
			}
			if(ok) 
			{
				if(f[i] != 0) f[i]->Copy(v);
			}
			else IERROR_LOW("Invalid key value.");
		}
		v->Delete();
	}
	else ok = false;
#ifdef I_CHECK1
	if(!ok) ReportAMissingKey(key);
#endif
	if(ok) mUnPackedSomething = true;
	return ok;
}


void iObject::PackValueRangeCollection(iString &s, const iObjectKey &key, const iRangeCollection * const *f, int num, int index, bool prefixed) const
{
	int i, j, n, i1, i2;
	if(f == 0) return;

	if(key.IndexHelper(iObjectKey::_Any,num,index,i1,i2)) return;
	s += key.PackingKey(index,prefixed);
	for(i=i1; f[i]!=0 && i<i2; i++) 
	{
		n = f[i]->GetN();
		s += key.FieldSeparator() + iString::FromNumber(f[i]->GetGlobalMin()) + key.SubSubSeparator() + iString::FromNumber(f[i]->GetGlobalMax());
		for(j=0; j<n; j++)
		{
			s += key.SubSeparator() + iString::FromNumber(f[i]->GetMin(j)) + key.SubSubSeparator() + iString::FromNumber(f[i]->GetMax(j));
		}
	}
	s += key.FieldSeparator() + key.Delimiter();
}


bool iObject::UnPackValueRangeCollection(const iString &s, const iObjectKey &key, iRangeCollection **f, int num)
{
	bool ok;
	int index, ind = key.FindInString(s,index);
	if(ind >=0)
	{
		int i, j, i1, i2;
		ok = true;
		iString tmp = s.Part(ind);
		if(key.IndexHelper(iObjectKey::_Any,num,index,i1,i2))
		{
			return false;
		}
		iRangeCollection* v = new iRangeCollection[num+1]; IERROR_ASSERT(v);
		iString tmp1, tmp2;
		float x1, x2;
		bool ok1;
		for(i=i1; ok && i<i2; i++)
		{
			tmp1 = tmp.Section(key.FieldSeparator(),i+1-i1,i+1-i1);
			x1 = tmp1.Section(key.SubSubSeparator(),0,0).ToFloat(ok1); ok = ok && ok1;
			x2 = tmp1.Section(key.SubSubSeparator(),1,1).ToFloat(ok1); ok = ok && ok1;
			if(ok) v[i].SetGlobalRange(x1,x2);
			j = 1;
			while(ok && !(tmp2=tmp1.Section(key.SubSeparator(),j,j++)).IsEmpty())
			{
				x1 = tmp2.Section(key.SubSubSeparator(),0,0).ToFloat(ok1); ok = ok && ok1;
				x2 = tmp2.Section(key.SubSubSeparator(),1,1).ToFloat(ok1); ok = ok && ok1;
				if(ok) 
				{
					if(j > 2) v[i].AddRange(x1,x2); else v[i].SetRange(0,x1,x2);
				}
			}
		}
		if(ok) 
		{
			for(i=i1; f[i]!=0 && i<i2; i++) f[i]->Copy(v+i);
		}
		else IERROR_LOW("Invalid key value.");
		delete [] v;
	}
	else ok = false;
#ifdef I_CHECK1
	if(!ok) ReportAMissingKey(key);
#endif
	if(ok) mUnPackedSomething = true;
	return ok;
}


void iObject::PackValuePosition(iString &s, const iObjectKey &key, const iPosition &val, int index, bool prefixed) const
{
	//
	//  Save the box position as a double array
	//
	this->PackValue(s,key,val.BoxPosition(),3,index,prefixed);
	//
	//  Now save the OpenGL position
	//
	const iObjectKey &tmp = iObjectKeyRegistry::GetTempKey(key.Type(),key.UnprefixedFullName()+"GL");
	this->PackValue(s,tmp,val,3,index,prefixed);
	iObjectKeyRegistry::ReleaseTempKey(tmp);
}


bool iObject::UnPackValuePosition(const iString &s, const iObjectKey &key, iPosition &val)
{
	int i; double x[3];
	//
	//  Read the box position as a double array
	//
	for(i=0; i<3; i++) x[i] = val.BoxPosition(i);
	bool ret1 = this->UnPackValue(s,key,x,3);
	if(ret1) val.SetBoxPosition(x);
	//
	//  Now read the OpenGL position: if it is present, it will overwrite the box value
	//
	const iObjectKey &tmp = iObjectKeyRegistry::GetTempKey(key.Type(),key.UnprefixedFullName()+"GL");
	for(i=0; i<3; i++) x[i] = val[i];
	bool ret2 = this->UnPackValue(s,tmp,x,3);
	if(ret2) val = x;
	iObjectKeyRegistry::ReleaseTempKey(tmp);

	return ret1 || ret2;
}


void iObject::PackValueDistance(iString &s, const iObjectKey &key, const iDistance &val, int index, bool prefixed) const
{
	//
	//  Save the box position as a double array
	//
	this->PackValue(s,key,val.BoxDistance(),index,prefixed);
	//
	//  Now save the OpenGL position
	//
	const iObjectKey &tmp = iObjectKeyRegistry::GetTempKey(key.Type(),key.UnprefixedFullName()+"GL");
	this->PackValue(s,tmp,val,index,prefixed);
	iObjectKeyRegistry::ReleaseTempKey(tmp);
}


bool iObject::UnPackValueDistance(const iString &s, const iObjectKey &key, iDistance &val)
{
	double d;
	//
	//  Read the box position as a double array
	//
	bool ret1 = this->UnPackValue(s,key,d);
	if(ret1) val.SetBoxDistance(d);
	//
	//  Now read the OpenGL position: if it is present, it will overwrite the box value
	//
	const iObjectKey &tmp = iObjectKeyRegistry::GetTempKey(key.Type(),key.UnprefixedFullName()+"GL");
	bool ret2 = this->UnPackValue(s,tmp,d);
	if(ret2) val = d;
	iObjectKeyRegistry::ReleaseTempKey(tmp);

	return ret1 || ret2;
}

