/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggdialogimagecomposer.h"


#include "icontrolmodule.h"
#include "iimagecomposer.h"
#include "iimagecomposerwindows.h"
#include "iimagefactory.h"
#include "imath.h"
#include "ishell.h"
#include "iviewmodule.h"

#include "iggframe.h"
#include "iggmainwindow.h"
#include "iggwidgetarea.h"
#include "iggwidgetkeybutton.h"
#include "iggwidgetkeycolorselection.h"
#include "iggwidgetkeyselectionbox.h"
#include "iggwidgetkeyslider.h"
#include "iggwidgetotherbutton.h"

#include "ibgwidgetareasubject.h"
#include "ibgwidgetselectionboxsubject.h"

#include "iggsubjectfactory.h"

//
//  Templates (needed for some compilers)
//
#include "iggwidgetkeyhandlertemplate.h"


using namespace iParameter;
using namespace iParameter;


namespace iggDialogImageComposer_Private
{
	class Area : public iggWidgetDrawArea
	{

	public:

		Area(iggDialogImageComposer *dialog, iggFrame *parent) : iggWidgetDrawArea(parent,true), mSizeFactor(0.75)
		{
			mDialog = dialog;
			mSubject->SetBackgroundColor(iColor(100,100,100));

			mPosX = mPosY = 0;
			mIsMovingForegroundWindow = false;
			mNeedsBaloonHelp = false;
		}

		virtual void OnMousePress(int dx, int dy, int b)
		{
			int i, j, n, x, y;
			iImageComposer *ic = mDialog->GetImageComposer();
			iImageComposerForegroundWindow *fw;

			mIsMovingForegroundWindow = false;
			if(b!=MouseButton::LeftButton || !this->FindPosition(ic,dx,dy,x,y)) return;

			//
			//  Find what is under mouse cursor
			//
			mPosX = mOldPosX = x;
			mPosY = mOldPosY = y;

			//
			//  Is there a foreground window? Search them backwards, as the later
			//  windows are higher
			//
			n = ic->GetNumberOfForegroundWindows();
			for(i=n-1; i>=0; i--) //  later added windows are on top, so loop in the reverse order
			{
				fw = ic->GetForegroundWindow(i);

				if(x>=fw->GetPositionX() && x<fw->GetPositionX()+fw->GetImageWidth() && y>=fw->GetPositionY() && y<fw->GetPositionY()+fw->GetImageHeight())
				{
					//
					//  Found a window - make it current
					//
					mDialog->SetCurrentForegroundWindow(i);
					mIsMovingForegroundWindow = true;
					mSubject->RequestPainting(DrawMode::Foreground);
					return;
				}
			}

			//
			//  No foreground window - is there a background one?
			//
			int bw = ic->GetBorderWidth();
			int tw = ic->GetTileWidth();
			int th = ic->GetTileHeight();
			if(ic->GetInnerBorder())
			{
				tw += bw;
				th += bw;
			}
			i = (x-bw)/tw;
			j = (y-bw)/th;
			if(i>=0 && i<ic->GetNumTilesX() && j>=0 && j<ic->GetNumTilesY())
			{
				mDialog->SetCurrentBackgroundWindow(i+j*ic->GetNumTilesX());
				mSubject->RequestPainting();
				this->UpdateDependents();
			}
		}

		virtual void OnMouseRelease(int /*dx*/, int /*dy*/, int b)
		{
			if(b != MouseButton::LeftButton) return;

			if(mIsMovingForegroundWindow)
			{
				iImageComposer *ic = mDialog->GetImageComposer();
				iImageComposerForegroundWindow *fw = ic->GetForegroundWindow(mDialog->GetCurrentForegroundWindow());
				fw->SetPosition(fw->GetPositionX()+mPosX-mOldPosX,fw->GetPositionY()+mPosY-mOldPosY);
				mIsMovingForegroundWindow = false;
				mSubject->RequestPainting(DrawMode::Foreground);
				this->UpdateDependents();
			}
		}

		virtual void OnMouseMove(int dx, int dy, int b)
		{
			if(b != MouseButton::LeftButton)
			{
				mIsMovingForegroundWindow = false;
				return;
			}

			if(mIsMovingForegroundWindow)
			{
				int x, y;
				iImageComposer *ic = mDialog->GetImageComposer();
				this->FindPosition(ic,dx,dy,x,y);
				mPosX = x;
				mPosY = y;
				mSubject->RequestPainting(DrawMode::Foreground);
			}
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			//
			//  Compute reduction factor
			//
			float f1 = mSizeFactor*mSubject->Width()/mDialog->GetImageComposer()->GetImageWidth(); 
			float f2 = mSizeFactor*mSubject->Height()/mDialog->GetImageComposer()->GetImageHeight(); 
			mScaleFactor = (f1 > f2) ? f2 : f1;
			if(mScaleFactor > 1.0) mScaleFactor = 1.0;

			this->iggWidgetDrawArea::UpdateWidgetBody();
		}

		virtual void DrawBackgroundBody()
		{
			static const iColor white = iColor(255,255,255);
			int i, x, y, v, n, nx, ny;
			bool ib;
			iColor c;
			iImageComposerBackgroundWindow *bw;

			//
			//  Cache a few things
			//
			iImageComposer *ic = mDialog->GetImageComposer();
			ib = ic->GetInnerBorder();
			nx = ic->GetNumTilesX();
			ny = ic->GetNumTilesY();

			//
			//  Compute reduced sizes
			//
			int drawnBorderWidth = round(mScaleFactor*ic->GetBorderWidth());
			if(ic->GetBorderWidth()>0 && drawnBorderWidth<1) drawnBorderWidth = 1;
			int drawnTileWidth = round(mScaleFactor*ic->GetTileWidth());
			int drawnTileHeight = round(mScaleFactor*ic->GetTileHeight());
			mDrawnFullWidth = nx*drawnTileWidth + 2*drawnBorderWidth;
			mDrawnFullHeight = ny*drawnTileHeight + 2*drawnBorderWidth;
			if(ib)
			{
				mDrawnFullWidth += drawnBorderWidth*(nx-1);
				mDrawnFullHeight += drawnBorderWidth*(ny-1);
			}

			mStartX = (mSubject->Width()-mDrawnFullWidth)/2;
			mStartY = (mSubject->Height()-mDrawnFullHeight)/2;

			//
			//  Draw
			//
			if(!ic->GetInnerBorder())
			{
				this->DrawWindowBase(mStartX,mStartY,mDrawnFullWidth,mDrawnFullHeight,white,drawnBorderWidth,ic->GetBorderColor());
			}

			n = nx*ny;
			for(i=0; i<n; i++)
			{
				bw = ic->GetBackgroundWindow(i);

				x = mStartX + bw->GetTileX()*drawnTileWidth;
				y = mStartY + bw->GetTileY()*drawnTileHeight;
				if(ic->GetInnerBorder())
				{
					x += bw->GetTileX()*drawnBorderWidth;
					y += bw->GetTileY()*drawnBorderWidth;
				}
				else
				{
					x += drawnBorderWidth;
					y += drawnBorderWidth;
				}
				if(bw->GetViewModule() != 0)
				{
					v = bw->GetViewModule()->GetWindowNumber();
					c = bw->GetViewModule()->GetBackgroundColor();
				}
				else
				{
					v = -1;
					c = white;
				}
				if(ic->GetInnerBorder())
				{
					this->DrawWindow(x,y,drawnTileWidth+2*drawnBorderWidth,drawnTileHeight+2*drawnBorderWidth,c,drawnBorderWidth,ic->GetBorderColor(),v,&bw->GetWallpaperImage(),i==mDialog->GetCurrentBackgroundWindow());
				}
				else
				{
					this->DrawWindow(x,y,drawnTileWidth,drawnTileHeight,c,0,ic->GetBorderColor(),v,&bw->GetWallpaperImage(),i==mDialog->GetCurrentBackgroundWindow());
				}
			}
		}

	
		virtual void DrawForegroundBody()
		{
			int i, x, y, w, h, v, n, bw;
			iColor bc;
			iImageComposerForegroundWindow *fw;

			//
			//  Cache a few things
			//
			iImageComposer *ic = mDialog->GetImageComposer();

			//
			//  Draw
			//
			bool needUpdate = false;
			n = ic->GetNumberOfForegroundWindows();
			int cfw = mDialog->GetCurrentForegroundWindow();
			for(i=0; i<n; i++)
			{
				fw = ic->GetForegroundWindow(i);

				//
				//  Compute window parameters
				//
				if(i==cfw && mIsMovingForegroundWindow)
				{
					x = round(mScaleFactor*(fw->GetPositionX()+mPosX-mOldPosX));
					y = round(mScaleFactor*(fw->GetPositionY()+mPosY-mOldPosY));
				}
				else
				{
					x = round(mScaleFactor*fw->GetPositionX());
					y = round(mScaleFactor*fw->GetPositionY());
				}

				w = round(mScaleFactor*fw->GetImageWidth());
				h = round(mScaleFactor*fw->GetImageHeight());

				if(x < 0) x = 0;
				if(y < 0) y = 0;
				if(x+w >= mDrawnFullWidth)
				{
					x = mDrawnFullWidth - w;
				}
				if(y+h >= mDrawnFullHeight)
				{
					y = mDrawnFullHeight - h;
				}
				x += mStartX;
				y += mStartY;
				v = fw->GetViewModule()->GetWindowNumber();
				bw = 1 + int(mScaleFactor*fw->GetBorderWidth()); if(fw->GetBorderWidth() == 0) bw = 0;
				bc = fw->GetBorderColor();

				//
				//  Is there a zoom source?
				//
				if(fw->GetZoomSource() != 0)
				{
					fw->UpdateWindow();
					needUpdate = true;

					int zx = mStartX + round(mScaleFactor*(fw->GetZoomSource()->GetContentsX()+(fw->GetZoomX()*fw->GetZoomSource()->GetContentsWidth()-0.5*fw->GetZoomWidth())));
					int zy = mStartY + round(mScaleFactor*(fw->GetZoomSource()->GetContentsY()+(fw->GetZoomY()*fw->GetZoomSource()->GetContentsHeight()-0.5*fw->GetZoomHeight())));
					int zw = round(mScaleFactor*fw->GetZoomWidth());
					int zh = round(mScaleFactor*fw->GetZoomHeight());

					this->DrawWindowBase(zx,zy,zw,zh,iColor::Invalid(),bw,bc);

					int wx, wy;
					if(fw->GetCoordinatesForZoomLine(0,wx,wy,x+bw/2,y+bw/2,w-bw+1,h-bw+1)) mSubject->DrawLine(zx+bw/2,zy+bw/2,wx,wy,bc,bw);
					if(fw->GetCoordinatesForZoomLine(1,wx,wy,x+bw/2,y+bw/2,w-bw+1,h-bw+1)) mSubject->DrawLine(zx+zw-1-bw/2,zy+bw/2,wx,wy,bc,bw);
					if(fw->GetCoordinatesForZoomLine(2,wx,wy,x+bw/2,y+bw/2,w-bw+1,h-bw+1)) mSubject->DrawLine(zx+bw/2,zy+zh-1-bw/2,wx,wy,bc,bw);
					if(fw->GetCoordinatesForZoomLine(3,wx,wy,x+bw/2,y+bw/2,w-bw+1,h-bw+1)) mSubject->DrawLine(zx+zw-1-bw/2,zy+zh-1-bw/2,wx,wy,bc,bw);
				}

				//
				//  Draw the window itself
				//
				this->DrawWindow(x,y,w,h,fw->GetViewModule()->GetBackgroundColor(),bw,bc,v,0,i==mDialog->GetCurrentForegroundWindow());
			}

			if(needUpdate) mDialog->UpdateZooms();

			if(!ic->IsActive())
			{
				w = round(0.9*mSubject->Width());
				h = round(0.9*mSubject->Height());
				x = (mSubject->Width()-w)/2;
				y = (mSubject->Height()-h)/2;
				mSubject->DrawTextLine(x,y,x+w-1,y+h-1,"BYPASSED",iColor(200,0,0));
			}
		}

		//
		//  Helpers
		//
		void DrawWindow(int x, int y, int w, int h, const iColor &c, int b, const iColor &bc, int v, const iImage* image, bool focus)
		{
			this->DrawWindowBase(x,y,w,h,c,b,bc);

			int x1 = x + b;
			int y1 = y + b;
			int x2 = x + w - b - 1;
			int y2 = y + h - b - 1;

			if(image != 0)
			{
				mSubject->DrawImage(x1,y1,x2,y2,*image,iImage::_Both);
			}
			if(v >= 0)
			{
				mSubject->DrawTextLine(x1,y1,x2,y2,iString::FromNumber(v+1),c.Reverse());
			}
			if(focus)
			{
				int j;
				for(j=0; j<4; j++)
				{
					mSubject->DrawLine(x1+j,y1+j,x2-j,y1+j,iColor::Black(),1,true);
					mSubject->DrawLine(x1+j,y2-j,x2-j,y2-j,iColor::Black(),1,true);
					mSubject->DrawLine(x1+j,y1+j,x1+j,y2-j,iColor::Black(),1,true);
					mSubject->DrawLine(x2-j,y1+j,x2-j,y2-j,iColor::Black(),1,true);
				}
			}
		}

		void DrawWindowBase(int x, int y, int w, int h, const iColor &c, int b, const iColor &bc)
		{
			if(b > 1)
			{
				mSubject->DrawRectangle(x    ,y    ,x+b-1,y+h-1,bc,bc,0);
				mSubject->DrawRectangle(x+w-b,y    ,x+w-1,y+h-1,bc,bc,0);
				mSubject->DrawRectangle(x    ,y    ,x+w-1,y+b-1,bc,bc,0);
				mSubject->DrawRectangle(x    ,y+h-b,x+w-1,y+h-1,bc,bc,0);
				if(c.IsValid()) mSubject->DrawRectangle(x+b  ,y+b  ,x+w-b-1,y+h-b-1,bc,c,0);
			}
			else
			{
				if(c.IsValid()) mSubject->DrawRectangle(x,y,x+w-1,y+h-1,bc,c,b); else if(b == 1)
				{
					mSubject->DrawLine(x    ,y    ,x    ,y+h-1,bc,1);
					mSubject->DrawLine(x+w-1,y    ,x+w-1,y+h-1,bc,1);
					mSubject->DrawLine(x    ,y    ,x+w-1,y    ,bc,1);
					mSubject->DrawLine(x    ,y+h-1,x+w-1,y+h-1,bc,1);
				}
			}
		}

		bool FindPosition(iImageComposer *ic, int dx, int dy, int &x, int &y)
		{
			x = round((dx-mStartX)/mScaleFactor);
			y = round((dy-mStartY)/mScaleFactor);
			return x>=0 && y>=0 && x<ic->GetImageWidth() && y<ic->GetImageHeight();
		}

		const float mSizeFactor;
		float mScaleFactor;
		int mStartX, mStartY, mDrawnFullWidth, mDrawnFullHeight;
		int mPosX, mPosY, mOldPosX, mOldPosY;
		bool mIsMovingForegroundWindow;
		iggDialogImageComposer *mDialog;
	};

	//
	//  Image size label
	//
	class ImageSizeLabel : public iggWidgetTextArea
	{

	public:

		ImageSizeLabel(iggDialogImageComposer *dialog, iggFrame *parent) : iggWidgetTextArea("%b640 x 480",parent)
		{
			mDialog = dialog;
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			this->SetText("%b"+iString::FromNumber(mDialog->GetImageComposer()->GetImageWidth())+" x "+iString::FromNumber(mDialog->GetImageComposer()->GetImageHeight()));
			this->iggWidgetTextArea::UpdateWidgetBody();
		}

		iggDialogImageComposer *mDialog;
	};

	//
	//  Window list combo box
	//
	class WindowListComboBox : public iggWidget
	{

	public:

		WindowListComboBox(bool bg, int comp, const iString &title, iggDialogImageComposer *dialog, iggFrame *parent) : iggWidget(parent)
		{
			mDialog = dialog;
			mComponent = (comp>0 && comp<3) ? comp : 0;
			mBackground = bg;
			mSubject = iggSubjectFactory::CreateWidgetComboBoxSubject(this,title);
			this->SetBaloonHelp("Assign a visualization window to a background tile");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			int i, val = this->GetShell()->GetControlModule()->GetNumberOfViewModules();
			if(mSubject->Count() != val+1)
			{
				mSubject->Clear();
				if(mBackground || mComponent>0) mSubject->InsertItem("None"); else mSubject->InsertItem("Select...");
				for(i=0; i<val; i++) mSubject->InsertItem("Window #"+iString::FromNumber(i+1));
				mSubject->SetValue(0);
			}
			if(mBackground)
			{
				if(mDialog->GetCurrentBackgroundWindow() < 0) this->Enable(false); else
				{
					this->Enable(true);
					if(mDialog->GetImageComposer()->GetBackgroundWindow(mDialog->GetCurrentBackgroundWindow())->GetViewModule(mComponent) == 0)
					{
						mSubject->SetValue(0);
					}
					else
					{
						mSubject->SetValue(mDialog->GetImageComposer()->GetBackgroundWindow(mDialog->GetCurrentBackgroundWindow())->GetViewModule(mComponent)->GetWindowNumber()+1);
					}
				}
			}
			else if(mComponent > 0)
			{
				if(mDialog->GetCurrentForegroundWindow() < 0) this->Enable(false); else
				{
					this->Enable(true);
					if(mDialog->GetImageComposer()->GetForegroundWindow(mDialog->GetCurrentForegroundWindow())->GetViewModule(mComponent) == 0)
					{
						mSubject->SetValue(0);
					}
					else
					{
						mSubject->SetValue(mDialog->GetImageComposer()->GetForegroundWindow(mDialog->GetCurrentForegroundWindow())->GetViewModule(mComponent)->GetWindowNumber()+1);
					}
				}
			}
		}

		virtual void OnInt1Body(int v)
		{
			if(mBackground)
			{
				if(mDialog->GetCurrentBackgroundWindow() >= 0)
				{
					if(v == 0)
					{
						mDialog->GetImageComposer()->GetBackgroundWindow(mDialog->GetCurrentBackgroundWindow())->SetViewModule(0,mComponent);
					}
					else
					{
						mDialog->GetImageComposer()->GetBackgroundWindow(mDialog->GetCurrentBackgroundWindow())->SetViewModule(this->GetShell()->GetControlModule()->GetViewModule(v-1),mComponent);
					}
				}
			}
			else
			{
				if(mComponent == 0)
				{
					if(v > 0)
					{
						mDialog->GetImageComposer()->AddForegroundWindow(this->GetShell()->GetControlModule()->GetViewModule(v-1));
						mDialog->SetCurrentForegroundWindow(mDialog->GetImageComposer()->GetNumberOfForegroundWindows()-1);
					}
				}
				else
				{
					if(v == 0)
					{
						mDialog->GetImageComposer()->GetForegroundWindow(mDialog->GetCurrentForegroundWindow())->SetViewModule(0,mComponent);
					}
					else
					{
						mDialog->GetImageComposer()->GetForegroundWindow(mDialog->GetCurrentForegroundWindow())->SetViewModule(this->GetShell()->GetControlModule()->GetViewModule(v-1),mComponent);
					}
				}
			}
		}

		bool mBackground;
		int mComponent;
		ibgWidgetComboBoxSubject *mSubject;
		iggDialogImageComposer *mDialog;
	};

	//
	//  Load wallpaper
	//
	class LoadWallpaperButton : public iggWidgetSimpleButton
	{

	public:

		LoadWallpaperButton(iggDialogImageComposer *dialog, iggFrame *parent) : iggWidgetSimpleButton("Empty tile background >>",parent)
		{
			mDialog = dialog;
			this->SetBaloonHelp("Load a background image for an empty tile");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			this->Enable(mDialog->GetCurrentBackgroundWindow()>=0 && mDialog->GetImageComposer()->GetBackgroundWindow(mDialog->GetCurrentBackgroundWindow())->GetViewModule() == 0);
		}

		virtual void Execute()
		{
			iString fn;
			if(mOldFileName.IsEmpty()) mOldFileName = this->GetShell()->GetEnvironment(Environment::Base);
			fn = this->GetMainWindow()->GetFileName("Open image file",mOldFileName,"Images (*.jpg *.jpeg *.pnm *.bmp *.png *.tif *.tiff)");

			if(mDialog->GetCurrentBackgroundWindow()>=0 && !mDialog->GetImageComposer()->GetBackgroundWindow(mDialog->GetCurrentBackgroundWindow())->LoadWallpaperImage(fn))
			{
				this->GetMainWindow()->PopupWindow(mDialog->GetFrame(),"Failed to load an image from file.",PopupWindow::Error);
			}
		}

		iString mOldFileName;
		iggDialogImageComposer *mDialog;
	};

	//
	//  remove foreground window
	//
	class RemoveWindowButton : public iggWidgetSimpleButton
	{

	public:

		RemoveWindowButton(iggDialogImageComposer *dialog, iggFrame *parent) : iggWidgetSimpleButton("Remove",parent)
		{
			mDialog = dialog;
			this->SetBaloonHelp("Remove current foreground window");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			this->Enable(mDialog->GetImageComposer()->GetNumberOfForegroundWindows()>0);
		}

		virtual void Execute()
		{
			mDialog->GetImageComposer()->RemoveForegroundWindow(mDialog->GetCurrentForegroundWindow());
			mDialog->SetCurrentForegroundWindow(mDialog->GetCurrentForegroundWindow());
			if(mDialog->GetImageComposer()->GetNumberOfForegroundWindows() == 0) this->Enable(false);
		}

		iggDialogImageComposer *mDialog;
	};

	//
	//  Scale discrete float spin box
	//
	class ScaleSpinBox : public iggWidgetKeyHandler<float>
	{

	public:

		ScaleSpinBox(const iObjectKey &key, iggDialogImageComposer *dialog, iggFrame *parent) : iggWidgetKeyHandler<float>(WidgetType::Other,key,RenderMode::NoRender,parent,0,0)
		{
			mSubject = iggSubjectFactory::CreateWidgetSpinBoxSubject(this,5,100,"",5);
			mDialog = dialog;
			this->SetBaloonHelp("Scale foreground window","Use this box to scale the image of the foreground window in the composed final image. The window size on the screen will not be changed, only the image of it will.");
		}

	protected:

		virtual void OnInt1Body(int)
		{
			this->ExecuteControl(true);
		}

		virtual void QueryValue(float &v) const
		{
			v = 0.01f*mSubject->GetValue();
		}

		virtual void UpdateValue(float v)
		{
			mSubject->SetValue(round(100.0*v));
		}

		ibgWidgetSpinBoxSubject *mSubject;
		iggDialogImageComposer *mDialog;
	};

	//
	//  Frame box that becomes disabled when there are no foreground widgets
	//
	class WindowPropertiesBox : public iggFrame
	{

	public:

		WindowPropertiesBox(iggDialogImageComposer *dialog, iggFrame *parent) : iggFrame("Current window...",parent,2)
		{
			mDialog = dialog;
		}

		virtual ~WindowPropertiesBox()
		{
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			if(mDialog->GetCurrentForegroundWindow() >= 0)
			{
				this->Enable(true);
				this->iggFrame::UpdateWidgetBody();
			}
			else this->Enable(false);
		}

		iggDialogImageComposer *mDialog;
	};

	//
	//  Frame box with extra VisualModule selections for pseudo-color composition
	//
	class PseudoColorBox : public iggFrame
	{

	public:

		PseudoColorBox(bool bg, iggDialogImageComposer *dialog, iggFrame *parent) : iggFrame("Pseudo color",parent,2)
		{
			mDialog = dialog;
			mBackground = bg;
			this->AddLine(new iggWidgetTextArea("Green channel",this),new WindowListComboBox(bg,1,"",dialog,this));
			this->AddLine(new iggWidgetTextArea("Blue channel",this),new WindowListComboBox(bg,2,"",dialog,this));
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			if(mBackground)
			{
				this->Enable(mDialog->GetImageComposer()->GetBackgroundWindow(mDialog->GetCurrentBackgroundWindow())!=0 && mDialog->GetImageComposer()->GetBackgroundWindow(mDialog->GetCurrentBackgroundWindow())->GetViewModule()!=0);
			}
			else
			{
				this->Enable(mDialog->GetImageComposer()->GetForegroundWindow(mDialog->GetCurrentForegroundWindow())!=0 && mDialog->GetImageComposer()->GetForegroundWindow(mDialog->GetCurrentForegroundWindow())->GetViewModule()!=0);
			}
			this->iggFrame::UpdateWidgetBody();
		}

		bool mBackground;
		iggDialogImageComposer *mDialog;
	};

	//
	//  Zoom source list combo box
	//
	class ZoomSourceListComboBox : public iggWidget
	{

	public:

		ZoomSourceListComboBox(const iString &title, iggDialogImageComposer *dialog, iggFrame *parent) : iggWidget(parent)
		{
			mNumBG = mNumFG = 0;
			mDialog = dialog;
			mSubject = iggSubjectFactory::CreateWidgetComboBoxSubject(this,title);
			this->SetBaloonHelp("Attach this window to another window as a zoom");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			int nbg = mDialog->GetImageComposer()->GetNumberOfBackgroundWindows();
			int nfg = mDialog->GetImageComposer()->GetNumberOfForegroundWindows();

			if(mNumFG!=nfg || mNumBG!=nbg)
			{
				mNumBG = nbg;
				mNumFG = nfg;
				mSubject->Clear();
				mSubject->InsertItem("None");

				int i;
				iImageComposer *c = mDialog->GetImageComposer();
				iImageComposerBackgroundWindow *wbg;
				for(i=0; i<nbg; i++)
				{
					wbg = c->GetBackgroundWindow(i);
					mSubject->InsertItem("Background ("+iString::FromNumber(wbg->GetTileX()+1)+","+iString::FromNumber(wbg->GetTileY()+1)+")");
				}
				for(i=0; i<nfg; i++)
				{
					mSubject->InsertItem("Foreground #"+iString::FromNumber(i+1));
				}
			}

			iImageComposerForegroundWindow *w = mDialog->GetImageComposer()->GetForegroundWindow(mDialog->GetCurrentForegroundWindow());
			if(w==0 || w->GetZoomSource()==0)
			{
				mSubject->SetValue(0);
			}
			else
			{
				int v = w->GetZoomSource()->GetId();
				if(v < 0) mSubject->SetValue(-v); else mSubject->SetValue(v+mNumBG);
			}
		}

		virtual void OnInt1Body(int v)
		{
			iImageComposerForegroundWindow *w = mDialog->GetImageComposer()->GetForegroundWindow(mDialog->GetCurrentForegroundWindow());
			if(w == 0)
			{
				IERROR_LOW("Current foreground window not set.");
				return;
			}

			if(v == 0)
			{
				w->SetZoomSource(0);
			}
			else if(v <= mNumBG)
			{
				w->SetZoomSource(mDialog->GetImageComposer()->GetBackgroundWindow(v-1));
			}
			else
			{
				w->SetZoomSource(mDialog->GetImageComposer()->GetForegroundWindow(v-mNumBG-1));
				this->UpdateWidget();
			}
		}

		int mNumBG, mNumFG;
		ibgWidgetComboBoxSubject *mSubject;
		iggDialogImageComposer *mDialog;
	};

	//
	//  update zooms button
	//
	class UpdateZoomsButton : public iggWidgetSimpleButton
	{

	public:

		UpdateZoomsButton(iggDialogImageComposer *dialog, iggFrame *parent) : iggWidgetSimpleButton("Update",parent)
		{
			mDialog = dialog;
			this->SetBaloonHelp("Update zoom arrangements");
		}

	protected:

		virtual void Execute()
		{
			mDialog->UpdateZooms();
		}

		iggDialogImageComposer *mDialog;
	};

	//
	//  zoom widgets frame
	//
	class ZoomWidgetsFrame : public iggFrame
	{

	public:

		ZoomWidgetsFrame(iggDialogImageComposer *dialog, iggFrame *parent) : iggFrame(parent,2)
		{
			mDialog = dialog;
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			iImageComposerForegroundWindow *win = mDialog->GetImageComposer()->GetForegroundWindow(mDialog->GetCurrentForegroundWindow());

			if(win!=0 && win->GetZoomSource()!=0)
			{
				win->UpdateWindow();
				this->Enable(!mDialog->GetImageComposer()->GetForegroundWindow(mDialog->GetCurrentForegroundWindow())->IsAutoZoom());
			}
		}

		iggDialogImageComposer *mDialog;
	};
};


using namespace iggDialogImageComposer_Private;


iggDialogImageComposer::iggDialogImageComposer(iggMainWindow *parent) : iggDialog(parent,0U,iImageFactory::FindIcon("imcomp.png"),"Image Composer","sr.gg.di",2)
{
	mCurrentBackgroundWindow = -1;
	mCurrentForegroundWindow = -1;

	iggFrame *tmp, *tmp2;

	tmp = new iggFrame(mFrame);
	tmp2 = new iggFrame(tmp,3);
	mLabel = new ImageSizeLabel(this,tmp2);
	tmp2->AddLine(new iggWidgetTextArea("Image size: ",tmp2),mLabel);
	tmp2->SetColStretch(0,0);
	tmp2->SetColStretch(1,0);
	tmp2->SetColStretch(2,10);
	tmp->AddLine(tmp2);

	mArea = new Area(this,tmp);
	tmp->AddLine(mArea);
	tmp->SetRowStretch(1,10);

	iggFrameBook *book = new iggFrameBook(mFrame);

	mFrame->AddLine(tmp,book);

	//
	//  Background page
	//
	tmp = new iggFrame(book,2);
	book->AddPage("Background",0,tmp);

	iggFrame *b = new iggFrame("Number of tiles",tmp,2);
	iggWidget *wid;
	wid = new iggWidgetKeySpinBox(1,10,"",0,iImageComposer::KeyNumTiles(),b,0);
	wid->AddDependent(mArea);
	wid->AddDependent(mLabel);
	b->AddLine(new iggWidgetTextArea("Horizontally",b),wid);
	wid = new iggWidgetKeySpinBox(1,10,"",0,iImageComposer::KeyNumTiles(),b,1);
	wid->AddDependent(mArea);
	wid->AddDependent(mLabel);
	b->AddLine(new iggWidgetTextArea("Vertically",b),wid);
	
	tmp->AddLine(b,2);

	b = new iggFrame("Current tile...",tmp);
	mBackgroundWindowsList = new WindowListComboBox(true,0,"Assign window",this,b);
	mBackgroundWindowsList->AddDependent(mArea);
	mBackgroundWindowsList->Enable(false);
	wid = new LoadWallpaperButton(this,b);
	wid->AddBuddy(mArea);
	PseudoColorBox *psb = new PseudoColorBox(true,this,b);
	mBackgroundWindowsList->AddDependent(psb);
	mArea->AddDependent(psb);
	b->AddLine(mBackgroundWindowsList);
	b->AddLine(wid);
	b->AddLine(psb);

	tmp->AddLine(b,2);
	tmp->AddSpace(10);

	tmp2 = new iggFrame(tmp,2);
	wid = new iggWidgetKeySpinBox(0,100,"",0,iImageComposer::KeyBorderWidth(),tmp2);
	wid->AddDependent(mArea);
	wid->AddDependent(mLabel);
	tmp2->AddLine(new iggWidgetTextArea("Border width  ",tmp2),wid);
	iggWidgetKeyColorSelection *bc = new iggWidgetKeyColorSelection(iImageComposer::KeyBorderColor(),tmp2,true);
	bc->SetText("Set");
	bc->AddDependent(mArea);
	tmp2->AddLine(new iggWidgetTextArea("Border color  ",tmp2),bc);
	wid = new iggWidgetKeyCheckBox("Draw border between tiles",iImageComposer::KeyInnerBorder(),tmp2);
	wid->AddDependent(mArea);
	wid->AddDependent(mLabel);
	tmp2->AddLine(wid,2);
	wid = new iggWidgetKeyCheckBox("Scale background to tile size",iImageComposer::KeyScaleBackground(),tmp2);
	wid->AddDependent(mArea);
	wid->AddDependent(mLabel);
	tmp2->AddLine(wid,2);
	tmp2->SetColStretch(1,10);

	tmp->AddLine(tmp2);
	tmp->AddSpace(10);

	//
	//  Foreground page
	//
	tmp = new iggFrame(book);
	book->AddPage("Foreground",0,tmp);

	wid = new WindowListComboBox(false,0,"Add new window",this,tmp);
	wid->AddDependent(mArea);

	tmp->AddLine(wid);
	tmp->AddSpace(10);

	b = new WindowPropertiesBox(this,tmp);
	wid->AddDependent(b);
	wid = new RemoveWindowButton(this,b);
	wid->AddDependent(mArea);
	b->AddLine(wid);
	mIndexHolders[0] = new ScaleSpinBox(iImageComposer::KeyForegroundWindowScale(),this,b);
	mIndexHolders[0]->AddDependent(mArea);
	b->AddLine(new iggWidgetTextArea("Scale",b),mIndexHolders[0]);
	mIndexHolders[1] = new iggWidgetKeySpinBox(0,100,"",0,iImageComposer::KeyForegroundWindowBorderWidth(),b,0);
	mIndexHolders[1]->AddDependent(mArea);
	b->AddLine(new iggWidgetTextArea("Border width",b),mIndexHolders[1]);
	mIndexHolders[2] = new iggWidgetKeyColorSelection(iImageComposer::KeyForegroundWindowBorderColor(),b,true,0);
	mIndexHolders[2]->AddDependent(mArea);
	b->AddLine(new iggWidgetTextArea("Border color",b),mIndexHolders[2]);
	psb = new PseudoColorBox(false,this,b);
	wid->AddDependent(psb);
	mArea->AddDependent(psb);
	b->AddLine(psb,2);

	tmp2 = new iggFrame("Show as zoom",b,2);
	wid = new ZoomSourceListComboBox("",this,tmp2);
	wid->AddDependent(mArea);
	tmp2->AddLine(new iggWidgetTextArea("Source",tmp2),wid);

	iggFrame *tmp3 = new ZoomWidgetsFrame(this,tmp2);
	mIndexHolders[3] = new iggWidgetKeyFloatSlider(0.0,1.0,100,0,3,"",iImageComposer::KeyForegroundWindowZoomX(),RenderMode::NoRender,tmp3,0);
	mIndexHolders[3]->AddDependent(mArea);
	tmp3->AddLine(new iggWidgetTextArea("Center X",tmp3),mIndexHolders[3]);
	mIndexHolders[4] = new iggWidgetKeyFloatSlider(0.0,1.0,100,0,3,"",iImageComposer::KeyForegroundWindowZoomY(),RenderMode::NoRender,tmp3,0);
	mIndexHolders[4]->AddDependent(mArea);
	tmp3->AddLine(new iggWidgetTextArea("Center Y",tmp3),mIndexHolders[4]);
	mIndexHolders[5] = new iggWidgetKeyFloatSlider(0.0,1.0,100,0,3,"",iImageComposer::KeyForegroundWindowZoomFactor(),RenderMode::NoRender,tmp3,0);
	mIndexHolders[5]->AddDependent(mArea);
	tmp3->AddLine(new iggWidgetTextArea("Factor",tmp3),mIndexHolders[5]);
	tmp3->SetColStretch(1,10);

	tmp2->AddLine(tmp3,2);

	mIndexHolders[6] = new iggWidgetKeyCheckBox("Use 4-line zoom",iImageComposer::KeyForegroundWindowZoom4Line(),tmp2,0);
	mIndexHolders[6]->AddDependent(mArea);
	tmp2->AddLine(mIndexHolders[6],2);

	tmp2->SetColStretch(1,10);

	b->AddSpace(2);
	b->AddLine(tmp2,2);

	tmp->AddLine(b);
	tmp->AddSpace(10);

	mFrame->SetColStretch(0,10);

	this->ResizeContents(700,500);
}


void iggDialogImageComposer::SetCurrentBackgroundWindow(int w)
{
	if(w>=0 && w<this->GetImageComposer()->GetNumTilesX()*this->GetImageComposer()->GetNumTilesY())
	{
		mCurrentBackgroundWindow = w;
	}
	else mCurrentBackgroundWindow = -1;
	mBackgroundWindowsList->UpdateWidget();
}


void iggDialogImageComposer::SetCurrentForegroundWindow(int w)
{
	int i;

	if(w>=0 && w<this->GetImageComposer()->GetNumberOfForegroundWindows())
	{
		this->GetImageComposer()->MoveToBack(w);
		mCurrentForegroundWindow = this->GetImageComposer()->GetNumberOfForegroundWindows() - 1;
		for(i=0; i<mNumIndexHolders; i++)
		{
			mIndexHolders[i]->SetIndex(mCurrentForegroundWindow);
		}
	}
	else mCurrentForegroundWindow = -1;
	this->UpdateDialog();
}


void iggDialogImageComposer::UpdateView()
{
	this->GetImageComposer()->UpdateSize();
	if(this->IsVisible())
	{
		mArea->UpdateWidget();
		mLabel->UpdateWidget();
	}
}


void iggDialogImageComposer::UpdateZooms()
{
	this->GetImageComposer()->ClearCache();
	this->GetImageComposer()->UpdateWindowList();
//	this->UpdateDialog();
	int i;
	for(i=3; i<mNumIndexHolders; i++)
	{
		mIndexHolders[i]->UpdateWidget();
	}
}


iImageComposer* iggDialogImageComposer::GetImageComposer() const
{
	return this->GetShell()->GetControlModule()->GetImageComposer();
}

#endif
