/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggframedatavariablelist.h"


#include "ibuffer.h"
#include "icontrolmodule.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatasubject.h"
#include "ierror.h"
#include "ishell.h"

#include "iggdatatypeprovider.h"
#include "iggwidgetarea.h"
#include "iggwidgetkeyselectionbox.h"

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "ibuffertemplate.h"
#include "iggreloadingtemplate.h"

iggFrameDataVariableList::iggFrameDataVariableList(const iggDataTypeProvider *provider, const iString &title, const iObjectKey &key, const iObjectKey *actkey, iggFrame *parent, int index, const iObjectKey *indkey) : iggFrame(parent,1), mKey(key), mActivationKey(actkey)
{
	//
	//  Decide, whether we are doing a ComboBox or a RadioBox
	//
	iggFrame *tmp = new iggFrame(title,this);
	mWidgetCombo = tmp;
	mBoxCombo = new iggWidgetKeyComboBox("",0,key,tmp,index,indkey);
	tmp->AddLine(mBoxCombo);
	mWidgetRadio = mBoxRadio = new iggWidgetKeyRadioBox(1,title,0,key,this,index,indkey);

	this->Define(provider);
}


iggFrameDataVariableList::iggFrameDataVariableList(const iDataInfo &info, const iggDataTypeProvider *provider, const iString &title, const iObjectKey &key, const iObjectKey *actkey, iggFrame *parent, int index, const iObjectKey *indkey) : iggFrame(parent,1), mKey(key), mActivationKey(actkey)
{
	//
	//  Decide, whether we are doing a ComboBox or a RadioBox
	//
	iggFrame *tmp = new iggFrame(title,this);
	mWidgetCombo = tmp;
	mBoxCombo = new iggReloading<iggWidgetKeyComboBox>(info,provider,"",0,key,tmp,index,indkey);
	tmp->AddLine(mBoxCombo);
	mWidgetRadio = mBoxRadio = new iggReloading<iggWidgetKeyRadioBox>(info,provider,1,title,0,key,this,index,indkey);

	this->Define(provider);
}


void iggFrameDataVariableList::Define(const iggDataTypeProvider *provider)
{
	mProvider = provider; IERROR_ASSERT(provider);

	mBoxCombo->AttachDataTypeProvider(provider);
	mBoxRadio->AttachDataTypeProvider(provider);

	mCompleted = false;
	mOffset = 0;

	this->AddLine(mWidgetCombo);
	this->AddLine(mWidgetRadio);
	this->AddSpace(1);

	mWidgetCombo->Show(false);
	mWidgetRadio->Show(false);
}


void iggFrameDataVariableList::SetIndex(int i)
{
	mBoxCombo->SetIndex(i);
	mBoxCombo->UpdateWidget();
	mBoxRadio->SetIndex(i);
	mBoxRadio->UpdateWidget();
}


void iggFrameDataVariableList::Complete(int extraoffset)
{
	if(!mCompleted)
	{
		mCompleted = true;

		mOffset = mBoxCombo->Count();
		mBoxCombo->SetOffset(mOffset + extraoffset);
		mBoxRadio->SetOffset(mOffset + extraoffset);

		this->UpdateList();
	}
	else 
	{
		IERROR_LOW("Cannot complete FrameDataVariableList twice.");
	}
}


void iggFrameDataVariableList::UpdateList()
{
	static iBuffer<iString> s;
	int i, np, na, n;
	bool ok;

	if(!mCompleted)
	{
		IERROR_LOW("FrameDataVariableList must be completed before the first update.");
		return;
	}

	//
	//  We need to set the DataType first
	//
	const iObjectKey &keyNumVars = mProvider->Translate(iDataSubject::KeyNumVars());
	const iObjectKey &keyName = mProvider->Translate(iDataSubject::KeyName());

	//
	//  How much data is loaded?
	//
	if(!this->GetShell()->GetControlModule()->QueryValue(keyNumVars,n))
	{
		IERROR_LOW("Unable to query the number of variables.");
		return;
	}

	//
	//  Check an activation key
	//
	na = n;
	if(mActivationKey != 0)
	{
		if(this->GetShell()->GetControlModule()->QueryValue(*mActivationKey,ok) && !ok) na = 0;
	}

	np = mBoxCombo->Count() - mOffset;
	s.Extend(n);
	if(n==0 || this->GetShell()->GetControlModule()->QueryValue(keyName,s,n))
	{
		for(i=0; i<n; i++)
		{
			mBoxCombo->SetItem(s[i],mOffset+i,i<na);
			if(i < 3) mBoxRadio->SetItem(s[i],mOffset+i,i<na);
		}
		for(i=n; i<3; i++)
		{
			mBoxRadio->SetItem("",mOffset+i,false);
		}
		for(i=n; i<np; i++)
		{
			mBoxCombo->RemoveItem(mOffset+n); // this deletes an item, so they shift down
		}
	}
}


void iggFrameDataVariableList::UpdateChildren()
{
	if(!mCompleted)
	{
		IERROR_LOW("FrameDataVariableList must be completed before the first update.");
		return;
	}

	this->UpdateList();

	if(mBoxCombo->Count() > 3+mOffset) // RadioBox cannot be too large
	{
		mWidgetCombo->Show(true);
		mWidgetRadio->Show(false);
	}
	else
	{
		mWidgetCombo->Show(false);
		mWidgetRadio->Show(true);
	}

	iggFrame::UpdateChildren();
}


void iggFrameDataVariableList::InsertItem(const iString &text)
{
	if(!mCompleted)
	{
		mBoxCombo->InsertItem(text);
		mBoxRadio->InsertItem(text);
	}
	else
	{
		IERROR_LOW("Cannot insert items in the completed FrameDataVariableList.");
	}
}

#endif
