/*
 * indicator-network - user interface for connman
 * Copyright 2010 Canonical Ltd.
 *
 * Authors:
 * Kalle Valo <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "connman-service.h"

#include <glib/gi18n.h>

#include "connman-service-xml.h"
#include "connman.h"
#include "marshal.h"

G_DEFINE_TYPE(ConnmanService, connman_service, G_TYPE_OBJECT)

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE((o), CONNMAN_TYPE_SERVICE, \
			       ConnmanServicePrivate))

typedef struct _ConnmanServicePrivate ConnmanServicePrivate;

struct _ConnmanServicePrivate {
  gchar *path;
  gboolean ready;

  /* connman service properties */
  ConnmanServiceState state;
  gchar *error;
  gchar *name;
  ConnmanServiceType type;
  ConnmanServiceSecurity security;
  guint8 strength;
  gboolean setup_required;
};

enum
{
  /* reserved */
  PROP_0,

  PROP_PATH,
  PROP_READY,

  PROP_STATE,
  PROP_ERROR,
  PROP_NAME,
  PROP_TYPE,
  PROP_SECURITY,
  PROP_STRENGTH,
  PROP_SETUP_REQUIRED,
};

#define SERVICE_NAME_UNKNOWN "<unnamed>"

struct service_state_string
{
  const gchar *str;
  ConnmanServiceState state;
};

static const struct service_state_string service_state_map[] = {
  { "idle",		CONNMAN_SERVICE_STATE_IDLE },
  { "failure",		CONNMAN_SERVICE_STATE_FAILURE },
  { "association",	CONNMAN_SERVICE_STATE_ASSOCIATION },
  { "configuration",	CONNMAN_SERVICE_STATE_CONFIGURATION },
  { "ready",		CONNMAN_SERVICE_STATE_READY },
  { "login",		CONNMAN_SERVICE_STATE_LOGIN },
  { "online",		CONNMAN_SERVICE_STATE_ONLINE },
  { "disconnect",	CONNMAN_SERVICE_STATE_DISCONNECT },
};

struct service_security_string
{
  const gchar *str;
  ConnmanServiceSecurity security;
};

static const struct service_security_string service_security_map[] = {
  { "none",		CONNMAN_SERVICE_SECURITY_NONE },
  { "wep",		CONNMAN_SERVICE_SECURITY_WEP },
  { "psk",		CONNMAN_SERVICE_SECURITY_PSK },
  { "wpa",		CONNMAN_SERVICE_SECURITY_PSK },
  { "rsn",		CONNMAN_SERVICE_SECURITY_PSK },
  { "ieee8021x",	CONNMAN_SERVICE_SECURITY_IEEE8021X },
};

struct service_type_string
{
  const gchar *str;
  ConnmanServiceType type;
};

static const struct service_type_string service_type_map[] = {
  { "ethernet",		CONNMAN_SERVICE_TYPE_ETHERNET },
  { "wifi",		CONNMAN_SERVICE_TYPE_WIFI },
  { "bluetooth",	CONNMAN_SERVICE_TYPE_BLUETOOTH },
  { "cellular",		CONNMAN_SERVICE_TYPE_CELLULAR },
};

struct service_mode_string
{
  const gchar *str;
  ConnmanServiceMode mode;
};

static const struct service_mode_string service_mode_map[] = {
  { "managed",		CONNMAN_SERVICE_MODE_MANAGED },
  { "adhoc",		CONNMAN_SERVICE_MODE_ADHOC },
};

void connman_service_connect(ConnmanService *self,
			     GAsyncReadyCallback callback,
			     gpointer user_data)
{
}

void connman_service_connect_finish(ConnmanService *self,
				    GAsyncResult *res,
				    GError **error)
{
}

void connman_service_disconnect(ConnmanService *self,
				GAsyncReadyCallback callback,
				gpointer user_data)
{
}

void connman_service_disconnect_finish(ConnmanService *self,
				       GAsyncResult *res,
				       GError **error)
{
}

ConnmanServiceState connman_service_get_state(ConnmanService *self)
{
  ConnmanServicePrivate *priv = GET_PRIVATE(self);

  return priv->state;
}

const gchar *connman_service_get_error(ConnmanService *self)
{
  ConnmanServicePrivate *priv = GET_PRIVATE(self);

  return priv->error;
}

const gchar *connman_service_get_name(ConnmanService *self)
{
  ConnmanServicePrivate *priv = GET_PRIVATE(self);

  return priv->name;
}

ConnmanServiceType connman_service_get_service_type(ConnmanService *self)
{
  ConnmanServicePrivate *priv = GET_PRIVATE(self);

  return priv->type;
}

ConnmanServiceSecurity connman_service_get_security(ConnmanService *self)
{
  ConnmanServicePrivate *priv = GET_PRIVATE(self);

  return priv->security;
}

guint8 connman_service_get_strength(ConnmanService *self)
{
  ConnmanServicePrivate *priv = GET_PRIVATE(self);

  return priv->strength;
}

gboolean connman_service_get_setup_required(ConnmanService *self)
{
  ConnmanServicePrivate *priv = GET_PRIVATE(self);

  return priv->setup_required;
}

const gchar *connman_service_get_path(ConnmanService *self)
{
  ConnmanServicePrivate *priv = GET_PRIVATE(self);

  g_return_val_if_fail(CONNMAN_IS_SERVICE(self), NULL);
  g_return_val_if_fail(priv != NULL, NULL);

  return priv->path;
}

gboolean connman_service_is_ready(ConnmanService *self)
{
  ConnmanServicePrivate *priv = GET_PRIVATE(self);

  g_return_val_if_fail(CONNMAN_IS_SERVICE(self), FALSE);
  g_return_val_if_fail(priv != NULL, FALSE);

  return priv->ready;
}

ConnmanServiceState connman_service_str2state(const gchar *state)
{
  const struct service_state_string *s;
  guint i;

  for (i = 0; i < G_N_ELEMENTS(service_state_map); i++) {
    s = &service_state_map[i];
    if (g_strcmp0(s->str, state) == 0)
      return s->state;
  }

  g_warning("unknown service state %s", state);

  return CONNMAN_SERVICE_STATE_IDLE;
}

const gchar *connman_service_state2str(ConnmanServiceState state)
{
  const struct service_state_string *s;
  guint i;

  for (i = 0; i < G_N_ELEMENTS(service_state_map); i++) {
    s = &service_state_map[i];
    if (s->state == state)
      return s->str;
  }

  g_warning("%s(): unknown state %d", __func__, state);

  return "unknown";
}

ConnmanServiceType connman_service_str2type(const gchar *type)
{
  const struct service_type_string *s;
  guint i;

  for (i = 0; i < G_N_ELEMENTS(service_type_map); i++) {
    s = &service_type_map[i];
    if (g_strcmp0(s->str, type) == 0)
      return s->type;
  }

  g_warning("unknown service type %s", type);

  return CONNMAN_SERVICE_TYPE_ETHERNET;
}

const gchar *connman_service_type2str(ConnmanServiceType type)
{
  const struct service_type_string *s;
  guint i;

  for (i = 0; i < G_N_ELEMENTS(service_type_map); i++) {
    s = &service_type_map[i];
    if (s->type == type)
      return s->str;
  }

  g_warning("%s(): unknown type %d", __func__, type);

  return "unknown";
}

ConnmanServiceSecurity connman_service_str2security(const gchar *security)
{
  const struct service_security_string *s;
  guint i;

  for (i = 0; i < G_N_ELEMENTS(service_security_map); i++) {
    s = &service_security_map[i];
    if (g_strcmp0(s->str, security) == 0)
      return s->security;
  }

  g_warning("unknown service security %s", security);

  return CONNMAN_SERVICE_SECURITY_NONE;
}

const gchar *connman_service_security2str(ConnmanServiceSecurity security)
{
  const struct service_security_string *s;
  guint i;

  for (i = 0; i < G_N_ELEMENTS(service_security_map); i++) {
    s = &service_security_map[i];
    if (s->security == security)
      return s->str;
  }

  g_warning("%s(): unknown security %d", __func__, security);

  return "unknown";
}

ConnmanServiceMode connman_service_str2mode(const gchar *mode)
{
  const struct service_mode_string *s;
  guint i;

  for (i = 0; i < G_N_ELEMENTS(service_mode_map); i++) {
    s = &service_mode_map[i];
    if (g_strcmp0(s->str, mode) == 0)
      return s->mode;
  }

  g_warning("unknown service mode %s", mode);

  return CONNMAN_SERVICE_MODE_MANAGED;
}

const gchar *connman_service_mode2str(ConnmanServiceMode mode)
{
  const struct service_mode_string *s;
  guint i;

  for (i = 0; i < G_N_ELEMENTS(service_mode_map); i++) {
    s = &service_mode_map[i];
    if (s->mode == mode)
      return s->str;
  }

  g_warning("%s(): unknown mode %d", __func__, mode);

  return "unknown";
}

static void connman_service_set_property(GObject *object, guint property_id,
					 const GValue *value,
					 GParamSpec *pspec)
{
  ConnmanService *self = CONNMAN_SERVICE(object);
  ConnmanServicePrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(priv != NULL);

  switch(property_id) {
  case PROP_PATH:
    g_free(priv->path);
    priv->path = g_value_dup_string(value);
    break;
  case PROP_READY:
    priv->ready = g_value_get_boolean(value);
    break;
  case PROP_STATE:
    priv->state = g_value_get_uint(value);
    break;
  case PROP_ERROR:
    g_free(priv->error);
    priv->error = g_value_dup_string(value);
    break;
  case PROP_NAME:
    g_free(priv->name);
    priv->name = g_value_dup_string(value);
    break;
  case PROP_TYPE:
    priv->type = g_value_get_uint(value);
    break;
  case PROP_SECURITY:
    priv->security = g_value_get_uint(value);
    break;
  case PROP_STRENGTH:
    priv->strength = g_value_get_uchar(value);
    break;
  case PROP_SETUP_REQUIRED:
    priv->setup_required = g_value_get_boolean(value);
    break;
  default:
    /* We don't have any other property... */
    G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
    break;
  }
}

static void connman_service_get_property(GObject *object, guint property_id,
					 GValue *value, GParamSpec *pspec)
{
  ConnmanService *self = CONNMAN_SERVICE(object);
  ConnmanServicePrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(priv != NULL);

  switch(property_id) {
  case PROP_PATH:
    g_value_set_string(value, priv->path);
    break;
  case PROP_READY:
    g_value_set_boolean(value, priv->ready);
    break;
  case PROP_STATE:
    g_value_set_uint(value, priv->state);
    break;
  case PROP_ERROR:
    g_value_set_string(value, priv->error);
    break;
  case PROP_NAME:
    g_value_set_string(value, priv->name);
    break;
  case PROP_TYPE:
    g_value_set_uint(value, priv->type);
    break;
  case PROP_SECURITY:
    g_value_set_uint(value, priv->security);
    break;
  case PROP_STRENGTH:
    g_value_set_uchar(value, priv->strength);
    break;
  case PROP_SETUP_REQUIRED:
    g_value_set_boolean(value, priv->setup_required);
    break;
  default:
    /* We don't have any other property... */
    G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
    break;
  }
}

static void connman_service_dispose(GObject *object)
{
  ConnmanService *self = CONNMAN_SERVICE(object);
  ConnmanServicePrivate *priv = GET_PRIVATE(self);

  G_OBJECT_CLASS(connman_service_parent_class)->dispose(object);

  if (priv->path != NULL) {
    g_free(priv->path);
    priv->path = NULL;
  }

  if (priv->error != NULL) {
    g_free(priv->error);
    priv->error = NULL;
  }

  if (priv->name != NULL) {
    g_free(priv->name);
    priv->name = NULL;
  }
}

static void connman_service_finalize(GObject *object)
{
  G_OBJECT_CLASS(connman_service_parent_class)->finalize(object);
}

static void connman_service_class_init(ConnmanServiceClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
  GParamSpec *pspec;

  g_type_class_add_private(klass, sizeof(ConnmanServicePrivate));

  gobject_class->dispose = connman_service_dispose;
  gobject_class->finalize = connman_service_finalize;
  gobject_class->set_property = connman_service_set_property;
  gobject_class->get_property = connman_service_get_property;

  pspec = g_param_spec_string("path",
			      "ConnmanService's DBus path",
			      "Set DBus path for the service",
			      NULL,
			      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY);
  g_object_class_install_property(gobject_class, PROP_PATH, pspec);

  pspec = g_param_spec_boolean("ready",
			       "ConnmanService's ready property",
			       "Informs when object is ready for use",
			       FALSE,
			       G_PARAM_READABLE);
  g_object_class_install_property(gobject_class, PROP_READY, pspec);

  pspec = g_param_spec_uint("state",
			    "ConnmanService's state property",
			    "The state of the service",
			    0, G_MAXUINT,
			    CONNMAN_SERVICE_STATE_IDLE,
			    G_PARAM_READABLE);
  g_object_class_install_property(gobject_class, PROP_STATE, pspec);

  pspec = g_param_spec_string("error",
			      "ConnmanService's error property",
			      "Last error code",
			      NULL,
			      G_PARAM_READABLE);
  g_object_class_install_property(gobject_class, PROP_ERROR, pspec);

  pspec = g_param_spec_string("name",
			      "ConnmanService's name property",
			      "Name of the service",
			      SERVICE_NAME_UNKNOWN,
			      G_PARAM_READABLE);
  g_object_class_install_property(gobject_class, PROP_NAME, pspec);

  pspec = g_param_spec_uint("type",
			    "ConnmanService's type property",
			    "Type of the service",
			    0, G_MAXUINT,
			    CONNMAN_SERVICE_TYPE_ETHERNET,
			    G_PARAM_READABLE);
  g_object_class_install_property(gobject_class, PROP_TYPE, pspec);

  pspec = g_param_spec_uint("security",
			    "ConnmanService's security property",
			    "Security type of the service",
			    0, G_MAXUINT,
			    CONNMAN_SERVICE_SECURITY_NONE,
			    G_PARAM_READABLE);
  g_object_class_install_property(gobject_class, PROP_SECURITY, pspec);

  pspec = g_param_spec_uint("strength",
			    "ConnmanService's strength property",
			    "Current signal strength of the service",
			    0, G_MAXUINT8,
			    0,
			    G_PARAM_READABLE);
  g_object_class_install_property(gobject_class, PROP_SECURITY, pspec);

  pspec = g_param_spec_boolean("setup-required",
			       "ConnmanService's setup-required property",
			       "Is user configuration needed, eg APN",
			       FALSE,
			       G_PARAM_READABLE);
  g_object_class_install_property(gobject_class, PROP_SETUP_REQUIRED, pspec);
}

static void connman_service_init(ConnmanService *self)
{
}

ConnmanService *connman_service_new(const gchar *path)
{
  ConnmanService *self;

  g_return_val_if_fail(path != NULL, NULL);

  self = g_object_new(CONNMAN_TYPE_SERVICE, "path", path, NULL);

  return self;
}
