/*
This file copyright (c) 2004, Zasha Weinberg
All rights reserved.

Redistribution and use in source and binary forms, 
with or without modification, are permitted 
provided that the following conditions are met:

- Redistributions of source code must retain the 
above copyright notice, this list of conditions 
and the following disclaimer. 
- Redistributions in binary form must reproduce 
the above copyright notice, this list of 
conditions and the following disclaimer in the 
documentation and/or other materials provided 
with the distribution. 
- Neither the name of the University of Washington 
nor the names of its contributors may be used to 
endorse or promote products derived from this 
software without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS 
AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR 
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED 
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND 
FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS 
OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN 
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

// abstract base class for a list of sequences to search.  constructor initializes it to no file.  
class SequenceSet{
public:
	virtual ~SequenceSet ();
	virtual bool /* has one */ Next (void) = 0;
	virtual int GetLength (void) = 0;
	virtual char *GetDigitizedSeq (void) = 0; // get "digitized" sequence, where 0=A, 1=C, 2=G, 3=T
	virtual const char *GetTextSeq (void) = 0;
	virtual bool IsReversed (void) = 0;
	virtual const char *GetCurrFileName (void) = 0;
	virtual const char *GetCurrVirtualFileName (void); // by default, equal to GetCurrFileName; but, if files generated by partition, this is how you can get the original file name
	virtual const char *GetSqinfoDesc (void) = 0;
	virtual const char *GetSeqName (void) = 0;
	virtual void DumpContext (FILE *out=stdout) = 0;

	std::string GetEmblId (void); // throws exception if the sequence doesn't seem to have an EMBL id, or if it wasn't converted into FASTA format in the way I already have done.
};
class SequenceSet_UsingSQINFO : public SequenceSet {
protected:
	virtual const SQINFO& GetSQINFO (void) = 0;
public:
	int GetLength (void);
	const char *GetSqinfoDesc (void);
	const char *GetSeqName (void);
};
// implements the reversing logic
class SequenceSet_Reversing : public SequenceSet_UsingSQINFO {
protected:
	const bool onlyForwardStrand;
	bool isReversed;
	char *seq;
	char *digitizedSeq;
	SQINFO sqinfo;
	virtual bool /* has one */ NextSeq (char *(&seq),SQINFO& sqinfo) = 0; // not counting reversal
public:
	SequenceSet_Reversing (bool onlyForwardStrand_);
	~SequenceSet_Reversing ();

	bool /* has one */ Next (void);
	const SQINFO& GetSQINFO (void);
	char *GetDigitizedSeq (void);
	const char *GetTextSeq (void);
	bool IsReversed (void);
	void DumpContext (FILE *out);
};
// implements SequenceSet for just one file
class SequenceSet_OneFile : public SequenceSet_Reversing {
protected:
	int sequenceNum;
	SQFILE *sqfp;
	SQINFO sqinfo;
	char *seq;
	std::string fileName,virtualFileName;
	bool NextSeq (char *(&seq),SQINFO& sqinfo);
public:
	SequenceSet_OneFile (const char *_fileName,bool onlyForwardStrand);
	~SequenceSet_OneFile ();
	const char *GetCurrFileName (void);
	const char *GetCurrVirtualFileName (void);
	void DumpContext (FILE *out);

	static int squidFormat;
};
// encapsulates a \n-separated list of files
class SequenceSet_FileList : public SequenceSet_UsingSQINFO {
protected:
	const bool onlyForwardStrand;
	CommaSepFileReader fileListFile;
	SequenceSet_OneFile *currFile;
	bool NextSeq (char *(&seq),SQINFO& sqinfo);
public:
	SequenceSet_FileList (const char *fileListFileName,bool onlyForwardStrand_);
	~SequenceSet_FileList ();
	bool /* has one */ Next (void);
	const SQINFO& GetSQINFO (void);
	char *GetDigitizedSeq (void); // get "digitized" sequence, where 0=A, 1=C, 2=G, 3=T
	bool IsReversed (void);
	const char *GetCurrFileName (void);
	void DumpContext (FILE *out);
	const char *GetTextSeq (void);
};
// given string, make set.  if string starts with '@', it's a file that's a list of fasta files.  otherwise, it's just a fasta file.
SequenceSet *MakeSequenceSet(const char *fastaFileString,bool onlyForwardStrand);

// implements SequenceSet by caching a file (encapsulated with another SequenceSet object) in memory, and replaying things.  This is probably pointless, but seems fun to do.
// doesn't store the text-format seq, since I don't anticipate that being used.  Also, doesn't do anything for DumpContext.
class InMemorySequenceSet : public SequenceSet {
protected:
	struct Seq {
		std::string name,desc,fileName,virtualFileName;
		std::vector<char> rnaSequence;
		int rnaSequenceLen;
		bool isReversed;
	};
	typedef std::list<Seq> SeqList;
	SeqList seqList;
	SeqList::const_iterator currIter;
public:
	InMemorySequenceSet (SequenceSet& sourceSequenceSet);
	~InMemorySequenceSet ();

	void Rewind (void);

	bool Next (void);
	int GetLength (void);
	char *GetDigitizedSeq (void);
	virtual const char *GetTextSeq (void);
	bool IsReversed (void);
	const char *GetCurrFileName (void);
	const char *GetCurrVirtualFileName (void);
	const char *GetSqinfoDesc (void);
	const char *GetSeqName (void);
	void DumpContext (FILE *out);
};
